﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Windows.Forms;
using System.Resources;
using System.Threading;
using System.Reflection;
using System.IO;
using System.Globalization;
using RootPro.RootProCAD;
using RootPro.RootProCAD.Command;
using RootPro.RootProCAD.Geometry;
using RootPro.RootProCAD.UI;

//////////////////////////////////////////////////////////////////////////////
// 図形をカレントレイヤにコピー・移動するアドイン
// 選択している図形を、作図時のレイヤ配置に関係なく、
// カレントレイヤにコピー・移動することができます。
//////////////////////////////////////////////////////////////////////////////

namespace RCAddInShapeCopyOntoTheLayerCS
{
	[System.AddIn.AddIn("AppAddIn", Version = "1.0", Publisher = "", Description = "")]
	public partial class AppAddIn
	{
		private void AppAddIn_Startup(object sender, EventArgs e)
		{
			AppDomain.CurrentDomain.AssemblyResolve += new ResolveEventHandler(ResolveAssembly);

			//コマンドの追加
			CommandManager.AddMacroCommand("RCAddInShapeCopyOntoTheLayerCS", Properties.Resources.CopyCommandName, MacroCommandCopy);
			CommandManager.AddMacroCommand("RCAddInShapeMoveOntoTheLayerCS", Properties.Resources.MoveCommandName, MacroCommandMove);
		}

		private void AppAddIn_Shutdown(object sender, EventArgs e)
		{
			//コマンドの削除
			CommandManager.RemoveMacroCommand(MacroCommandCopy);
			CommandManager.RemoveMacroCommand(MacroCommandMove);
		}

		#region VSTA generated code
		private void InternalStartup()
		{
			this.Startup += new System.EventHandler(AppAddIn_Startup);
			this.Shutdown += new System.EventHandler(AppAddIn_Shutdown);
		}
		#endregion

		/// <summary>
		/// 選択している図形をカレントレイヤにコピーする
		/// </summary>
		private void MacroCommandCopy()
		{
			Document doc = ActiveDocument;         //ドキュメントの取得
			Drawing drawing = doc.CurrentDrawing;  //部分図の取得
			SelectedShapeCollection select = doc.SelectionManager.SelectedShapes;
			Layer currentLayer = doc.LayerTable.CurrentLayer;       //カレントレイヤの取得


			Shape shape = null;
			int selectCount = select.Count;
			if (selectCount == 0)
			{
				MessageBox.Show(Properties.Resources.ConfirmMessage1);
				return;
			}

			// UNDO 可能な操作を開始
			doc.UndoManager.BeginUndoUnit();

			for (int i = 0; i < selectCount; i++)
			{
				shape = select[i].Shape;  //部分図の最初の図形を検索

				Shape copyShape = shape.Copy();
				copyShape.Layer = currentLayer;     //カレントレイヤに設定
			}

			MessageBox.Show(string.Format(Properties.Resources.ConfirmMessage2, selectCount));


			// UNDO 可能な操作を終了
			doc.UndoManager.EndUndoUnit();
		}


		/// <summary>
		///  選択している図形をカレントレイヤに移動する
		/// </summary>
		private void MacroCommandMove()
		{
			Document doc = ActiveDocument;         //ドキュメントの取得
			Drawing drawing = doc.CurrentDrawing;  //部分図の取得
			SelectedShapeCollection select = doc.SelectionManager.SelectedShapes;
			Layer currentLayer = doc.LayerTable.CurrentLayer;       //カレントレイヤの取得

			// UNDO 可能な操作を開始
			doc.UndoManager.BeginUndoUnit();

			Shape shape = null;
			int selectCount = select.Count;
			if (selectCount == 0)
			{
				MessageBox.Show(Properties.Resources.ConfirmMessage1);
				return;
			}


			for (int i = 0; i < selectCount; i++)
			{
				shape = select[i].Shape;  //部分図の最初の図形を検索
				shape.Layer = currentLayer;
			}

			MessageBox.Show(string.Format(Properties.Resources.ConfirmMessage3, selectCount));


			// UNDO 可能な操作を終了
			doc.UndoManager.EndUndoUnit();
		}

		private Assembly ResolveAssembly(object sender, ResolveEventArgs args)
		{
			AssemblyName assemblyName = new AssemblyName(args.Name);

			foreach (Assembly assembly in AppDomain.CurrentDomain.GetAssemblies())
			{
				if (AssemblyName.ReferenceMatchesDefinition(assemblyName, assembly.GetName()))
					return assembly;
			}

			if (assemblyName.Name.EndsWith(".resources"))
			{
				if (assemblyName.CultureName.Equals(Thread.CurrentThread.CurrentUICulture.Name, StringComparison.OrdinalIgnoreCase))
				{
					string dllPath = Path.GetDirectoryName(this.AddInFullPath);
					dllPath = Path.Combine(dllPath, assemblyName.CultureName);
					dllPath = Path.Combine(dllPath, assemblyName.Name + ".dll");

					if (File.Exists(dllPath))
					{
						try
						{
							return Assembly.LoadFrom(dllPath);
						}
						catch
						{
						}
					}
				}
			}

			return null;
		}
	}
}
