﻿using RootPro.RootProCAD;
using RootPro.RootProCAD.Command;
using RootPro.RootProCAD.Geometry;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Windows.Controls;
using System.Windows.Forms;
using System.Windows.Shapes;
using static RCAddInShadowDrawing.DockingBarUserControl;
using Shape = RootPro.RootProCAD.Shape;

/// <summary>
/// 日影・天空率プログラム
/// 概要はDockingBarUserControl.cs参照
/// ここは、主に天空率・天空図を作成するためのコードを書いている
/// 
/// ※使用許諾
/// ・このアドインを実行した結果についての一切の責任はお使いになられる方にあります。
/// ・このアドインは、プログラムソース自体を改変・流用することが可能です。
/// 　また、このプログラムソースを元に独自のアドインやソフトウェアを作成し、無償・有償を問わず第三者に配布することも認めます。
/// 　ただし、改変・流用する場合は、株式会社ルートプロ(RootPro Corporation)作成の
/// 　当プログラムソースを参考にしていることを第三者にわかるように明記してください。
/// 　
/// 　Copyright @ 2020 RootPro Co.,Ltd. (Japan)
/// 
/// </summary>

namespace RCAddInShadowDrawing
{
    partial class DockingBarUserControl
    {
        // メンバ変数
        private const bool RC_BaseTriangleCheck = false;// 三斜三角形の元となる三角形をチェック描画(開発デバッグ用)

        // 作成コマンド実行時で測定点を点列で指定した時、天空図の間隔スケールパラメータアイテム
        private string[] spaceXModeListParameterItems = { "自動", "0.25", "0.5", "0.75", "1.0", "1.5", "1.75", "2.0" };

        /// <summary>
        /// 天空図作成時のベースライン3Dクラス
        /// </summary>
        public class BaseLine3D
        {
            /// <summary>
            /// コンストラクタ―
            /// </summary>
            /// <param name="baseLineID"></param>
            /// <param name="line"></param>
            /// <param name="startPointHeight"></param>
            /// <param name="endPointHeight"></param>
            public BaseLine3D(int baseLineID, LineShape line, double startPointHeight, double endPointHeight)
            {
                this.baseLineID = baseLineID;
                this.line = line;
                this.startPointHeight = startPointHeight;
                this.endPointHeight = endPointHeight;

                connectLineIDList = new List<int>();

                startPointSidePositionNumber = -1;
                endPointSidePositionNumber = -1;

                isDrawStartPointSidePositionNumberAtSkymap = false;
                isDrawEndPointSidePositionNumberAtSkymap = false;
            }

            public int baseLineID { get; private set; }// この線に付加する番号
            public LineShape line { get; private set; }
            public double startPointHeight { get; private set; }
            public double endPointHeight { get; private set; }

            public List<int> connectLineIDList { get; }// つながる線のIDリスト

            public int startPointSidePositionNumber { get; private set; }// 位置番号(始点側)(未設定は-1)
            public int endPointSidePositionNumber { get; private set; }// 位置番号(終点側)(未設定は-1)

            public bool isDrawStartPointSidePositionNumberAtSkymap { get; set; }// 天空図に位置番号を描画するかフラグ(始点側)
            public bool isDrawEndPointSidePositionNumberAtSkymap { get; set; }// 天空図に位置番号を描画するかフラグ(終点側)


            /// <summary>
            /// 繋がるLineIDを追加
            /// </summary>
            /// <param name="lineID"></param>
            public void AddConnectID(int baseLineID)
            {
                if(!connectLineIDList.Contains(baseLineID))
                    connectLineIDList.Add(baseLineID);
            }

            /// <summary>
            /// 指定したLineIDの線と繋がっているか
            /// </summary>
            /// <param name="baseLineID"></param>
            /// <returns></returns>
            public bool IsConnectID(int baseLineID)
            {
                if(connectLineIDList.Contains(baseLineID))
                    return true;

                return false;
            }

            /// <summary>
            /// 位置番号のセット
            /// </summary>
            /// <param name="number"></param>
            /// <returns></returns>
            public int SetPositionNumber(int number)
            {
                // 始点側の番号セット
                startPointSidePositionNumber = number;

                // 終点側は始点側と高さが異なる場合
                if(Math.Abs(startPointHeight - endPointHeight) > Precision.Confusion)
                {
                    endPointSidePositionNumber = number + 1;// startPointSidePositionNumber+1
                    return number + 2; // 次の番号を返す
                }
                else
                {
                    // 始点側と同じ高さなら
                    endPointSidePositionNumber = number; // startPointSidePositionNumberと同じ
                }

                return number + 1; // 次の番号を返す
            }

            /// <summary>
            /// 位置番号(始点側まはた主点側)が設定されているか
            /// </summary>
            /// <returns></returns>
            public bool IsPositionNumber()
            {
                if(startPointSidePositionNumber > 0 || endPointSidePositionNumber > 0)
                    return true;

                return false;
            }

            /// <summary>
            /// 指定した点からの距離取得(ベース線の始点との距離)
            /// </summary>
            /// <param name="point"></param>
            /// <returns></returns>
            public double GetDistanceFromStartPoint(Point2d point)
            {
                return Get2PointsDistance(point, line.StartPoint);
            }

            /// <summary>
            /// 指定した点からの距離取得(ベース線の終点との距離)
            /// </summary>
            /// <param name="point"></param>
            /// <returns></returns>
            public double GetDistanceFromEndPoint(Point2d point)
            {
                return Get2PointsDistance(point, line.EndPoint);
            }
        }

        /// <summary>
        /// 配置位置図形クラス
        /// </summary>
        public class PositionNumberShape
        {
            /// <summary>
            /// コンストラクタ
            /// </summary>
            public PositionNumberShape()
            {
                this.Number = -1;
                this.Shapes = new List<Shape>();
            }

            public int Number;
            public List<Shape> Shapes;
        }

        /// <summary>
        /// 三斜計算モード
        /// </summary>
        public enum TriclinicMode
        {
            TriclinicMode_Standard = 0, // 基準建物用
            TriclinicMode_Plan = 1,     // 計画建物用
        };

        /// <summary>
        /// 三斜求積表と建物位置表の位置モード
        /// </summary>
        enum SkyMapTableLocateMode
        {
            SkyMapTableLocateMode_Non = -1, // なし
            SkyMapTableLocateMode_LowerLower = 0, // 下・下(三斜求積表・建物位置表の順番)
            SkyMapTableLocateMode_RightLower = 1, // 右・下(三斜求積表・建物位置表の順番)
            SkyMapTableLocateMode_Free = 2, // 任意
        }

        /// <summary>
        /// 三角形の辺クラス
        /// 天空図の中心点から放射状に延びる線
        /// </summary>
        public class TriangleLine
        {
            public TriangleLine(int baseLineID, Point2d orgPoint, Point2d vertexPoint)
            {
                this.baseLineID = baseLineID;
                this.vertexPoint = vertexPoint;

                double lineAngleReg = Math.Atan2(vertexPoint.Y - orgPoint.Y, vertexPoint.X - orgPoint.X);
                lineAngleDeg = lineAngleReg * (180.0 / Math.PI) + (lineAngleReg > 0.0 ? 0.0 : 360.0);// 0 ～ 360度
            }

            // 底辺の角度(水平角度)
            public double lineAngleDeg { get; private set; }

            // 正射影の線のID == 正射影の線の元になった建物の線のID
            public int baseLineID { get; private set; }

            // 正射影の線の端点（ここから円の中心までの線が底辺を表す線になる）
            public Point2d vertexPoint { get; private set; }
        }

        /// <summary>
        /// 三角形クラス(三斜三角形の元の三角形)
        /// 天空率計算過程で三斜三角形を作成する元の情報
        /// この三角形を既定の角度以内の大きさに分割したり
        /// 内接させるか外接させるかで頂点を変えたりして三斜三角形にする
        /// </summary>
        public class BaseTriangleInfo
        {
            public Point2d centerPoint { get; private set; } // 天空図の円の中心点(三角形の頂点の一つ)
            public Point2d vertexPoint1 { get; private set; }// 頂点1;
            public Point2d vertexPoint2 { get; private set; }// 頂点2;

            // 正射影の線のID == 正射影の線元になった建物の線のID
            public int baseLineID { get; private set; }// 頂点1と頂点2の間になる正射影の線のID

            public BaseTriangleInfo(Point2d centerPoint, Point2d vertexPoint1, Point2d vertexPoint2, int baseLineID)
            {
                this.centerPoint = centerPoint;
                this.vertexPoint1 = vertexPoint1;
                this.vertexPoint2 = vertexPoint2;

                this.baseLineID = baseLineID;
            }
        }

        /// <summary>
        /// 三斜三角形クラス
        /// </summary>
        public class TriclinicTriangle
        {
            public TriclinicMode triclinicMode { get; private set; }// 三斜計算モード

            // 番号
            public int triangleNumber { get; private set; }

            // 三斜中心点座標
            public Point2d centerPoint { get; private set; }

            // 底辺長さ
            public double baselineLength { get; private set; }
            // 底辺の端点
            public Point2d baselinePoint { get; private set; }
            // 底辺の角度
            public double baselineAngleDeg { get; private set; }

            // 高さ
            public double hight { get; private set; }
            // 高さを表す線の座標1(中心から伸びるもう一つの底辺の端点)
            public Point2d hightlineTopPoint { get; private set; }
            // もう一つの底辺の角度
            public double otherlineAngleDeg { get; private set; }
            // 高さを表す線の底辺上の点
            public Point2d hightlinePointOnBaseline { get; private set; }

            // 面積
            public double triangleArea { get; private set; }

            /// <summary>
            /// コンストラクタ―
            /// </summary>
            /// <param name="triangleNumber"></param>
            /// <param name="line1"></param>
            /// <param name="line2"></param>
            /// <param name="triclinicMode"></param>
            public TriclinicTriangle(int triangleNumber, LineShape line1, LineShape line2, TriclinicMode triclinicMode, AddInSettings addInSettings)
            {
                this.triclinicMode = triclinicMode;

                this.triangleNumber = triangleNumber;

                // 2つの線の始点は同じでそれが三斜三角形の中心点(天空図の中心となる点)
                Point2d vertexPoint11 = line1.StartPoint;
                Point2d vertexPoint12 = line1.EndPoint;
                Point2d vertexPoint21 = line2.StartPoint;
                Point2d vertexPoint22 = line2.EndPoint;

                Debug.Assert(Get2PointsDistance(vertexPoint11, vertexPoint21) <= Precision.Confusion);
                this.centerPoint = vertexPoint11;

                // 中心から伸びる底辺の長さで各座標や長さを決める
                PointOnShape pointOnShape;
                double length1 = line1.Length;
                double length2 = line2.Length;
                if(length1 >= length2)
                {
                    baselinePoint = vertexPoint12;
                    hightlineTopPoint = vertexPoint22;
                    baselineLength = length1;
                    baselineAngleDeg = line1.Angle;
                    otherlineAngleDeg = line2.Angle;
                    pointOnShape = line1.GetClosestPoint(hightlineTopPoint);
                }
                else
                {
                    baselinePoint = vertexPoint22;
                    hightlineTopPoint = vertexPoint12;
                    baselineLength = length2;
                    baselineAngleDeg = line2.Angle;
                    otherlineAngleDeg = line1.Angle;
                    pointOnShape = line2.GetClosestPoint(hightlineTopPoint);
                }

                // 底辺の長さ
                baselineLength = GetSkymapRoundValue(baselineLength, triclinicMode, addInSettings.skymapTriangleLengthValueDecimals);// 三斜求積モードに合わせた値(「切り捨て」or 「切り上げ」)の取得

                // 高さ
                hightlinePointOnBaseline = pointOnShape.Point;
                hight = Get2PointsDistance(hightlineTopPoint, hightlinePointOnBaseline);
                hight = GetSkymapRoundValue(hight, triclinicMode, addInSettings.skymapTriangleLengthValueDecimals);// 三斜求積モードに合わせた値(「切り捨て」or 「切り上げ」)の取得

                // 面積
                triangleArea = baselineLength * hight * 0.5;
                triangleArea = GetSkymapRoundValue(triangleArea, triclinicMode, addInSettings.skymapAreaValueDecimals);// 三斜求積モードに合わせた値(「切り捨て」or 「切り上げ」)の取得
            }
        }


        /// <summary>
        /// 正射影の線クラス
        /// </summary>
        public class OrthographicProjectionShape
        {
            /// <summary>
            /// コンストラクタ―
            /// </summary>
            /// <param name="baseLineID"></param>
            /// <param name="polylineShape"></param>
            public OrthographicProjectionShape(int baseLineID, PolylineShape polylineShape)
            {
                this.baseLineID = baseLineID;
                this.polylineShape = polylineShape;
            }

            public int baseLineID { get; private set; }// 元となった建物の線のID
            public PolylineShape polylineShape { get; private set; }

        }

        /// <summary>
        /// 交点情報クラス
        /// </summary>
        public class IntersectionPointInfo
        {
            public double parameter { get; private set; }
            public Point2d point { get; private set; }
            public int baseLineID { get; private set; }// 元となった建物の線のID
            public int crossLineType { get; private set; } // 交点を持つ相手の線のタイプ(正射影の上の線:1、下の線:-1)

            /// <summary>
            /// コンストラクタ―
            /// </summary>
            /// <param name="parameter"></param>
            /// <param name="point"></param>
            /// <param name="baseLineID"></param>
            /// <param name="crossLineType"></param>
            public IntersectionPointInfo(double parameter, Point2d point, int baseLineID, int crossLineType)
            {
                this.parameter = parameter;
                this.point = point;
                this.baseLineID = baseLineID;
                this.crossLineType = crossLineType;
            }
        }

        /// <summary>
        /// 交点情報リストセット
        /// </summary>
        public class IntersectionPointInfoListSet
        {
            public LineShape lineShape { get; private set; }
            public List<IntersectionPointInfo> intersectionPointInfoList { get; private set; }

            /// <summary>
            /// コンストラクター
            /// </summary>
            /// <param name="lineShape"></param>
            /// <param name="intersectionPointInfoList"></param>
            public IntersectionPointInfoListSet(LineShape lineShape, List<IntersectionPointInfo> intersectionPointInfoList)
            {
                this.lineShape = lineShape;
                this.intersectionPointInfoList = intersectionPointInfoList;
            }
        }

        /// <summary>
        /// 交点情報クラス2
        /// </summary>
        public class IntersectionPointInfo2
        {
            public double parameter { get; private set; }
            public int index { get; private set; }

            /// <summary>
            /// コンストラクタ―
            /// </summary>
            /// <param name="parameter"></param>
            /// <param name="index"></param>
            public IntersectionPointInfo2(double parameter, int index)
            {
                this.parameter = parameter;
                this.index = index;
            }
        }

        public class IntersectionPointInfo3
        {
            public double parameter { get; private set; }
            public int index { get; private set; }
            public int baseLineID { get; private set; }

            /// <summary>
            /// コンストラクタ―
            /// </summary>
            /// <param name="parameter"></param>
            /// <param name="index"></param>
            /// <param name="baseLineID"></param>
            public IntersectionPointInfo3(double parameter, int index, int baseLineID)
            {
                this.parameter = parameter;
                this.index = index;
                this.baseLineID = baseLineID;
            }
        }

        /// <summary>
        /// 天空率(三斜求積による)属性クラス
        /// </summary>
        public class TriclinicSkyRateAttribute
        {
            public TriclinicMode triclinicMode { get; private set; }// 三斜計算モード

            public double skyMapRadius { get; private set; }// 天空図の半径
            public double sectorAngleDeg { get; private set; }// 扇形中心角
            public double totalTriclinicTriangleArea { get; private set; }// 三斜面積合計

            /// <summary>
            /// コンストラクタ―
            /// </summary>
            /// <param name="triclinicMode"></param>
            /// <param name="skyMapRadius"></param>
            /// <param name="sectorAngleDeg"></param>
            /// <param name="totalTriclinicTriangleArea"></param>
            public TriclinicSkyRateAttribute(TriclinicMode triclinicMode, double skyMapRadius, double sectorAngleDeg, double totalTriclinicTriangleArea, AddInSettings addInSettings)
            {
                Debug.Assert(skyMapRadius > Precision.Confusion);

                this.triclinicMode = triclinicMode;

                // 三斜求積モードに合わせた値(「切り捨て」or 「切り上げ」)にしておく
                this.skyMapRadius = skyMapRadius;// GetSkymapRoundValue(skyMapRadius, triclinicMode);
                this.sectorAngleDeg = sectorAngleDeg;// GetSkymapRoundValue(sectorAngleDeg, triclinicMode);
                this.totalTriclinicTriangleArea = GetSkymapRoundValue(totalTriclinicTriangleArea, triclinicMode, addInSettings.skymapAreaValueDecimals);

                // 天空図円面積
                double skyMapCircleArea = this.skyMapRadius * this.skyMapRadius * Math.PI;
                this.skyMapCircleArea = GetSkymapRoundValue(skyMapCircleArea, triclinicMode, addInSettings.skymapAreaValueDecimals);

                // 扇形面積
                double sectorArea = this.skyMapCircleArea * this.sectorAngleDeg / 360.0;
                this.sectorArea = GetSkymapRoundValue(sectorArea, triclinicMode, addInSettings.skymapAreaValueDecimals);

                // 水平円射影面積(建築物投影面積)　(引き算なので有効桁に変化はないのでGetSkymapRoundValueしない)
                this.circularProjectionArea = this.sectorArea - totalTriclinicTriangleArea; // 水平円射影面積(建築物投影面積) = 扇形面積 - 三斜面積合計

                // 天空率 = (天空図円面積 - 水平円射影面積) / 天空図円面積 * 100.0
                double skyRate = (this.skyMapCircleArea - this.circularProjectionArea) / this.skyMapCircleArea * 100.0;
                this.skyRate = GetSkymapRoundValue(skyRate, triclinicMode, addInSettings.skyRateValueDecimals);
            }

            public double skyMapCircleArea { get; private set; }// 天空図円面積
            public double sectorArea { get; private set; }// 扇形面積
            public double circularProjectionArea { get; private set; }// 水平円射影面積(建築物投影面積)
            public double skyRate { get; private set; }// 天空率
        }

        /// <summary>
        /// 位置表のデータアイテムクラス
        /// </summary>
        public class PositionDataItem
        {
            /// <summary>
            /// 位置No.
            /// </summary>
            public int Number { get; private set; }

            /// <summary>
            /// 距離（配置図）
            /// </summary>
            public double Distance { get; private set; }

            /// <summary>
            /// 高さ（配置図）
            /// </summary>
            public double Hight { get; private set; }

            /// <summary>
            /// 方位角（天空図）
            /// </summary>
            public double AngleDeg { get; private set; }

            /// <summary>
            /// 仰角:h（天空図）
            /// </summary>
            public double AngleDegH { get; private set; }

            /// <summary>
            /// R*Cos(h) （天空図）
            /// </summary>
            public double DistanceH { get; private set; }

            /// <summary>
            /// コンストラクター
            /// </summary>
            /// <param name="number">位置No.</param>
            /// <param name="distance">距離</param>
            /// <param name="hight">高さ</param>
            /// <param name="angleDeg">方位角</param>
            /// <param name="angleDegH">仰角:h</param>
            /// <param name="distanceH">R*Cos(h)</param>
            public PositionDataItem(int number, double distance, double hight, double angleDeg, double angleDegH, double distanceH)
            {
                Number = number;
                Distance = distance;
                Hight = hight;
                AngleDeg = angleDeg;
                AngleDegH = angleDegH;
                DistanceH = distanceH;
            }
        }

        /// <summary>
        /// 天空率を求めるとき、三斜求積モードに合わせた値の取得
        /// 計画建物用の場合「切り捨て」、基準建物用の場合「切り上げ」
        /// </summary>
        /// <param name="value"></param>
        /// <param name="triclinicMode"></param>
        /// <param name="decimals">小数部桁数</param>
        /// <returns></returns>
        static private double GetSkymapRoundValue(double value, TriclinicMode triclinicMode, int decimals)
        {
            Debug.Assert(triclinicMode == TriclinicMode.TriclinicMode_Plan || triclinicMode == TriclinicMode.TriclinicMode_Standard);

            double roundValue = 0.0;

            // 計画建物用の場合
            if(triclinicMode == TriclinicMode.TriclinicMode_Plan)
            {
                roundValue = Truncate(value, decimals);//「切り捨て」
            }
            // 基準建物用の場合
            else if(triclinicMode == TriclinicMode.TriclinicMode_Standard)
            {
                roundValue = RoundUp(value, decimals); //「切り上げ」
            }

            return roundValue;
        }


        /// <summary>
        /// 値の小数部を切り捨て、指定の桁数に丸める
        /// </summary>
        /// <param name="value">値</param>
        /// <param name="decimals">小数部桁数</param>
        /// <returns>丸めた値</returns>
        static public double Truncate(double value, int decimals)
        {
            // 小数部桁数の10の累乗を取得
            double pow = Math.Pow(10, decimals);
            return 0 <= value ? Math.Floor(value * pow) / pow : Math.Ceiling(value * pow) / pow;
        }

        /// <summary>
        /// 値の小数部を切り上げ、指定の桁数に丸める
        /// </summary>
        /// <param name="value">値</param>
        /// <param name="decimals">小数部桁数</param>
        /// <returns>丸めた値</returns>
        static public double RoundUp(double value, int decimals)
        {
            // 小数部桁数の10の累乗を取得
            double pow = Math.Pow(10, decimals);
            return 0 <= value ? Math.Ceiling(value * pow) / pow : Math.Floor(value * pow) / pow;
        }

        /// <summary>
        /// 計画建物 ラジオボタンクリック時処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void planRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 適合建物ラジオボタンクリック時処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void fitRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 建物内部の上は屋根 チェックボタンクリック時処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void topRoofCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 測定点No. チェックボックス変更時の処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void samplingPointNumberCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            // 各コントロールの状態更新
            UpDateControlState();

            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 測定点No. 変更時の処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void measurementPointcUpDown_ValueChanged(object sender, EventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 測定点No. の番号を作図するチェック 変更時の処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void drawSamplingPointNumberCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 天空率 想定面の高さ 変更時の処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void locationHeightTextBox2_TextChanged(object sender, EventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 天空図の半径 文字変更時の処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void hemisphereRadiusUpDown_TextChanged(object sender, EventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 天空図の半径 変更時の処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void hemisphereRadiusUpDown_ValueChanged(object sender, EventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 天空図の半径 倍率 変更時の処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void drawScaleComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 最大分割角度 変更時の処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void maxSeparateAngleComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 天空率のみを表示 チェックボタンクリック時処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void onlySkyRateCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            // 三斜求積図を作図しない場合は、コマンドパラメータに
            // 「配置図に番号を作図」があれば削除するため、コマンド実行中なら更新させる
            Command command = this.app.CommandManager.CurrentCommand;
            if(command != null)
            {
                if(command.UniqueName == ShadowDrawingSkyMapCommandName)
                    UpdateTableLocateCommandParameter(command);
            }

            // 各コントロールの状態更新
            UpDateControlState();

            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// ハッチングの間隔 コンボボックスの選択変更時の処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void hatchComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 目盛 コンボボックスの選択変更時の処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void scaleComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 三斜求積図 チェックボタンクリック時処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void triclinicTriangleCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            // 三斜求積図を作図する場合、コマンドパラメータに「配置図に番号を作図」を
            // 選べるようにするため、コマンド実行中なら更新させる
            Command command = this.app.CommandManager.CurrentCommand;
            if(command != null)
            {
                if(command.UniqueName == ShadowDrawingSkyMapCommandName)
                    UpdateTableLocateCommandParameter(command);
            }

            // 各コントロールの状態更新
            UpDateControlState();

            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 天空図の全ての頂点に位置番号を付ける チェックボタンクリック時処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void drawAllPointNumberCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 三斜求積表 チェックボタンクリック時処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void quadratureTableCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 建物位置表 チェックボタンクリック時処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void positionTableCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 太陽軌跡を表示 チェックボタンクリック時処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void solarTrajectoryCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            // 各コントロールの状態更新
            UpDateControlState();

            //ラバーバンドクリア
            ClearRubberBand();
        }

        // 10分間隔 チェックボタンクリック時処理
        private void solarTrajectoryInterval10MinutesCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 四季 チェックボタンクリック時処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void seasonsCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 方向補正 チェックボタンクリック時処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void directCorrectionCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 円と正射影の面積を表示 チェックボタンクリック時処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>

        private void circularProjectionAreaCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 天空率を表示 チェックボタンクリック時処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void skyRateCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 文字サイズ調整バーの値変更時処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void textSizeTrackBar_ValueChanged(object sender, EventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 天空図の作成ボタン
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void skyMapCommandButton_Click(object sender, EventArgs e)
        {
            app.CommandManager.CancelCurrentCommand();

            // コマンドの実行
            app.CommandManager.ExecuteCommand(ShadowDrawingSkyMapCommandName);
        }


        /// <summary>
        /// コマンドパラメータ切り替えイベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void CommandSkyMap_ParameterCurrentIndexChanged(Object sender, CommandParameterGroupCurrentIndexChangedEventArgs e)
        {
            // パラメータの切り替えによる処理（違うタイプのパラメーターに切り替えたり、他のパラメータを増やしたりする）
            Command command = (Command)sender;

            CommandParameterGroup commandParameterGroup = e.ParameterGroup;
            int currentParameterItemID = commandParameterGroup.CurrentItem.ID;

            // カレントパラメータが(測定点・測定点(連続))で変更する場合
            if(currentParameterItemID == samplingPointParameterId || currentParameterItemID == samplingPointsParameterId)
            {
                ParameterItemCollection parameterItemCollection = commandParameterGroup.Items;
                int itemCount = parameterItemCollection.Count;

                int oldItemIndex = e.OldIndex;
                int newItemIndex = e.NewIndex;
                Debug.Assert(oldItemIndex >= 0 && oldItemIndex < itemCount);
                Debug.Assert(newItemIndex >= 0 && newItemIndex < itemCount);

                // 変更がないなら何もしない
                if(newItemIndex == oldItemIndex)
                    return;

                ParameterItem oldParamItem = parameterItemCollection[oldItemIndex];
                ParameterItem newParamItem = parameterItemCollection[newItemIndex];

                int oldParamID = oldParamItem.ID;
                int newParamID = newParamItem.ID;
                int changedSamplingPointParameterID = -1;

                // 「測定点 → 測定点(連続)」
                if(oldParamID == samplingPointParameterId && newParamID == samplingPointsParameterId)
                {
                    changedSamplingPointParameterID = samplingPointsParameterId;
                }
                // 「測定点(連続) → 測定点」
                else if(oldParamID == samplingPointsParameterId && newParamID == samplingPointParameterId)
                {
                    changedSamplingPointParameterID = samplingPointParameterId;
                }

                // 測定点を点列で指定する場合に表示するコマンドパラメータの状態更新
                UpdateCommandParameterWithSamplingPointsParameter(command, changedSamplingPointParameterID);

                // 表に関係したコマンドパラメータの更新
                UpdateTableLocateCommandParameter(command, changedSamplingPointParameterID);
            }
        }

        /// <summary>
        /// 測定点の選ばれているItemIDを取得
        /// </summary>
        /// <param name="command"></param>
        /// <returns></returns>
        private int GetSamplingPointParameterItemID(Command command)
        {
            int samplingPointParameterItemID = -1;

            CommandParameterGroupCollection commandParameterGroupCollection = command.ParameterGroups;
            foreach(CommandParameterGroup commandParameterGroup in commandParameterGroupCollection)
            {
                int currentParameterItemID = commandParameterGroup.CurrentItem.ID;
                if(currentParameterItemID == samplingPointParameterId || currentParameterItemID == samplingPointsParameterId)
                {
                    samplingPointParameterItemID = currentParameterItemID;
                    break;
                }
            }

            return samplingPointParameterItemID;
        }

        /// <summary>
        /// 測定点を点列で指定する場合に表示するコマンドパラメータの状態更新
        /// </summary>
        /// <param name="command"></param>
        /// <param name="changedSamplingPointParameterID"></param>
        private void UpdateCommandParameterWithSamplingPointsParameter(Command command, int changedSamplingPointParameterID = -1)
        {
            // 測定点が連続指定か
            int samplingPointParameterItemID = changedSamplingPointParameterID;
            bool samplingPointParameterItemIdIsPoints = false;
            if(samplingPointParameterItemID < 0)
                samplingPointParameterItemID = GetSamplingPointParameterItemID(command);

            Debug.Assert(samplingPointParameterItemID > 0);
            if(samplingPointParameterItemID == samplingPointsParameterId)
                samplingPointParameterItemIdIsPoints = true;

            // 測定点が点列指定の場合
            if(samplingPointParameterItemIdIsPoints && !onlySkyRateCheckBox.Checked)
            {   
                // フォームの測定面の高さ変更不可に
                locationHeightTextBox2.Enabled = false;

                // 測定面の高さ(連続)パラメータ
                TextParameterItem samplingPointHeightsParameterItem = (TextParameterItem)command.GetParameterItemByID(samplingPointHeightsParameterId);
                if(samplingPointHeightsParameterItem == null)
                {
                    // 配置点パラメータの前に挿入する
                    ParameterItem locatePointPrameterItem = command.GetParameterItemByID(locatePointParameterId);
                    int locatePointPrameterItemIndex = locatePointPrameterItem.Index;

                    CommandParameterGroup commandParameterGroup = command.ParameterGroups.Insert(locatePointPrameterItemIndex, ParameterItemType.Text, samplingPointHeightsParameterId, "測定面の高さ(連続)");
                    samplingPointHeightsParameterItem = (TextParameterItem)command.GetParameterItemByID(samplingPointHeightsParameterId);
                    samplingPointHeightsParameterItem.ValueSaveMode = ParameterValueSaveMode.OneTime; // パラメータ値の保存方法
                    samplingPointHeightsParameterItem.Description = "各測定点の高さをカンマ(,)で区切って指定します。測定点の数と同じ数指定する必要があります。例:測定点(連続)パラメーターで4点指定した場合、0.2, 0.1, 0.3, 0.2";
                }

                // 配置間隔スケールリストパラメータ
                ListParameterItem spaceXModeListParameterItem = (ListParameterItem)command.GetParameterItemByID(spaceXParameterId);
                if(spaceXModeListParameterItem == null)
                {
                    // 配置点パラメータの前に挿入する
                    ParameterItem locatePointPrameterItem = command.GetParameterItemByID(locatePointParameterId);
                    int locatePointPrameterItemIndex = locatePointPrameterItem.Index;

                    CommandParameterGroup commandParameterGroup = command.ParameterGroups.Insert(locatePointPrameterItemIndex, ParameterItemType.List, spaceXParameterId, "配置間隔スケール");
                    spaceXModeListParameterItem = (ListParameterItem)command.GetParameterItemByID(spaceXParameterId);
                    string[] items1 = spaceXModeListParameterItems;
                    spaceXModeListParameterItem.Items = items1;
                    spaceXModeListParameterItem.SelectedIndex = addInSettings.oldSelectedSpaceXModeListCommandParameterItemIndex;
                    spaceXModeListParameterItem.ValueSaveMode = ParameterValueSaveMode.Always; // パラメータ値の保存方法
                    spaceXModeListParameterItem.Description = "次の天空図との間隔を天空図の円の半径に対してどのぐらいの倍率で離すかを指定します。";
                }

                addInSettings.oldSelectedSpaceXModeListCommandParameterItemIndex = spaceXModeListParameterItem.SelectedIndex;// 選択されたモードを保持

                // 三斜求積表や建物位置表を作成しない場合
                if(!(triclinicTriangleCheckBox.Checked && (quadratureTableCheckBox.Checked || positionTableCheckBox.Checked)) && !onlySkyRateCheckBox.Checked)
                {
                    // 枠モードリストパラメータ
                    ListParameterItem withFrameModeListParameterItem = (ListParameterItem)command.GetParameterItemByID(withFrameParameterId);
                    if(withFrameModeListParameterItem == null)
                    {
                        // 配置点パラメータの前に挿入する
                        ParameterItem locatePointPrameterItem = command.GetParameterItemByID(locatePointParameterId);
                        int locatePointPrameterItemIndex = locatePointPrameterItem.Index;

                        CommandParameterGroup commandParameterGroup = command.ParameterGroups.Insert(locatePointPrameterItemIndex, ParameterItemType.List, withFrameParameterId, "枠");
                        withFrameModeListParameterItem = (ListParameterItem)command.GetParameterItemByID(withFrameParameterId);
                        string[] items1 = { "なし", "1段", "左[]付1段", "2段", "左[]付2段" };
                        withFrameModeListParameterItem.Items = items1;
                        withFrameModeListParameterItem.SelectedIndex = addInSettings.oldSelectedWithFrameModeListCommandParameterItemIndex;
                        withFrameModeListParameterItem.ValueSaveMode = ParameterValueSaveMode.Always; // パラメータ値の保存方法
                        withFrameModeListParameterItem.Description = "天空図を囲む枠を同時に作成します。2段を選択した場合は上にコメント行が付きます。";
                    }

                    addInSettings.oldSelectedWithFrameModeListCommandParameterItemIndex = withFrameModeListParameterItem.SelectedIndex;// 選択されたモードを保持
                }
                else
                {
                    // 枠モードリストパラメータ
                    ListParameterItem withFrameModeListParameterItem = (ListParameterItem)command.GetParameterItemByID(withFrameParameterId);
                    if(withFrameModeListParameterItem != null)
                    {
                        addInSettings.oldSelectedWithFrameModeListCommandParameterItemIndex = withFrameModeListParameterItem.SelectedIndex;// 選択されたモードを保持

                        CommandParameterGroup commandParameterGroup = withFrameModeListParameterItem.Group;
                        command.ParameterGroups.Remove(commandParameterGroup);
                    }
                }
            }
            else
            {
                // 測定面の高さ(連続)パラメータ
                TextParameterItem samplingPointHeightsParameterItem = (TextParameterItem)command.GetParameterItemByID(samplingPointHeightsParameterId);
                if(samplingPointHeightsParameterItem != null)
                {
                    CommandParameterGroup commandParameterGroup = samplingPointHeightsParameterItem.Group;
                    command.ParameterGroups.Remove(commandParameterGroup);
                }

                // 配置間隔スケールリストパラメータ
                ListParameterItem spaceXModeListParameterItem = (ListParameterItem)command.GetParameterItemByID(spaceXParameterId);
                if(spaceXModeListParameterItem != null)
                {
                    addInSettings.oldSelectedSpaceXModeListCommandParameterItemIndex = spaceXModeListParameterItem.SelectedIndex;// 選択されたモードを保持

                    CommandParameterGroup commandParameterGroup = spaceXModeListParameterItem.Group;
                    command.ParameterGroups.Remove(commandParameterGroup);
                }

                // 枠モードリストパラメータ
                ListParameterItem withFrameModeListParameterItem = (ListParameterItem)command.GetParameterItemByID(withFrameParameterId);
                if(withFrameModeListParameterItem != null)
                {
                    addInSettings.oldSelectedWithFrameModeListCommandParameterItemIndex = withFrameModeListParameterItem.SelectedIndex;// 選択されたモードを保持

                    CommandParameterGroup commandParameterGroup = withFrameModeListParameterItem.Group;
                    command.ParameterGroups.Remove(commandParameterGroup);
                }

                // フォームの測定面の高さ変更可に
                locationHeightTextBox2.Enabled = true;
            }
        }

        /// <summary>
        /// 表に関係したコマンドパラメータの更新
        /// </summary>
        /// <param name="command"></param>
        /// <param name="changedSamplingPointParameterID"></param>
        private void UpdateTableLocateCommandParameter(Command command, int changedSamplingPointParameterID = -1)
        {
            // 三斜求積図を作成する場合、「配置図に番号を作図」パラメータを追加する
            if(triclinicTriangleCheckBox.Checked && !onlySkyRateCheckBox.Checked)
            {
                ListParameterItem listParameterItem00 = (ListParameterItem)command.GetParameterItemByID(withNumberModeParameterId);
                if(listParameterItem00 == null)
                {
                    CommandParameterGroup commandParameterGroup0 = command.ParameterGroups.Insert(0, ParameterItemType.List, withNumberModeParameterId, "配置図に番号を作図");
                    listParameterItem00 = (ListParameterItem)command.GetParameterItemByID(withNumberModeParameterId); // パラメータアイテム
                    string[] items0 = { "しない", "する" };
                    listParameterItem00.Items = items0;
                    listParameterItem00.SelectedIndex = 0;
                    listParameterItem00.ValueSaveMode = ParameterValueSaveMode.OneTime; // パラメータ値の保存方法
                    listParameterItem00.Description = "三斜求積図を作成する場合、建物の位置を確認するために図面上の高さを指定した建物の頂点に番号を作図することができます。作図する場合は「する」を選んでください。";
                }
            }
            else
            {
                ListParameterItem listParameterItem00 = (ListParameterItem)command.GetParameterItemByID(withNumberModeParameterId);
                if(listParameterItem00 != null)
                {
                    CommandParameterGroup commandParameterGroup0 = listParameterItem00.Group;
                    command.ParameterGroups.Remove(commandParameterGroup0);
                }
            }

            // 測定点が連続指定か
            int samplingPointParameterItemID = changedSamplingPointParameterID;
            bool samplingPointParameterItemIdIsPoints = false;
            if(samplingPointParameterItemID < 0)
                samplingPointParameterItemID = GetSamplingPointParameterItemID(command);

            Debug.Assert(samplingPointParameterItemID > 0);
            if(samplingPointParameterItemID == samplingPointsParameterId)
                samplingPointParameterItemIdIsPoints = true;

            ListParameterItem tableLocateModeListParameterItem = null;// 表の配置点モードパラメータ

            // 三斜求積図を作成し三斜求積表や建物位置表を作成するとき(測定点(連続)ではなく)(天空率のみ表示でもない)
            if(!samplingPointParameterItemIdIsPoints && triclinicTriangleCheckBox.Checked && (quadratureTableCheckBox.Checked || positionTableCheckBox.Checked) && !onlySkyRateCheckBox.Checked)
            {
                // 表の配置点モード
                // 配置点パラメータの前に挿入する
                ParameterItem locatePointPrameterItem = command.GetParameterItemByID(locatePointParameterId);
                int locatePointPrameterItemIndex = locatePointPrameterItem.Index;

                // リストパラメータ
                tableLocateModeListParameterItem = (ListParameterItem)command.GetParameterItemByID(tableLocateModeParameterId);
                if(tableLocateModeListParameterItem == null)
                {
                    CommandParameterGroup commandParameterGroup3 = command.ParameterGroups.Insert(locatePointPrameterItemIndex, ParameterItemType.List, tableLocateModeParameterId, "表の配置位置");
                    tableLocateModeListParameterItem = (ListParameterItem)command.GetParameterItemByID(tableLocateModeParameterId);
                    string[] items1 = { "下・下", "右・下", "任意" };// 順番は変えないように
                    tableLocateModeListParameterItem.Items = items1;
                    tableLocateModeListParameterItem.SelectedIndex = addInSettings.oldSelectedTableLocateModeListCommandParameterItemIndex;
                    tableLocateModeListParameterItem.ValueSaveMode = ParameterValueSaveMode.Always; // パラメータ値の保存方法
                    tableLocateModeListParameterItem.Description = "三斜求積図を作成する場合、三斜求積表と建物位置表を天空図に対してどこに配置するかを選びます。";
                }

                addInSettings.oldSelectedTableLocateModeListCommandParameterItemIndex = tableLocateModeListParameterItem.SelectedIndex;// 選択されたモードを保持
            }
            else
            {
                tableLocateModeListParameterItem = (ListParameterItem)command.GetParameterItemByID(tableLocateModeParameterId);
                if(tableLocateModeListParameterItem != null)
                {
                    CommandParameterGroup commandParameterGroup1 = tableLocateModeListParameterItem.Group;
                    command.ParameterGroups.Remove(commandParameterGroup1);
                }

                tableLocateModeListParameterItem = null;
            }

            // 三斜求積表を作成するときの表の配置点
            if(tableLocateModeListParameterItem != null && quadratureTableCheckBox.Checked && addInSettings.oldSelectedTableLocateModeListCommandParameterItemIndex == 2) // 2:任意に配置点を指定する場合
            {
                // 座標パラメータを追加する
                PointParameterItem parameterItem4a = (PointParameterItem)command.GetParameterItemByID(table1LocatePointParameterId);
                if(parameterItem4a == null)
                {
                    CommandParameterGroup commandParameterGroup4a = command.ParameterGroups.Add(ParameterItemType.Point, table1LocatePointParameterId, "三斜求積表の配置点");
                    parameterItem4a = (PointParameterItem)command.GetParameterItemByID(table1LocatePointParameterId);
                    parameterItem4a.Description = "配置する三斜求積表の座標を指定してください。";
                }
            }
            else
            {
                PointParameterItem parameterItem4a = (PointParameterItem)command.GetParameterItemByID(table1LocatePointParameterId);
                if(parameterItem4a != null)
                {
                    CommandParameterGroup commandParameterGroup1 = parameterItem4a.Group;
                    command.ParameterGroups.Remove(commandParameterGroup1);
                }
            }

            // 建物位置表を作成するときの表の配置点
            if(tableLocateModeListParameterItem != null && positionTableCheckBox.Checked && addInSettings.oldSelectedTableLocateModeListCommandParameterItemIndex == 2) // 2:任意に配置点を指定する場合
            {
                // 座標パラメータを追加する
                PointParameterItem parameterItem4b = (PointParameterItem)command.GetParameterItemByID(table2LocatePointParameterId);
                if(parameterItem4b == null)
                {
                    CommandParameterGroup commandParameterGroup4b = command.ParameterGroups.Add(ParameterItemType.Point, table2LocatePointParameterId, "建物位置表の配置点");
                    parameterItem4b = (PointParameterItem)command.GetParameterItemByID(table2LocatePointParameterId); // パラメータアイテム
                    parameterItem4b.Description = "配置する建物位置表の座標を指定してください。";
                }
            }
            else
            {
                PointParameterItem parameterItem4b = (PointParameterItem)command.GetParameterItemByID(table2LocatePointParameterId);
                if(parameterItem4b != null)
                {
                    CommandParameterGroup commandParameterGroup1 = parameterItem4b.Group;
                    command.ParameterGroups.Remove(commandParameterGroup1);
                }
            }

        }


        /// <summary>
        /// マウス移動イベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void CommandSkyMap_MouseMove(Object sender, CommandMouseMoveEventArgs e)
        {
            Command command = (Command)sender;
            // Command.GetTemporaryParameterItemByID を使用してコマンドの一時的なパラメータアイテムを取得します。
            // 一時的なパラメータアイテムからは、すでにパラメータの入力が確定している場合は確定している入力値が、
            // 入力が確定していない場合はマウスの現在位置などの一時的な値が取得できます。

            // 天空率のみを表示するフラグ
            bool onlySkyRate = onlySkyRateCheckBox.Checked;

            // 作図倍率
            string drawScaleText = drawScaleComboBox.Text;
            // 文字列をdouble型に変換
            double drawScale = 1.0;
            if(!double.TryParse(drawScaleText, out drawScale) || drawScale <= 0.0 || drawScale > 1000.0)
            {
                return;
            }
           
            //ドキュメントの取得
            Document doc = app.ActiveDocument;
            Drawing drawing = doc.CurrentDrawing;

            int samplePointParameterIndex = 0;// 測定点パラメータインデックス
            // 配置図に番号を作図モードパラメータがある場合は2番目
            bool drawPositionNumberText = false;
            ListParameterItem listParameterItem00 = (ListParameterItem)command.GetParameterItemByID(withNumberModeParameterId);
            if(listParameterItem00 != null)
            {
                samplePointParameterIndex = 1;

                // 配置図に番号を作図モード
                if(listParameterItem00.SelectedIndex == 1)//  { "しない", "する" };
                    drawPositionNumberText = true;
            }

            CommandParameterGroupCollection commandParameterGroupCollection = command.ParameterGroups;
            CommandParameterGroup commandParameterGroup = commandParameterGroupCollection[samplePointParameterIndex];
            ParameterItem samplingPointItem = commandParameterGroup.CurrentItem;
            int samplingPointItemID = samplingPointItem.ID;
            ParameterItem currentParameterItem = command.CurrentParameterItem;

            bool locatePointsIsEmpty = true;
            bool samplingPointsIsEmpty = true;
            // 測定点が点指定の場合
            if(samplingPointItemID == samplingPointParameterId)
            {
                // 測定点           
                PointParameterItem pointParameterItem01 = (PointParameterItem)command.GetParameterItemByID(samplingPointParameterId);
                if(pointParameterItem01 == null)
                    return;

                // 配置点           
                PointParameterItem pointParameterItem02 = (PointParameterItem)command.GetParameterItemByID(locatePointParameterId);
                if(pointParameterItem02 == null)
                    return;

                if(pointParameterItem01.IsEmpty && pointParameterItem02.IsEmpty)
                    return;

                if(!pointParameterItem01.IsEmpty)
                    samplingPointsIsEmpty = false;

                if(!pointParameterItem02.IsEmpty)
                    locatePointsIsEmpty = false;
            }
            // 測定点が点列指定の場合
            else if(samplingPointItemID == samplingPointsParameterId)
            {
                // 測定点配列  
                PointListParameterItem pointListParameterItem = (PointListParameterItem)command.GetParameterItemByID(samplingPointsParameterId);
                if(pointListParameterItem == null)
                    return;

                // 配置点           
                PointParameterItem pointParameterItem02 = (PointParameterItem)command.GetParameterItemByID(locatePointParameterId);
                if(pointParameterItem02 == null)
                    return;

                if(pointListParameterItem.IsEmpty && pointParameterItem02.IsEmpty)
                    return;

                if(!pointListParameterItem.IsEmpty)
                    samplingPointsIsEmpty = false;

                if(!pointParameterItem02.IsEmpty)
                    locatePointsIsEmpty = false;

                // 配置点が未確定で測定点が確定していない場合は測定点のラバーバンドだけ
                if(locatePointsIsEmpty == true && !pointListParameterItem.Complete)
                {
                    DrawRubberBandSamplingPointNumber(command, doc, drawScale);
                    return;
                }
            }

            // 表の配置位置
            // リストパラメータ
            SkyMapTableLocateMode skyMapTableLocateMode = SkyMapTableLocateMode.SkyMapTableLocateMode_LowerLower;// 配置位置モード（初期値："下・下"）
            Point2d quadratureTableLocatePoint = new Point2d();// 三斜求積表の配置点座標
            Point2d positionTableLocatePoint = new Point2d();// 建物位置表の配置点座標
            bool makeQuadratureTable = false;// 三斜求積表作成フラグ
            bool makePositionTable = false;// 建物位置表作成フラグ
            if(triclinicTriangleCheckBox.Checked && !onlySkyRate)
            {
                // 表の配置の仕方パラメータがあって、尚且つ測定点と配置点どちらかが確定している場合に表を作る
                ListParameterItem tableLocateModeListParameterItem = (ListParameterItem)command.GetParameterItemByID(tableLocateModeParameterId);
                if(tableLocateModeListParameterItem != null && (!samplingPointsIsEmpty || !locatePointsIsEmpty))
                {
                    skyMapTableLocateMode = (SkyMapTableLocateMode)tableLocateModeListParameterItem.SelectedIndex;// 表の配置位置:"下・下", "右・下", "任意"

                    // 表の配置位置が「任意」だった場合、表の配置点座標を取得
                    if(skyMapTableLocateMode == SkyMapTableLocateMode.SkyMapTableLocateMode_Free)
                    {
                        // 三斜求積表の配置点座標
                        if(quadratureTableCheckBox.Checked)
                        {
                            bool dontMakeQuadratureTable = false;
                            PointParameterItem pointParameterItem03 = (PointParameterItem)command.GetParameterItemByID(table1LocatePointParameterId);
                            if(pointParameterItem03 == null)
                                return;

                            // 建物位置表も作成する場合、三斜求積表と建物位置の配置点が決まっていなければ建物位置表は一時パラメーターで作成しない
                            if(positionTableCheckBox.Checked && !locatePointsIsEmpty)
                            {
                                PointParameterItem pointParameterItem04 = (PointParameterItem)command.GetParameterItemByID(table2LocatePointParameterId);
                                if(pointParameterItem03.IsEmpty && pointParameterItem04.IsEmpty)
                                {
                                    if(currentParameterItem.ID != table1LocatePointParameterId)
                                        dontMakeQuadratureTable = true;
                                }
                            }

                            // 三斜求積表の配置点が未確定でカレントが測定点か配置点のときも作らない
                            if(pointParameterItem03.IsEmpty && (currentParameterItem.ID == samplingPointParameterId || currentParameterItem.ID == locatePointParameterId))
                                dontMakeQuadratureTable = true;

                            if(!dontMakeQuadratureTable)
                            {
                                pointParameterItem03 = (PointParameterItem)command.GetTemporaryParameterItemByID(table1LocatePointParameterId);
                                if(pointParameterItem03.IsEmpty)
                                    return;

                                quadratureTableLocatePoint = pointParameterItem03.Point;
                                makeQuadratureTable = true;
                            }
                        }

                        // 建物位置表の配置点座標
                        if(positionTableCheckBox.Checked)
                        {
                            bool dontMakePositionTable = false;
                            PointParameterItem pointParameterItem04 = (PointParameterItem)command.GetParameterItemByID(table2LocatePointParameterId);
                            if(pointParameterItem04 == null)
                                return;

                            // 三斜求積表も作成する場合、三斜求積表と建物位置の配置点が決まっていなければ建物位置表は一時パラメーターで作成しない
                            if(quadratureTableCheckBox.Checked && !locatePointsIsEmpty)
                            {
                                PointParameterItem pointParameterItem03 = (PointParameterItem)command.GetParameterItemByID(table1LocatePointParameterId);
                                if(pointParameterItem03 == null)
                                    return;

                                if(pointParameterItem03.IsEmpty && pointParameterItem04.IsEmpty)
                                {
                                    if(currentParameterItem.ID != table2LocatePointParameterId)
                                        dontMakePositionTable = true;
                                }
                            }

                            // 建物位置表の配置点が未確定でカレントが測定点か配置点のときも作らない
                            if(pointParameterItem04.IsEmpty && (currentParameterItem.ID == samplingPointParameterId || currentParameterItem.ID == locatePointParameterId))
                                dontMakePositionTable = true;

                            if(!dontMakePositionTable)
                            {
                                pointParameterItem04 = (PointParameterItem)command.GetTemporaryParameterItemByID(table2LocatePointParameterId);
                                if(pointParameterItem04.IsEmpty)
                                    return;

                                positionTableLocatePoint = pointParameterItem04.Point;
                                makePositionTable = true;
                            }
                        }
                    }
                    else
                    {
                        makeQuadratureTable = quadratureTableCheckBox.Checked;
                        makePositionTable = positionTableCheckBox.Checked;
                    }
                }
                else
                {
                    makeQuadratureTable = quadratureTableCheckBox.Checked;
                    makePositionTable = positionTableCheckBox.Checked;
                }
            }

            // 一時パラメータを取得する(測定点)
            List<Point2d> samplingPointList = new List<Point2d>();
            List<double> locationHeightList = null;// 測定面の高さ(連続)のリスト
            // 測定点が点指定の場合
            if(samplingPointItemID == samplingPointParameterId)
            {
                PointParameterItem pointParameterItem1 = (PointParameterItem)command.GetTemporaryParameterItemByID(samplingPointParameterId);
                if(pointParameterItem1 == null)
                    return;

                if(pointParameterItem1.IsEmpty)
                    return;

                Point2d samplingPoint = new Point2d(pointParameterItem1.Point.X, pointParameterItem1.Point.Y);
                samplingPointList.Add(samplingPoint);

                // 測定面の高さ
                string locationHeightTextBoxText = locationHeightTextBox2.Text;// フォームから取得
                // 文字列をdouble型に変換
                double locationHeight = 0.0;
                if(!double.TryParse(locationHeightTextBoxText, out locationHeight))
                {
                    //MessageBox.Show("測定面の高さの数値が正しくありません。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    //return;
                    locationHeight = 0.0;
                }

                locationHeightList = new List<double>();
                locationHeightList.Add(locationHeight);

            }
            // 測定点が点列指定の場合
            else if(samplingPointItemID == samplingPointsParameterId)
            {
                PointListParameterItem pointListParameterItem = (PointListParameterItem)command.GetTemporaryParameterItemByID(samplingPointsParameterId);
                int pointSize = pointListParameterItem.Points.Length;
                for(int i = 0; i < pointSize; i++)
                {
                    Point2d samplingPoint = new Point2d(pointListParameterItem.Points[i].X, pointListParameterItem.Points[i].Y);
                    samplingPointList.Add(samplingPoint);
                }

                // 測定面の高さ(連続)パラメータから点列数に合わせた高さのリスト取得
                locationHeightList = GetLocationHeightList(command, pointSize, true);
                if(locationHeightList == null)
                    return;
            }

            // 一時パラメータを取得する(配置点)
            PointParameterItem pointParameterItem2 = (PointParameterItem)command.GetTemporaryParameterItemByID(locatePointParameterId);
            if(pointParameterItem2 == null)
                return;

            if(pointParameterItem2.IsEmpty)
                return;

            // 配置点
            Point2d locatePoint = pointParameterItem2.Point;

            // 配置点が決定しているなら
            if(locatePointsIsEmpty == false)
            {
                // 測定点が確定していない場合はラバーバンドは作り直し
                ClearRubberBand();
            }

            // 表を作成する時の配置座標
            Point2d firstPoint = samplingPointList.First();
            Point2d tempQuadratureTableLocatePoint = firstPoint;
            Point2d tempPositionTableLocatePoint = firstPoint;
            Point2d quadratureTableLocatePoint2 = locatePoint;
            Point2d positionTableLocatePoint2 = locatePoint;
            // 表の配置位置が「任意」だった場合
            if(skyMapTableLocateMode == SkyMapTableLocateMode.SkyMapTableLocateMode_Free)
            {
                tempQuadratureTableLocatePoint = new Point2d();
                tempPositionTableLocatePoint = new Point2d();

                quadratureTableLocatePoint2 = quadratureTableLocatePoint;
                positionTableLocatePoint2 = positionTableLocatePoint;
            }

            // 図形の作成
            List<Shape> quadratureTableShapes = null;// 三斜求積表
            List<Shape> positionTableShapes = null;// 位置表
            List<PositionNumberShape> optionShapes = null;// 配置図への位置番号文字
            int samplingPointNumber = Decimal.ToInt32(samplingPointNumberUpDown.Value);// 測定点番号
            int samplingPointListCount = samplingPointList.Count();

            Debug.Assert(locationHeightList != null);
            Debug.Assert(locationHeightList.Count() == samplingPointListCount);

            if(rubberBandShapes == null)
            {
                // 配置点はとりあえず測定点で作成する
                for(int i = 0; i < samplingPointListCount; i++)
                {
                    Point2d samplingPoint = samplingPointList[i];   // 測定点
                    double locationHeight = locationHeightList[i];  // 測定面の高さ
                    List<Shape> shapes = CreateShapesForSkyMapCommand(drawing, drawPositionNumberText, samplingPoint, locationHeight, samplingPointNumber + i, makeQuadratureTable, makePositionTable, skyMapTableLocateMode, tempQuadratureTableLocatePoint, tempPositionTableLocatePoint, out quadratureTableShapes, out positionTableShapes, out optionShapes, drawScale, onlySkyRate, true);
                    if(shapes == null)
                    {
                        app.CommandManager.CancelCurrentCommand();
                        return;
                    }

                    if(rubberBandShapes == null)
                        rubberBandShapes = new List<Shape>();

                    // 天空図
                    rubberBandShapes.AddRange(shapes);

                    rubberBandShapesCountList.Add(shapes.Count());// 図形数を保持

                    // 三斜求積表
                    if(quadratureTableShapes != null)
                    {
                        if(rubberBandTable1Shapes == null)
                            rubberBandTable1Shapes = new List<Shape>();

                        rubberBandTable1Shapes.AddRange(quadratureTableShapes);

                        rubberBandTable1ShapesCountList.Add(quadratureTableShapes.Count());// 図形数を保持

                    }

                    // 位置表
                    if(positionTableShapes != null)
                    {
                        if(rubberBandTable2Shapes == null)
                            rubberBandTable2Shapes = new List<Shape>();

                        rubberBandTable2Shapes.AddRange(positionTableShapes);

                        rubberBandTable2ShapesCountList.Add(positionTableShapes.Count());// 図形数を保持
                    }

                    // 配置図の番号(最初の一回目のものだけ使う)
                    if(optionShapes != null && rubberBandOptionShapes == null)
                    {
                        rubberBandOptionShapes = new List<Shape>();

                        foreach(PositionNumberShape optionShape in optionShapes)
                            rubberBandOptionShapes.AddRange(optionShape.Shapes);
                    }
                }
            }

            List<Shape> rubberBandShapes2 = new List<Shape>();

            // 天空図
            Debug.Assert(rubberBandShapes != null);
            double hemisphereRadius = (double)GetHemisphereRadius(true);// 天空図半径(整数で受け取る)
            if(hemisphereRadius < 0.0)
                return;

            double textSizeRatio = GetSkymapTextSizeRatio();// 文字大きさ調整値も反映させる
            // 測定点を連続指定したときの配置点間隔(幅)
            double spaceX = GetSpaceX(command, hemisphereRadius, textSizeRatio, drawScale);

            Debug.Assert(samplingPointListCount == rubberBandShapesCountList.Count);

            Point2d firstCircleLocatePoint = new Point2d(locatePoint.X, locatePoint.Y);

            // 枠を作成する場合
            ListParameterItem withFrameParameterItem = (ListParameterItem)command.GetParameterItemByID(withFrameParameterId);
            if(withFrameParameterItem != null)
            {
                int frameType = withFrameParameterItem.SelectedIndex;
                if(frameType > 0)
                {
                    double circleRadius = hemisphereRadius * drawScale;
                    List<Shape> shapes = CreateFrameShapes(drawing, locatePoint, circleRadius, spaceX, frameType, samplingPointListCount, out firstCircleLocatePoint);
                    foreach(Shape shape in shapes)
                        rubberBandShapes2.Add(shape);
                }
            }

            // 天空図と表
            int lastRubberBandShapesCounter = 0;
            int lastRubberBandTable1ShapesCounter = 0;
            int lastRubberBandTable2ShapesCounter = 0;
            for(int i = 0; i < samplingPointListCount; i++)
            {
                Point2d samplingPoint = samplingPointList[i];
                int shapeCount = rubberBandShapesCountList[i];
                // マウスの位置に移動
                Point2d locatePoint2 = new Point2d(firstCircleLocatePoint.X + spaceX * (double)i, firstCircleLocatePoint.Y);

                // 天空図
                for(int j = lastRubberBandShapesCounter; j < lastRubberBandShapesCounter + shapeCount; j++)
                {
                    Shape shape = rubberBandShapes[j];
                    Shape shape2 = shape.Clone();
                    shape2.Transform(samplingPoint, drawScale, drawScale, 0.0, locatePoint2);
                    rubberBandShapes2.Add(shape2);
                }

                lastRubberBandShapesCounter += shapeCount;

                // 三斜求積表
                if(rubberBandTable1Shapes != null)
                {
                    Debug.Assert(samplingPointListCount == rubberBandTable1ShapesCountList.Count);

                    int rubberBandTable1ShapesCount = rubberBandTable1ShapesCountList[i];
                    for(int j = lastRubberBandTable1ShapesCounter; j < lastRubberBandTable1ShapesCounter + rubberBandTable1ShapesCount; j++)
                    {
                        Shape shape = rubberBandTable1Shapes[j];

                        // マウスの位置に移動
                        Shape shape2 = shape.Clone();

                        // 配置位置が任意の場合(測定点は1点指定のみ)
                        if(skyMapTableLocateMode == SkyMapTableLocateMode.SkyMapTableLocateMode_Free)
                        {
                            Debug.Assert(samplingPointListCount == 1);
                            shape2.Transform(tempQuadratureTableLocatePoint, drawScale, drawScale, 0.0, quadratureTableLocatePoint2);
                        }
                        else
                        {
                            shape2.Transform(samplingPoint, drawScale, drawScale, 0.0, locatePoint2);
                        }

                        rubberBandShapes2.Add(shape2);
                    }

                    lastRubberBandTable1ShapesCounter += rubberBandTable1ShapesCount;
                }

                // 位置表
                if(rubberBandTable2Shapes != null)
                {
                    Debug.Assert(samplingPointListCount == rubberBandTable2ShapesCountList.Count);

                    int rubberBandTable2ShapesCount = rubberBandTable2ShapesCountList[i];
                    for(int j = lastRubberBandTable2ShapesCounter; j < lastRubberBandTable2ShapesCounter + rubberBandTable2ShapesCount; j++)
                    {
                        Shape shape = rubberBandTable2Shapes[j];

                        // マウスの位置に移動
                        Shape shape2 = shape.Clone();

                        // 配置位置が任意の場合(測定点は1点指定のみ)
                        if(skyMapTableLocateMode == SkyMapTableLocateMode.SkyMapTableLocateMode_Free)
                        {
                            Debug.Assert(samplingPointListCount == 1);
                            shape2.Transform(tempPositionTableLocatePoint, drawScale, drawScale, 0.0, positionTableLocatePoint2);
                        }
                        else
                        {
                            shape2.Transform(samplingPoint, drawScale, drawScale, 0.0, locatePoint2);
                        }

                        rubberBandShapes2.Add(shape2);
                    }

                    lastRubberBandTable2ShapesCounter += rubberBandTable2ShapesCount;
                }
            }

            // 配置図の番号
            if(rubberBandOptionShapes != null && rubberBandOptionShapes.Count > 0)
                rubberBandShapes2.AddRange(rubberBandOptionShapes);

            // 測定点の番号作図図形
            if(samplingPointNumberCheckBox.Checked && drawSamplingPointNumberCheckBox.Checked)
            {
                double textHight = hemisphereRadius * customSettings.skymapDefaultTextSizeRatio * textSizeRatio; // 表中などの文字の大きさ、天空図円の大きさのx%を元にしてサイズ調整
                double positionNumberTextHight = textHight * drawScale;// 配置図の位置を示す番号文字の大きさ
                List<Shape> samplingPointNumberShapes = CreateSamplingPointNumberShapes(samplingPointList, positionNumberTextHight, doc);
                rubberBandShapes2.AddRange(samplingPointNumberShapes);
            }

            try
            {
                // ラバーバンド描画
                command.DrawShapeRubberBand(rubberBandShapes2.ToArray());
            }
            catch
            {
                throw;
            }
        }

        /// <summary>
        /// 測定点の番号だけのラバーバンド描画
        /// </summary>
        /// <param name="command"></param>
        /// <param name="doc"></param>
        /// <param name="drawScale"></param>
        void DrawRubberBandSamplingPointNumber(Command command, Document doc, double drawScale)
        {
            if(samplingPointNumberCheckBox.Checked && drawSamplingPointNumberCheckBox.Checked)
            {
                List<Point2d> samplingPointList = new List<Point2d>();
                PointListParameterItem pointListParameterItem = (PointListParameterItem)command.GetTemporaryParameterItemByID(samplingPointsParameterId);
                int pointSize = pointListParameterItem.Points.Length;
                for(int i = 0; i < pointSize; i++)
                {
                    Point2d samplingPoint = new Point2d(pointListParameterItem.Points[i].X, pointListParameterItem.Points[i].Y);
                    samplingPointList.Add(samplingPoint);
                }

                List<Shape> rubberBandShapes2 = new List<Shape>();

                double hemisphereRadius = (double)GetHemisphereRadius(true);// 天空図半径(整数で受け取る)
                double textSizeRatio = GetSkymapTextSizeRatio();// 文字大きさ調整値も反映させる

                double textHight = hemisphereRadius * customSettings.skymapDefaultTextSizeRatio * textSizeRatio; // 表中などの文字の大きさ、天空図円の大きさのx%を元にしてサイズ調整
                double positionNumberTextHight = textHight * drawScale;// 配置図の位置を示す番号文字の大きさ
                List<Shape> samplingPointNumberShapes = CreateSamplingPointNumberShapes(samplingPointList, positionNumberTextHight, doc);
                rubberBandShapes2.AddRange(samplingPointNumberShapes);

                try
                {
                    // ラバーバンド描画
                    command.DrawShapeRubberBand(rubberBandShapes2.ToArray());
                }
                catch
                {
                    throw;
                }
            }
        }


        /// <summary>
        /// コマンドパラメータ変更イベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void CommandSkyMap_ParameterChanged(Object sender, CommandParameterChangedEventArgs e)
        {
            //ラバーバンド表示を削除する
            ClearRubberBand();
 
            Command command = (Command)sender;

            // 天空率のみを表示するフラグ
            bool onlySkyRate = onlySkyRateCheckBox.Checked;

            //ドキュメントの取得
            Document doc = app.ActiveDocument;
            Drawing drawing = doc.CurrentDrawing;

            // 作図倍率
            string drawScaleText = drawScaleComboBox.Text;
            //文字列をdouble型に変換
            double drawScale = 1.0;
            if(!double.TryParse(drawScaleText, out drawScale) || drawScale <= 0.0 || drawScale > 1000.0)
            {
                string strErr = "作図倍率が適正な値ではありません。";
                MessageBox.Show(strErr, AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                e.CommandState = CommandState.End;
                goto ENDLABEL;
            }

            // パラメータの「配置図に番号を作図」モードが変わった場合と
            if(e.ParameterID == withNumberModeParameterId)
            {
                // この関数の最後にこのモードを「しない」にする処理があって、そのときのイベントでもここにくる。
                // そのまま実行すると2度図形を作図してしまうため、このパラメータが変わったときはなにもしない。
                return;
            }

            // パラメータの表の配置位置のモードが変わった場合
            //if(e.ParameterID == tableLocateModeParameterId) // プログラム中でパラメーターの切り替えで追加したとき、
            // 選択するモードをセットするとイベントが発生するので、ここはカレントでったときだけにする
            if(command.CurrentParameterItem.ID == tableLocateModeParameterId)
            {
                // 表に関係したコマンドパラメータの更新
                UpdateTableLocateCommandParameter(command);
            }

            int samplePointParameterIndex = 0;// 測定点パラメータインデックス
            // 配置図に番号を作図モードパラメータがある場合は2番目
            bool drawPositionNumberText = false;
            ListParameterItem withNumberModeParameterItem = (ListParameterItem)command.GetParameterItemByID(withNumberModeParameterId);
            if(withNumberModeParameterItem != null)
            {
                samplePointParameterIndex = 1;
                if(withNumberModeParameterItem.SelectedIndex == 1)//  { "しない", "する" };
                    drawPositionNumberText = true;
            }

            CommandParameterGroupCollection commandParameterGroupCollection = command.ParameterGroups;
            CommandParameterGroup commandParameterGroup = commandParameterGroupCollection[samplePointParameterIndex];
            ParameterItem samplingPointItem = commandParameterGroup.CurrentItem;
            int samplingPointItemID = samplingPointItem.ID;

            // 測定点
            List<Point2d> samplingPointList = new List<Point2d>();
            List<double> locationHeightList = null;// 測定面の高さ(連続)のリスト
            // 測定点が点指定の場合
            if(samplingPointItemID == samplingPointParameterId)
            {
                PointParameterItem pointParameterItem1 = (PointParameterItem)command.GetParameterItemByID(samplingPointParameterId);
                if(pointParameterItem1 == null)
                    return;

                if(pointParameterItem1.IsEmpty)
                    return;

                Point2d samplingPoint = new Point2d(pointParameterItem1.Point.X, pointParameterItem1.Point.Y);
                samplingPointList.Add(samplingPoint);

                // 測定面の高さ
                string locationHeightTextBoxText = locationHeightTextBox2.Text;// フォームから取得
                // 文字列をdouble型に変換
                double locationHeight = 0.0;
                if(!double.TryParse(locationHeightTextBoxText, out locationHeight))
                {
                    MessageBox.Show("測定面の高さの数値が正しくありません。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return;
                }

                locationHeightList = new List<double>();
                locationHeightList.Add(locationHeight);

            }
            // 測定点が点列指定の場合
            else if(samplingPointItemID == samplingPointsParameterId)
            {
                // 測定点配列  
                PointListParameterItem pointListParameterItem = (PointListParameterItem)command.GetParameterItemByID(samplingPointsParameterId);
                if(pointListParameterItem == null)
                    return;

                // 点列指定で空なら何もしない
                if(pointListParameterItem.IsEmpty)
                    return;

                int pointSize = pointListParameterItem.Points.Length;
                if(pointSize > customSettings.maxSamplingPointSize)
                {
                    string strErr = "点列数が多すぎます。";
                    MessageBox.Show(strErr, AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    e.CommandState = CommandState.End;
                    return;
                }

                // 入力が完了していなければ続ける(配置点がセットされていると1点指定しただけで終了してしまうので、そうならないようにする。Enterキーなどで完了させる)
                if(!pointListParameterItem.Complete)
                {
                    e.CommandState = CommandState.Continue;
                    return;
                }

                for(int i = 0; i < pointSize; i++)
                {
                    Point2d samplingPoint = new Point2d(pointListParameterItem.Points[i].X, pointListParameterItem.Points[i].Y);
                    samplingPointList.Add(samplingPoint);
                }

                // 測定面の高さ(連続)パラメータから点列数に合わせた高さのリスト取得
                locationHeightList = GetLocationHeightList(command, pointSize, false);
                if(locationHeightList == null)
                {
                    string strErr = "測定面の高さ(連続)の入力が正しくありません。";// 測定面の高さ(連続)が測定点の数より多いか、不正な文字が含まれている
                    MessageBox.Show(strErr, AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    e.CommandState = CommandState.Continue;
                    return;
                }

                int locationHeightListCount = locationHeightList.Count();
                if(locationHeightListCount > 0 && pointSize > 0 && locationHeightListCount != pointSize)
                {
                    string strErr = "測定点の数と測定面の高さの数が一致していません。";
                    MessageBox.Show(strErr, AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    e.CommandState = CommandState.Continue;
                    return;
                }
            }

            // 配置点
            PointParameterItem pointParameterItem2 = (PointParameterItem)command.GetParameterItemByID(locatePointParameterId);
            if(pointParameterItem2 == null)
                return;

            if(pointParameterItem2.IsEmpty)
                return;

            Point2d locatePoint = pointParameterItem2.Point;

            // 表の配置位置
            // リストパラメータ
            SkyMapTableLocateMode skyMapTableLocateMode = SkyMapTableLocateMode.SkyMapTableLocateMode_LowerLower;// 配置位置モード（初期値："下・下"）
            Point2d quadratureTableLocatePoint = new Point2d();// 三斜求積表の配置点座標
            Point2d positionTableLocatePoint = new Point2d();// 建物位置表の配置点座標
            bool makeQuadratureTable = false;// 三斜求積表作成フラグ
            bool makePositionTable = false;// 建物位置表作成フラグ
            // 三斜求積図作成で、各オプションの状態に従う
            if(triclinicTriangleCheckBox.Checked && !onlySkyRate)
            {
                ListParameterItem tableLocateModeListParameterItem = (ListParameterItem)command.GetParameterItemByID(tableLocateModeParameterId);
                if(tableLocateModeListParameterItem != null)
                {
                    skyMapTableLocateMode = (SkyMapTableLocateMode)tableLocateModeListParameterItem.SelectedIndex;// 表の配置位置:"下・下", "右・下", "任意"

                    // 表の配置位置が「任意」だった場合、表の配置点座標を取得
                    if(skyMapTableLocateMode == SkyMapTableLocateMode.SkyMapTableLocateMode_Free)
                    {
                        // 三斜求積表の配置点座標
                        if(quadratureTableCheckBox.Checked)
                        {
                            PointParameterItem pointParameterItem03 = (PointParameterItem)command.GetParameterItemByID(table1LocatePointParameterId);
                            if(pointParameterItem03 == null)
                                return;

                            if(pointParameterItem03.IsEmpty)
                                return;

                            quadratureTableLocatePoint = pointParameterItem03.Point;
                            makeQuadratureTable = true;
                        }

                        // 建物位置表の配置点座標
                        if(positionTableCheckBox.Checked)
                        {
                            PointParameterItem pointParameterItem04 = (PointParameterItem)command.GetParameterItemByID(table2LocatePointParameterId);
                            if(pointParameterItem04 == null)
                                return;

                            if(pointParameterItem04.IsEmpty)
                                return;

                            positionTableLocatePoint = pointParameterItem04.Point;
                            makePositionTable = true;
                        }
                    }
                    else
                    {
                        makeQuadratureTable = quadratureTableCheckBox.Checked;
                        makePositionTable = positionTableCheckBox.Checked;
                    }
                }
                else
                {
                    makeQuadratureTable = quadratureTableCheckBox.Checked;
                    makePositionTable = positionTableCheckBox.Checked;
                }
            }

            // ウェイトカーソル
            Cursor.Current = Cursors.WaitCursor;

            // 図形の作成
            List<Shape> quadratureTableShapes = null;// 三斜求積表
            List<Shape> positionTableShapes = null;// 位置表
            List<PositionNumberShape> optionShapes = null;// 配置図への位置番号文字
            double hemisphereRadius = (double)GetHemisphereRadius();// 天空図半径(整数で受け取る)
            if(hemisphereRadius < 0.0)
            {
                string strErr = "天空図の半径値が正しく入力されていません。";
                MessageBox.Show(strErr, AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);

                // 入力可能な状態に
                hemisphereRadiusUpDown.Focus();
                hemisphereRadiusUpDown.Select(0, hemisphereRadiusUpDown.Text.Length); // テキスト全選択

                e.CommandState = CommandState.End;
                return;
            }


            double textSizeRatio = GetSkymapTextSizeRatio();// 文字大きさ調整値も反映させる
            // 測定点を連続指定したときの配置点間隔(幅)
            double spaceX = GetSpaceX(command, hemisphereRadius, textSizeRatio, drawScale);

            int samplingPointNumber = Decimal.ToInt32(samplingPointNumberUpDown.Value);// 測定点番号
            int samplingPointListCount = samplingPointList.Count();

            List<Shape> shapes2 = new List<Shape>();
            Point2d firstCircleLocatePoint = new Point2d(locatePoint.X, locatePoint.Y);

            // 枠を作成する場合
            ListParameterItem withFrameParameterItem = (ListParameterItem)command.GetParameterItemByID(withFrameParameterId);
            if(withFrameParameterItem != null)
            {
                int frameType = withFrameParameterItem.SelectedIndex;
                if(frameType > 0)
                {
                    double circleRadius = hemisphereRadius * drawScale;
                    List<Shape> shapes = CreateFrameShapes(drawing, locatePoint, circleRadius, spaceX, frameType, samplingPointListCount, out firstCircleLocatePoint);
                    shapes2.AddRange(shapes);
                }
            }
            
            Debug.Assert(locationHeightList != null);
            Debug.Assert(locationHeightList.Count() == samplingPointListCount);

            // 天空図と表
            for(int i = 0; i < samplingPointListCount; i++)
            {
                Point2d samplingPoint = samplingPointList[i];   // 測定点
                double locationHeight = locationHeightList[i];  // 測定面の高さ
                List<Shape> shapes = CreateShapesForSkyMapCommand(drawing, drawPositionNumberText, samplingPoint, locationHeight, samplingPointNumber + i, makeQuadratureTable, makePositionTable, skyMapTableLocateMode, quadratureTableLocatePoint, positionTableLocatePoint, out quadratureTableShapes, out positionTableShapes, out optionShapes, drawScale, onlySkyRate);
                if(shapes == null)
                {
                    //e.CommandState = CommandState.Continue;
                    e.CommandState = CommandState.End;
                    goto ENDLABEL;
                }

                // 表の配置位置が「任意」以外だった場合は表の図形はここで追加
                if(skyMapTableLocateMode != SkyMapTableLocateMode.SkyMapTableLocateMode_Free)
                {
                    // 三斜求積表
                    if(quadratureTableShapes != null)
                        shapes.AddRange(quadratureTableShapes);

                    // 位置表
                    if(positionTableShapes != null)
                        shapes.AddRange(positionTableShapes);
                }

                // マウスの位置に移動
                Point2d locatePoint2 = new Point2d(firstCircleLocatePoint.X + spaceX * (double)i, firstCircleLocatePoint.Y);
                foreach(Shape shape in shapes)
                {
                    shape.Transform(samplingPoint, drawScale, drawScale, 0.0, locatePoint2);
                    shapes2.Add(shape);
                }

                // 表の配置位置が「任意」だった場合は表の図形はここで追加
                if(skyMapTableLocateMode == SkyMapTableLocateMode.SkyMapTableLocateMode_Free)
                {
                    // 三斜求積表
                    if(quadratureTableShapes != null)
                    {
                        foreach(Shape shape1 in quadratureTableShapes)
                        {
                            // 拡大・縮小
                            shape1.Scale(quadratureTableLocatePoint, drawScale, drawScale);
                            shapes2.Add(shape1);
                        }
                    }

                    // 位置表
                    if(positionTableShapes != null)
                    {
                        foreach(Shape shape1 in positionTableShapes)
                        {
                            // 拡大・縮小
                            shape1.Scale(positionTableLocatePoint, drawScale, drawScale);
                            shapes2.Add(shape1);
                        }
                    }
                }

                // 配置図の位置番号(最初の1回目のだけ追加)
                if(optionShapes != null && i == 0)
                {
                    foreach(PositionNumberShape optionShape in optionShapes)
                        shapes2.AddRange(optionShape.Shapes);
                }
            }

            // 測定点の番号作図図形
            if(samplingPointNumberCheckBox.Checked && drawSamplingPointNumberCheckBox.Checked)
            {
                double textHight = hemisphereRadius * customSettings.skymapDefaultTextSizeRatio * textSizeRatio; // 表中などの文字の大きさ、天空図円の大きさのx%を元にしてサイズ調整
                double positionNumberTextHight = textHight * drawScale;// 配置図の位置を示す番号文字の大きさ
                List<Shape> samplingPointNumberShapes = CreateSamplingPointNumberShapes(samplingPointList, positionNumberTextHight, doc);
                shapes2.AddRange(samplingPointNumberShapes);
            }

            // Undoの開始
            doc.UndoManager.BeginUndoUnit();

            // 図面上にプロットする

            // 作図
            foreach(Shape shape in shapes2)
            {
                // 描画図形追加
                drawing.Shapes.Add(shape);
            }

            // Undo の終了
            doc.UndoManager.EndUndoUnit();

            // コマンド実行
            e.CommandState = CommandState.Execute;

            // 「配置図に番号を作図」パラメータの初期値セットし直し
            // (変更イベントが発生するのでこの関数のはじめでこのパラメータ時は何もしないようにしている)
            if(withNumberModeParameterItem != null)
                withNumberModeParameterItem.SelectedIndex = 0;// "しない"

            // 測定点No.カウントアップ
            if(samplingPointNumberCheckBox.Checked == true)
            {
                samplingPointNumberUpDown.Value += samplingPointListCount;
                samplingPointNumberUpDown.Update();
            }

        ENDLABEL:

            //ラバーバンドクリア
            ClearRubberBand();

        }

        /// <summary>
        /// 測定点が点列指定の場合の測定面の高さリスト取得
        /// </summary>
        /// <param name="command"></param>
        /// <param name="pointSize"></param>
        /// <param name="defaultZero">測定点の数に足りない部分は0</param>
        /// <returns></returns>
        private List<double> GetLocationHeightList(Command command, int pointSize, bool defaultZero)
        {
            //PointListParameterItem pointListParameterItem = (PointListParameterItem)command.GetTemporaryParameterItemByID(samplingPointsParameterId);
            //Debug.Assert(pointListParameterItem != null);
            //int pointSize = pointListParameterItem.Points.Length;

            // 測定面の高さ(連続)パラメータ
            TextParameterItem samplingPointHeightsParameterItem = (TextParameterItem)command.GetParameterItemByID(samplingPointHeightsParameterId);
            Debug.Assert(samplingPointHeightsParameterItem != null);

            string itemText = samplingPointHeightsParameterItem.Text;

            List<string> textList = itemText.Split(',').ToList();
            int textListCount = itemText == "" ? 0:textList.Count;
            int count = defaultZero ? pointSize : textListCount;

            // Defaultなしのとき文字が多すぎる場合nullで返す
            if(!defaultZero && textListCount > pointSize)
                return null;

            List<double> locationHeightList = new List<double>();
            for(int i = 0; i < count; i++)
            {
                if(i < textListCount)
                {
                    string text = textList[i];
                    text = text.Trim();// 前後の空白削除

                    double locationHeight = 0.0;
                    if(double.TryParse(text, out locationHeight))
                    {
                        locationHeightList.Add(locationHeight);
                    }
                    // 不正な文字があったとしてnullで返す
                    else
                    {
                        return null;
                    }
                }
                else
                {
                    locationHeightList.Add(0.0);
                }
            }

            return locationHeightList;
        }

        /// <summary>
        /// 測定点の番号図形リスト作成
        /// </summary>
        /// <param name="samplingPointList"></param>
        /// <param name="textHight"></param>
        /// <param name="doc"></param>
        /// <returns></returns>
        private List<Shape> CreateSamplingPointNumberShapes(List<Point2d> samplingPointList, double textHight, Document doc)
        {
            List<Shape> samplingPointNumberShapes = new List<Shape>();

            int startNumber = (int)samplingPointNumberUpDown.Value;

            int listSize = samplingPointList.Count();
            for(int i = 0; i < listSize; i++)
            {
                string samplingPointNumberText = customSettings.samplingNumberHeadText + (startNumber + i).ToString();
                Point2d textLocatePoint = samplingPointList[i];

                Layer currentLayer = doc.LayerTable.CurrentLayer;
                AddTextShape(samplingPointNumberShapes, samplingPointNumberText, textLocatePoint, 0.0, textHight, 0.0, Alignment.BottomLeft, customSettings.triclinicTextColorNumber, doc, currentLayer);

                // 点マーク
                double pointMarkerSize = textHight * addInSettings.samplingPointMarkerSizeRatio;
                AddPointMarkerShape(samplingPointNumberShapes, textLocatePoint, customSettings.samplingPointMarkerType, pointMarkerSize, -1, doc, currentLayer);
            }

            return samplingPointNumberShapes;
        }


        /// <summary>
        /// 測定点を連列で指定して天空図を複数配置するときの間隔取得
        /// </summary>
        /// <param name="command"></param>
        /// <param name="hemisphereRadius"></param>
        /// <param name="textSizeRatio"></param>
        /// <param name="drawScale"></param>
        /// <returns></returns>
        private double GetSpaceX(Command command, double hemisphereRadius, double textSizeRatio, double drawScale)
        {
            double spaceXRatio = customSettings.skymapSpaceRatioDefault;

            // 測定点を連続指定したときの配置点間隔(幅) 自動のときの値を初期値に
            double spaceX = (2.0 * hemisphereRadius + hemisphereRadius * spaceXRatio * textSizeRatio) * drawScale;

            ListParameterItem listParameterItem = (ListParameterItem)command.GetParameterItemByID(spaceXParameterId);
            if(listParameterItem != null)
            {
                int selectedIndex = listParameterItem.SelectedIndex;
                string selectedItemText = listParameterItem.Items[selectedIndex];
                // 自動以外だった場合
                if(selectedIndex > 0)
                {
                    if(!double.TryParse(selectedItemText, out spaceXRatio))
                    {
                        Debug.Assert(false);
                    }

                    // 天空図の中心と中心の間の距離 = 半径 + 半径 + 半径 * spaceXRatio
                    spaceX = (2.0 * hemisphereRadius + hemisphereRadius * spaceXRatio) * drawScale;
                }
            }

            return spaceX;
        }

        /// <summary>
        /// 天空図作成コマンド実行
        /// </summary>
        /// <param name="drawing"></param>
        /// <param name="drawPositionNumberText"></param>
        /// <param name="samplingPoint"></param>
        /// <param name="locationHeight">測定面の高さ</param>
        /// <param name="samplingPointNumber"></param>
        /// <param name="makeQuadratureTable"></param>
        /// <param name="makePositionTable"></param>
        /// <param name="skyMapTableLocateMode">表の配置位置モード</param>
        /// <param name="quadratureTableLocatePoint">三斜求積表の配置点座標(表の配置位置モード=任意)</param>
        /// <param name="positionTableLocatePoint">建物位置表の配置点座標(表の配置位置モード=任意)</param>
        /// <param name="quadratureTableShapes"></param>
        /// <param name="positionTableShapes"></param>
        /// <param name="optionShapes"></param>
        /// <param name="drawScale"></param>
        /// <param name="onlySkyRate">天空図描画フラグ</param>
        /// <param name="forRubberBand"></param>
        /// <returns></returns>
        private List<Shape> CreateShapesForSkyMapCommand(Drawing drawing, bool drawPositionNumberText, Point2d samplingPoint, double locationHeight, int samplingPointNumber, bool makeQuadratureTable, bool makePositionTable, SkyMapTableLocateMode skyMapTableLocateMode, Point2d quadratureTableLocatePoint, Point2d positionTableLocatePoint, out List<Shape> quadratureTableShapes, out List<Shape> positionTableShapes, out List<PositionNumberShape> optionShapes, double drawScale, bool onlySkyRate, bool forRubberBand = false)
        {
            // 図形リスト
            List<Shape> shapes = null;

            int err = 0;
            quadratureTableShapes = null;
            positionTableShapes = null;
            optionShapes = null;

            Document doc = drawing.Document;

            // 実行できる状態の図面か
            if(!CheckDocument(doc, true))
                return null;

            // 観測地点の緯度
            double latitudeDeg = 0; // (deg)

            // 日赤緯(冬至:-23.45°、春秋分:0°、夏至:23.45°、任意時期)
            double solarDeclinationDeg = 0.0;// 初期値:春秋分(deg)

            // 観測地点の緯度、日赤緯、測定面の高さ
            if(!GetShadowBaseDataSub(out latitudeDeg, out solarDeclinationDeg))
                return null;// エラーメッセージは表示済み

            // 太陽軌跡を表示するフラグ(三斜求積図を作成しないときだけ有効)
            bool drawSolarTrajectoryFlg = (triclinicTriangleCheckBox.Checked == false && solarTrajectoryCheckBox.Checked == true && !onlySkyRate) ? true : false;
            bool solarTrajectoryInterval10Minutes = false;// 10分間隔フラグ
            bool drawSeasonsSolarTrajectoryFlg = false;// 四季フラグ
            bool directCorrectionFlg = false;// 南補正フラグ
            // 太陽軌跡を表示する場合のフラグ値
            if(drawSolarTrajectoryFlg)
            {
                // 10分間隔フラグ
                if(solarTrajectoryInterval10MinutesCheckBox.Checked == true)
                    solarTrajectoryInterval10Minutes = true;

                // 四季フラグ
                if(seasonsCheckBox.Checked == true)
                    drawSeasonsSolarTrajectoryFlg = true;

                // 南補正（常に北を上に回転移動させて描画）
                if(directCorrectionCheckBox.Checked == true)
                    directCorrectionFlg = true;

                // ToDo 4半球平行射影図（等距離射影）今後検討、、、?
            }

            // 真北の角度を探す(太陽軌跡を表示する場合のみ)
            double dueNorthDeg = 0.0;// 真北の角度(deg)
            if(drawSolarTrajectoryFlg)
            {
                int result = DueNorthSearch(doc, out dueNorthDeg);
                if(result != 1)
                {
                    if(result == 0)
                        MessageBox.Show("真北が設定されていません。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    else if(result == -1)
                        MessageBox.Show("真北が図面中に複数あります。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);

                    return null;
                }
            }

            // 高さの文字のリスト取得
            var heightTextShapeList = new List<TextShape>();
            int count = GetWithAttributesTextShapeList(doc, attributesValueZ, heightTextShapeList);
            if(count == 0)
            {
                err = 2;
                goto ERRLABEL;
            }

            double maxHeightValue = 0.0;
            if(!ChkHeightTextShapeList(heightTextShapeList, out maxHeightValue))
            {
                err = 3;
                goto ERRLABEL;
            }

            // 実行できるレイアウトの図面かチェック
            if(!CheckLayout(doc, heightTextShapeList))
            {
                err = 4;
                goto ERRLABEL;
            }

            // 高さの「ベース線」(高さが書かれている建物の縁を表す線)リストを取得
            List<List<BaseLine>> baseLinesList = GetHeightBaseLineShapeList(doc, heightTextShapeList);
            if(baseLinesList.Count == 0)
            {
                err = 5;
                goto ERRLABEL;
            }

            // 三斜計算モード
            TriclinicMode triclinicMode = TriclinicMode.TriclinicMode_Standard;// 適合建物用
            if(planRadioButton.Checked)
                triclinicMode = TriclinicMode.TriclinicMode_Plan;// 計画建物用

            // 三斜求積図作成フラグ
            bool drawTriclinicTriangle = false;
            if(triclinicTriangleCheckBox.Checked && !onlySkyRate)
                drawTriclinicTriangle = true;

            // 天空図の全ての頂点に位置番号を付けるフラグ
            bool drawAllPointNumber = false;
            if(drawAllPointNumberCheckBox.Checked && !onlySkyRate)
                drawAllPointNumber = true;

            shapes = new List<Shape>();
            LayerTable layerTable = doc.LayerTable;

            // ベース線から端点の高さを持った線クラス(BaseLine3D)のリスト作成
            int linesCount = 0;
            List<BaseLine3D> baseLine3DList = new List<BaseLine3D>();
            List<BaseLine3D> overhangBaseLine3DList = new List<BaseLine3D>();
            foreach(List<BaseLine> baseLines in baseLinesList)
            {
                foreach(BaseLine baseLine in baseLines)
                {
                    // 高さ
                    // レイヤ―グループとレイヤ名にあるレベルを取得してその値も反映させる
                    int drawingID = baseLine.line.Drawing.ID;
                    int layerID = baseLine.line.LayerID;
                    Layer layer = layerTable.GetLayerByID(layerID);
                    Layer parentlayer = layer.ParentLayer;

                    string layerName = layer.Name;
                    string layerGroupName = parentlayer.Name;// ルートの場合は空 parentlayerID == -1

                    double heightValue1 = 0.0;
                    double bottomValue1 = 0.0;
                    double heightValue2 = 0.0;
                    double bottomValue2 = 0.0;

                    try
                    {
                        // 高さを表す文字から高さの値を取得
                        string heightText1 = baseLine.startPointSideHeightText.Text;
                        if(!GetHeightValue(heightText1, layerGroupName, layerName, locationHeight, out heightValue1, out bottomValue1))
                        {
                            err = 6;
                            goto ERRLABEL;
                        }
                    }
                    catch
                    {
                        err = 6;
                        goto ERRLABEL;
                    }

                    try
                    {
                        // 高さを表す文字から高さの値を取得
                        string heightText2 = baseLine.endPointSideHeightText.Text;
                        if(!GetHeightValue(heightText2, layerGroupName, layerName, locationHeight, out heightValue2, out bottomValue2))
                        {
                            err = 6;
                            goto ERRLABEL;
                        }
                    }
                    catch
                    {
                        err = 6;
                        goto ERRLABEL;
                    }

                    // 端点の高さを持った線クラス(BaseLine3D)作成
                    linesCount++;
                    int baseLineID = linesCount;// baseLineID：1～になる
                    BaseLine3D baseLine3D = new BaseLine3D(baseLineID, baseLine.line, heightValue1, heightValue2);

                    // リストに追加
                    baseLine3DList.Add(baseLine3D);

                    // オーバーハングの線について
                    // "D:\Temp\天空図アドインテスト(オーバーハング).rpcd"
                    if(bottomValue1 > 0.0 || bottomValue2 > 0.0)
                    {
                        BaseLine3D overhangBaseLine3D = new BaseLine3D(baseLineID, baseLine.line, bottomValue1, bottomValue2);
                        overhangBaseLine3DList.Add(overhangBaseLine3D);
                    }
                }
            }

            // baseLine3DListの接続性(端点が同じものを繋いだID情報)更新
            UpdateConnectList(baseLine3DList);

            Point2d centerPoint = samplingPoint;// 天空図の中心点
            double hemisphereRadius = (double)GetHemisphereRadius(forRubberBand);// 天空図半径(整数で受け取る)
            if(hemisphereRadius < 0.0)
            {
                err = 8;
                goto ERRLABEL;
            }

            // 正射影の線の配列作成
            // 建物のトップ部分（屋根の縁）
            List<OrthographicProjectionShape> orthographicProjectionShapeList = ConvertOrthographicProjectionShapeList(centerPoint, hemisphereRadius, baseLine3DList);
            // オーバーハング対応 
            List<OrthographicProjectionShape> overhangOrthographicProjectionShapeList = ConvertOrthographicProjectionShapeList(centerPoint, hemisphereRadius, overhangBaseLine3DList);

            // 建物の角部分の線
            List<LinePoints> cornerLines = CreateCornerLines(orthographicProjectionShapeList, overhangOrthographicProjectionShapeList, centerPoint, hemisphereRadius);

            // 書込みレイヤ
            Layer currentLayer = layerTable.CurrentLayer;

            // 天空図の情報表示文字の大きさ
            // 文字サイズ調整比
            double textSizeRatio = GetSkymapTextSizeRatio();// 文字大きさ調整値
            double textHight = hemisphereRadius * customSettings.skymapDefaultTextSizeRatio * textSizeRatio; // 表中などの文字の大きさ、天空図円の大きさのx%を元にしてサイズ調整
            double textHight2 = textHight * 0.5;// 天空図中の文字の大きさ
            double positionNumberTextHight = textHight * drawScale;// 配置図の位置を示す番号文字の大きさ

            // 目盛
            string scaleText = "";
            if(scaleComboBox.Text != "" && !onlySkyRate)
            {
                scaleText = scaleComboBox.SelectedItem.ToString();
                double scaleAngleDeg = 90.0;// 目盛の角度(初期値)
                try
                {
                    // 文字列から十進数角度に変換
                    scaleAngleDeg = app.NumericTextConverter.DmsTextToDegree(scaleText);
                }
                catch(Exception)
                {
                    err = 1;
                    goto ERRLABEL;
                }

                List<Shape> measureShapes = CreatMemeasureShapeList(centerPoint, hemisphereRadius, dueNorthDeg, scaleAngleDeg, doc, currentLayer);
                if(measureShapes.Count > 0)
                {
                    // 方向補正する場合
                    if(directCorrectionFlg)
                        RotateAndMirrorShapes(measureShapes, centerPoint, -dueNorthDeg, 90.0);// 回転＆鏡像移動

                    shapes.AddRange(measureShapes);
                }
            }

            // 太陽軌跡を描画する場合
            if(drawSolarTrajectoryFlg == true)
            {
                // 東西南北表記
                List<Shape> ewsnShapes = CreatEWSNShapeList(centerPoint, hemisphereRadius, dueNorthDeg, doc, currentLayer);
                if(ewsnShapes.Count > 0)
                {
                    // 方向補正する場合
                    if(directCorrectionFlg)
                        RotateAndMirrorShapes(ewsnShapes, centerPoint, -dueNorthDeg, 90.0);// 回転＆鏡像移動

                    shapes.AddRange(ewsnShapes);
                }

                // 太陽軌跡の図形作成
                List<Shape> solarSeasonNameShape = new List<Shape>();// 冬至、春秋分、夏至太陽の文字図形
                List<Shape> solarTrajectoryShapes = CreatsolarTrajectoryShapeList(centerPoint, hemisphereRadius, dueNorthDeg, latitudeDeg, solarTrajectoryInterval10Minutes, drawSeasonsSolarTrajectoryFlg, textHight, solarSeasonNameShape, doc, currentLayer);
                if(solarTrajectoryShapes.Count > 0)
                {
                    // 方向補正する場合
                    if(directCorrectionFlg)
                        RotateAndMirrorShapes(solarTrajectoryShapes, centerPoint, -dueNorthDeg, 90.0);// 回転＆鏡像移動

                    shapes.AddRange(solarTrajectoryShapes);
                }

                if(solarSeasonNameShape.Count > 0)
                {
                    // 方角に合わせて回転(「四季の太陽軌跡」文字図形リスト)
                    if(directCorrectionFlg)
                    {
                        foreach(Shape shape in solarSeasonNameShape)
                            shape.Rotate(centerPoint, -dueNorthDeg);
                    }

                    shapes.AddRange(solarSeasonNameShape);
                }


            }

            // 測定点が建物の内側フラグ
            bool isCenterPointInside = IsPointInBuilding(centerPoint, hemisphereRadius, baseLine3DList, orthographicProjectionShapeList);
            bool isIinsideTopRoof = topRoofCheckBox.Checked; // 建物内部の上が屋根かどうか

            // 建物部分のハッチング
            if(hatchComboBox.Text != "" && !onlySkyRate)
            {
                string hatchAngleText = hatchComboBox.SelectedItem.ToString();
                double hatchAngleDeg = 2.0;// ハッチングの線の間隔角度
                try
                {
                    // 文字列から十進数角度に変換
                    hatchAngleDeg = app.NumericTextConverter.DmsTextToDegree(hatchAngleText);
                }
                catch(Exception)
                {
                    err = 1;
                    goto ERRLABEL;
                }

                List<Shape> hatchShapes = CreateHatchShapes(centerPoint, hemisphereRadius, hatchAngleDeg, baseLine3DList, orthographicProjectionShapeList, overhangOrthographicProjectionShapeList, isCenterPointInside, isIinsideTopRoof, doc, currentLayer);
                // ハッチングの線を配列の最初に
                if(hatchShapes.Count > 0)
                {
                    // 方向補正する場合
                    if(directCorrectionFlg)
                        RotateAndMirrorShapes(hatchShapes, centerPoint, -dueNorthDeg, 90.0);// 回転＆鏡像移動

                    shapes.InsertRange(0, hatchShapes);
                }
            }

            // 最大分割角度
            string maxSeparateAngleText = maxSeparateAngleComboBox.Text;
            // 角度文字列から十進数角度に変換
            double maxSeparateAngleDeg = app.NumericTextConverter.DmsTextToDegree(maxSeparateAngleText);

            // 天空率計算(選択してる求積タイプ（計画建物or適合建物)に合わせて三斜求積の計算する)
            // 三斜求積図関係
            List<TriclinicTriangle> triclinicTriangleList = null;
            TriclinicSkyRateAttribute triclinicSkyRateAttribute = null;
            List<PositionNumberShape> positionShapes = null;// 位置を表す文字作成(位置に点図形も)
            List<PositionNumberShape> positionShapes2 = null;
            if(!(isCenterPointInside && isIinsideTopRoof))// 測定点が内部で且つ上が屋根ではないときに計算
            {
                // 三斜三角形の元となる三角形のリストを作成
                List<BaseTriangleInfo> triangleInfoList = CreateBaseTriangleInfoList(orthographicProjectionShapeList, centerPoint, hemisphereRadius);

                // 三斜三角形の元となる三角形の図形作成(チェック用)
                if(RC_BaseTriangleCheck == true && !onlySkyRate)
                {
                    List<Shape> triangleInfoListShape = CreateTriangleInfoListShapeList(triangleInfoList, hemisphereRadius, isCenterPointInside, doc, currentLayer);
                    if(triangleInfoListShape.Count > 0)
                    {
                        shapes.AddRange(triangleInfoListShape);
                    }
                }

                // 三斜求積図(三斜三角形のリスト)作成
                triclinicTriangleList = CreateTriclinicTriangleList(triclinicMode, triangleInfoList, orthographicProjectionShapeList, maxSeparateAngleDeg, centerPoint, hemisphereRadius);

                // 天空率属性値取得
                triclinicSkyRateAttribute = GetSkyRate(triclinicMode, hemisphereRadius, triclinicTriangleList);

                // 三斜求積図を描画
                if(drawTriclinicTriangle && !onlySkyRate)
                {
                    Debug.Assert(directCorrectionFlg == false);

                    // 三斜求積図(三斜三角形)描画用図形の作成
                    List<Shape> triclinicTriangleShapes = CreateTriclinicTriangleShapeList(triclinicTriangleList, triclinicMode, hemisphereRadius, isCenterPointInside, doc, currentLayer, textHight2);
                    if(triclinicTriangleShapes.Count > 0)
                    {
                        shapes.AddRange(triclinicTriangleShapes);
                    }

                    // 位置情報
                    // 位置番号を生成
                    UpdateBaseLine3DPositionNumber(baseLine3DList, overhangBaseLine3DList);

                    if(optionShapes == null)
                        optionShapes = new List<PositionNumberShape>();

                    // 位置を表す文字作成(位置に点図形も)
                    positionShapes = CreateSkymapPositionTexts(baseLine3DList, orthographicProjectionShapeList, triangleInfoList, triclinicMode, drawPositionNumberText, drawAllPointNumber, optionShapes, doc, currentLayer, positionNumberTextHight, textHight2);
                    if(positionShapes.Count > 0)
                    {
                        //shapes.AddRange(positionShapes);// 後で追加
                    }

                    // オーバーハング対応
                    if(orthographicProjectionShapeList.Count > 0)
                    {
                        positionShapes2 = CreateSkymapPositionTexts(overhangBaseLine3DList, overhangOrthographicProjectionShapeList, triangleInfoList, triclinicMode, false, drawAllPointNumber, optionShapes, doc, currentLayer, positionNumberTextHight, textHight2);
                        if(positionShapes2.Count > 0)
                        {
                            //shapes.AddRange(positionShapes2);// 後で追加
                        }
                    }
                }
            }

            if(!onlySkyRate)
            {
                // 正射影の図形を作成
                List<Shape> orthographicProjectionShapes = CreateOrthographicProjectionShapeList(orthographicProjectionShapeList, overhangOrthographicProjectionShapeList, cornerLines, doc, currentLayer);
                if(orthographicProjectionShapes.Count > 0)
                {
                    // 方向補正する場合
                    if(directCorrectionFlg)
                        RotateAndMirrorShapes(orthographicProjectionShapes, centerPoint, -dueNorthDeg, 90.0);// 回転＆鏡像移動

                    shapes.AddRange(orthographicProjectionShapes);
                }

                // 天空図の円
                List<Shape> skymapCircleShapes = CreateSkymapCircleShapeList(centerPoint, hemisphereRadius, doc, currentLayer);
                if(skymapCircleShapes.Count > 0)
                {
                    shapes.AddRange(skymapCircleShapes);
                }

                // 中心に点
                double pointMarkerSize = textHight * addInSettings.positionPointMarkerSizeRatio;
                AddPointMarkerShape(shapes, centerPoint, customSettings.positionPointMarkerType, pointMarkerSize, customSettings.samplingPointMarkerColorNumber, doc, currentLayer);
            }

            // 天空率等の情報文字、表の作成 //
            // 天空率等の基本情報図形作成
            BndBox infoShapesBndBox;
            List<Shape> infoTextBox1Shapes = CreateSkyRateBasicInfoShapes(samplingPointNumber, centerPoint, hemisphereRadius, textHight, scaleText, latitudeDeg, locationHeight, triclinicMode, triclinicSkyRateAttribute, drawSolarTrajectoryFlg, onlySkyRate, doc, currentLayer, out infoShapesBndBox);
            if(infoTextBox1Shapes.Count > 0)
            {
                shapes.AddRange(infoTextBox1Shapes);
            }

            // 三斜求積表と建物位置表
            if((makeQuadratureTable || makePositionTable)  && !onlySkyRate)
            {
                // 測定点が建物の内側で建物内部の上が屋根のときは計算しない
                if(isCenterPointInside && isIinsideTopRoof)
                {
                    err = 7;
                    goto ERRLABEL;
                }

                double offset = textHight * 3;
                Point2d[] infoShapesBndBoxPoints = infoShapesBndBox.GetBoundingBox();

                // 配置座標(とりあえず天空図の下)
                Point2d quadratureTableLocatePoint2 = new Point2d(infoShapesBndBoxPoints[0].X + infoShapesBndBox.GetWidth() * 0.5, infoShapesBndBoxPoints[0].Y - offset);
                if(skyMapTableLocateMode == SkyMapTableLocateMode.SkyMapTableLocateMode_Free)// 任意の場合は指定した座標
                    quadratureTableLocatePoint2 = quadratureTableLocatePoint;

                // 建物位置表の配置点（仮）
                Point2d positionTableLocatePoint2 = new Point2d(quadratureTableLocatePoint2.X, quadratureTableLocatePoint2.Y);

                // 三斜求積表の図形作成(天空図の右か下、または任意の場所)
                if(makeQuadratureTable && triclinicTriangleList != null && triclinicSkyRateAttribute != null)
                {
                    quadratureTableShapes = CreateQuadratureTableShapes(scaleText, locationHeight, centerPoint, hemisphereRadius, maxSeparateAngleText, triclinicTriangleList, triclinicSkyRateAttribute, skyMapTableLocateMode, quadratureTableLocatePoint2, textHight, forRubberBand, doc, currentLayer);
                    if(quadratureTableShapes != null && quadratureTableShapes.Count > 0)
                    {
                        // 建物位置表の配置点（仮）を移動（表の配置が下・下の場合）
                        if(skyMapTableLocateMode == SkyMapTableLocateMode.SkyMapTableLocateMode_LowerLower)
                        {
                            // 全体の範囲
                            BndBox quadratureTableBndBox = GetShapesBndBox(quadratureTableShapes);

                            Debug.Assert(quadratureTableBndBox != null);
                            positionTableLocatePoint2.Y -= quadratureTableBndBox.GetHeight() + offset;
                        }
                    }
                }

                // 位置表データアイテムリスト作成取得
                List<PositionDataItem> positionDataItemList = null;
                if(makePositionTable || optionShapes != null)
                    positionDataItemList = CreatePositionDataItemList(dueNorthDeg, samplingPoint, hemisphereRadius, baseLine3DList, overhangBaseLine3DList);

                // 建物位置表の図形作成(天空図の直ぐ下か天空図の下に三斜求積表があってその下、または任意の場所)
                if(makePositionTable)
                {
                    // 配置点
                    if(skyMapTableLocateMode == SkyMapTableLocateMode.SkyMapTableLocateMode_Free)// 任意の場合は指定した座標
                        positionTableLocatePoint2 = positionTableLocatePoint;

                    // 表の作成
                    positionTableShapes = CreatePositionTableShapes(scaleText, positionDataItemList, locationHeight, hemisphereRadius, triclinicMode,
                                                                skyMapTableLocateMode, positionTableLocatePoint2, textHight, forRubberBand, doc, currentLayer);

                    // 三斜求積図中の位置の文字で表にないものを除いて作図
                    if(positionShapes != null)
                    {
                        RemoveNonNumberPositionNumberShapes(positionDataItemList, positionShapes);
                        foreach(PositionNumberShape positionShape in positionShapes)
                            shapes.AddRange(positionShape.Shapes);
                    }

                    if(positionShapes2 != null)
                    {
                        RemoveNonNumberPositionNumberShapes(positionDataItemList, positionShapes2);
                        foreach(PositionNumberShape positionShape in positionShapes2)
                            shapes.AddRange(positionShape.Shapes);
                    }
                }

                // optionShapesの文字で表にないものを除く
                if(optionShapes != null && positionDataItemList != null)
                    RemoveNonNumberPositionNumberShapes(positionDataItemList, optionShapes);
            }

            return shapes;


        // エラーがあった場合のメッセージ表示
        ERRLABEL:
            if(err != 0)
            {
                string strErr = "";
                if(err == 1)
                    strErr = "";
                else if(err == 2)
                    strErr = "高さの設定された線がありません。";
                else if(err == 3)
                    strErr = "高さのデータまたは測定面の高さが正しくありません・";
                else if(err == 4)
                    strErr = "実行できるレイアウトの図面ではありません。";
                else if(err == 5)
                    strErr = "高さの「ベース線」取得でエラーになりました。";
                else if(err == 6)
                    strErr = "測定面より低い高さの線があるか、レイヤ名、レイヤーグループ名でレベルの高さが適切でなないためエラーになりました。";
                else if(err == 7)
                    strErr = "測定点が建物内部で上が屋根の場合計算しません。";
                else if(err == 8)
                    strErr = "天空図の半径の値が正しくありません。";

                if(strErr != "")
                    MessageBox.Show(strErr, AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);

                return null;
            }

            return null;
        }

        /// <summary>
        /// 指定した配置位置図形リストから位置表データアイテムリストにない番号のものを削除する
        /// </summary>
        /// <param name="positionDataItemList"></param>
        /// <param name="positionShapes"></param>
        private void RemoveNonNumberPositionNumberShapes(List<PositionDataItem> positionDataItemList, List<PositionNumberShape> positionShapes)
        {
            int shapesSize = positionShapes.Count;
            for(int i = shapesSize - 1; i >= 0; i--)
            {
                PositionNumberShape positionNumberShape = positionShapes[i];
                if(!IsPositionNumber(positionDataItemList, positionNumberShape.Number))
                    positionShapes.RemoveAt(i);
            }
        }


        /// <summary>
        /// 天空図の半径取得
        /// </summary>
        /// <param name="isRubberBand"></param>
        /// <returns></returns>
        private int GetHemisphereRadius(bool isRubberBand = false)
        {
            int hemisphereRadius = -1;
            int errDefault = 100;

            try
            {
                // ユーザーが編集中のテキストを強制的に反映させる
                hemisphereRadiusUpDown.Validate();

                // Text プロパティから文字列を取得して int に変換
                hemisphereRadius = int.Parse(hemisphereRadiusUpDown.Text);

                if(hemisphereRadius < hemisphereRadiusUpDown.Minimum || hemisphereRadius > hemisphereRadiusUpDown.Maximum)
                {
                    if(isRubberBand)
                        hemisphereRadius = errDefault;
                    else
                        hemisphereRadius = -1;
                }

            }
            catch
            {
                // 例外が発生した場合
                // Debug.WriteLine($"エラー: {ex.Message}");
                if(isRubberBand)
                    hemisphereRadius = errDefault;
                else
                    hemisphereRadius = -1;
            }

            return hemisphereRadius;
        }

        /// <summary>
        /// 位置表データアイテムリストに指定した番号のものがあるか
        /// </summary>
        /// <param name="positionDataItemList"></param>
        /// <param name="number"></param>
        /// <returns></returns>
        private bool IsPositionNumber(List<PositionDataItem> positionDataItemList, int number)
        {
            foreach(PositionDataItem item in positionDataItemList)
            {
                if(item.Number == number)
                    return true;
            }

            return false;
        }

        /// <summary>
        /// 点列に同じ座標の点があるか
        /// </summary>
        /// <param name="points"></param>
        /// <param name="point"></param>
        /// <returns></returns>
        private bool IsSamePointInPoints(List<Point2d> points, Point2d point)
        {
            foreach(Point2d pointInPoints in points)
            {
                if(point.Equals(pointInPoints))
                    return true;
            }

            return false;
        }

        /// <summary>
        /// 天空図の文字大きさ調整値取得
        /// </summary>
        /// <returns></returns>
        private double GetSkymapTextSizeRatio()
        {
            double textSizeRatio = textSizeTrackBar.Value;// textSizeTrackBar.Value : 0 ～　200 中間は100 　倍率を50%～200% に
            if(textSizeRatio <= 100)
                textSizeRatio = (50 + (50 * textSizeRatio * 0.01)) * 0.01;
            else
                textSizeRatio = (100 + (textSizeRatio - 100)) * 0.01;

            return textSizeRatio;
        }

        /// <summary>
        /// ベース線から正射影線に変換
        /// </summary>
        /// <param name="centerPoint"></param>
        /// <param name="hemisphereRadius"></param>
        /// <param name="baseLine3DList"></param>
        /// <returns></returns>
        private List<OrthographicProjectionShape> ConvertOrthographicProjectionShapeList(Point2d centerPoint, double hemisphereRadius, List<BaseLine3D> baseLine3DList)
        {
            List<OrthographicProjectionShape> orthographicProjectionShapeList = new List<OrthographicProjectionShape>();

            // ベース線から正射影の線を作る
            foreach(BaseLine3D baseLine in baseLine3DList)
            {
                List<Point2d> orthographicProjectionPoints = new List<Point2d>();

                int baseLineID = baseLine.baseLineID;
                Point2d lineStartPoint = baseLine.line.StartPoint;
                Point2d lineEndPoint = baseLine.line.EndPoint;
                double baseLineStartPointHeight = baseLine.startPointHeight;
                double baseLineEndPointHeight = baseLine.endPointHeight;

                // 測定点Cを中心とした角度
                double angleDeg = app.Geometry.GetAngle(centerPoint, lineStartPoint, lineEndPoint);
                Debug.Assert(angleDeg <= 360.0);
                if(angleDeg > 180.0)
                    angleDeg = 360.0 - angleDeg;

                // n度以内に分けたときの分割数(JCBA方式基準で「天空率の三射求積の分割角度は最大で10度」とあるため)
                double separateAngleDeg = customSettings.orthographicProjectionSeparateAngleDeg;// この角度で刻む(小さくすると処理が遅くなる。5度ぐらいがいいか。)
                int separateCount = (int)angleDeg / (int)separateAngleDeg;// (int):doubleの小数点以下は切り捨て
                if(angleDeg % separateAngleDeg > 0.0 || separateCount == 0)
                    separateCount++;

                // 線の始点から終点に向かう角度
                double lineStartToEndAngleDeg = app.Geometry.GetAngle(lineStartPoint, lineEndPoint);

                // 線の長さ
                double lineLength = app.Geometry.GetDistance(lineStartPoint, lineEndPoint);

                for(int i = 0; i <= separateCount; i++)
                {
                    // 分割したベース線上の点
                    Point2d pointOnBaseLine = new Point2d(lineStartPoint.X + (lineLength / (double)separateCount * (double)i), lineStartPoint.Y);
                    pointOnBaseLine = app.Geometry.RotatePoint(pointOnBaseLine, lineStartPoint, lineStartToEndAngleDeg);// 回転移動

                    // 高さ(始点側と終点側で高さが違う(斜めの)場合も考える)
                    double pointOnBaseLineHeight = baseLineStartPointHeight + ((baseLineEndPointHeight - baseLineStartPointHeight) / (double)separateCount * (double)i);

                    // 正射影(せいしゃえい:Orthographic projection)の点の位置を求める
                    // 天空図の中心点から正射影の点までの距離と角度
                    if(!GetOrthographicProjectionDistanceAndAngle(centerPoint, hemisphereRadius, pointOnBaseLine, pointOnBaseLineHeight, out double centerToOrthographicProjectionDistance, out double centerToPointAngleDeg))
                        continue;

                    // 正射影の座標
                    Point2d orthographicProjectionPoint = new Point2d(centerPoint.X + centerToOrthographicProjectionDistance, centerPoint.Y);
                    orthographicProjectionPoint = app.Geometry.RotatePoint(orthographicProjectionPoint, centerPoint, centerToPointAngleDeg);// 回転移動

                    // 点列追加
                    orthographicProjectionPoints.Add(orthographicProjectionPoint);
                }

                // 正射影の線の図形作成とリストに追加
                // 点列コピー
                Point2d[] polylinePoints = new Point2d[orthographicProjectionPoints.Count];
                for(int i = 0; i < orthographicProjectionPoints.Count; i++)
                    polylinePoints[i] = orthographicProjectionPoints[i];

                PolylineShape polylineShape = app.ShapeFactory.CreatePolyline(polylinePoints);

                // 正射影の線の配列として取っておく
                OrthographicProjectionShape orthographicProjectionShape = new OrthographicProjectionShape(baseLineID, polylineShape);
                orthographicProjectionShapeList.Add(orthographicProjectionShape);
            }

            return orthographicProjectionShapeList;
        }

        /// <summary>
        /// 天空図の中心点から正射影の点までの距離と角度を計算取得
        /// </summary>
        /// <param name="centerPoint"></param>
        /// <param name="hemisphereRadius"></param>
        /// <param name="pointOnBaseLine"></param>
        /// <param name="pointOnBaseLineHeight"></param>
        /// <param name="centerToOrthographicProjectionDistance"></param>
        /// <param name="centerToPointAngleDeg"></param>
        /// <returns></returns>
        private bool GetOrthographicProjectionDistanceAndAngle(Point2d centerPoint, double hemisphereRadius, Point2d pointOnBaseLine, double pointOnBaseLineHeight,
                                                                    out double centerToOrthographicProjectionDistance, out double centerToPointAngleDeg)
        {
            centerToOrthographicProjectionDistance = 0.0;
            centerToPointAngleDeg = 0.0;

            // 天空図の中心からベース上の点までの距離
            double centerToPointDistance = app.Geometry.GetDistance(centerPoint, pointOnBaseLine);

            // 天空図の中心からベース上の高さ分上の点までの距離
            double centerToPointTopDistance = Math.Sqrt(pointOnBaseLineHeight * pointOnBaseLineHeight + centerToPointDistance * centerToPointDistance);
            //Debug.Assert(centerToPointTopDistance > RcPrecisionConfusion); // 中心点の真上で高さも0の場合は0
            if(!(centerToPointTopDistance > RcPrecisionConfusion))
                return false;

            // 天空図の中心点から正射影の点までの距離
            centerToOrthographicProjectionDistance = (hemisphereRadius / centerToPointTopDistance) * centerToPointDistance;

            // 天空図の中心からベース上の点の角度
            centerToPointAngleDeg = app.Geometry.GetAngle(centerPoint, pointOnBaseLine);

            return true;
        }

        /// <summary>
        /// 建物の角の線リスト作成
        /// </summary>
        /// <param name="orthographicProjectionShapeList"></param>
        /// <param name="overhangOrthographicProjectionShapeList"></param>
        /// <param name="centerPoint"></param>
        /// <param name="hemisphereRadius"></param>
        /// <returns></returns>
        private List<LinePoints> CreateCornerLines(List<OrthographicProjectionShape> orthographicProjectionShapeList,
                               List<OrthographicProjectionShape> overhangOrthographicProjectionShapeList, Point2d centerPoint, double hemisphereRadius)
        {
            List<LinePoints> cornerLines = new List<LinePoints>();

            // 建物のトップ（屋根の縁）の角部分の点列
            List<Point2d> topCornerPoints = new List<Point2d>();

            // オーバーハングがある場合は、まずその角から優先して線を作成する
            foreach(OrthographicProjectionShape overhangOrthographicProjectionShape in overhangOrthographicProjectionShapeList)
            {
                // 底の正射影の線
                PolylineShape polylineShape1 = overhangOrthographicProjectionShape.polylineShape;
                int pointsLength = polylineShape1.Points.Length;
                if(pointsLength < 1)
                    continue;

                Point2d stratPoint1 = polylineShape1.Points.First();

                int baseLineID = overhangOrthographicProjectionShape.baseLineID;

                // 上の正射影の線
                // 正射影の線のID == 正射影の線の元になった建物の線のインデックス
                OrthographicProjectionShape orthographicProjectionShape = orthographicProjectionShapeList.FirstOrDefault(o => o.baseLineID == baseLineID);
                // 正射影の線連続線図形
                if(orthographicProjectionShape != null)
                {
                    PolylineShape polylineShape2 = orthographicProjectionShape.polylineShape;
                    Point2d cornerPoint = polylineShape2.Points.First();
                    if(!IsSamePointInPoints(topCornerPoints, cornerPoint))
                    {
                        topCornerPoints.Add(cornerPoint);// この点を記録して、今後同じところに線は付かない

                        if(!cornerPoint.Equals(stratPoint1))
                        {
                            LinePoints cornerLine = new LinePoints(cornerPoint, stratPoint1);
                            cornerLines.Add(cornerLine);
                        }
                    }
                }
            }

            foreach(OrthographicProjectionShape orthographicProjectionShape in orthographicProjectionShapeList)
            {
                PolylineShape polylineShape = orthographicProjectionShape.polylineShape;
                int pointsLength = polylineShape.Points.Length;
                if(pointsLength < 1)
                    continue;

                Point2d cornerPoint = polylineShape.Points.First();

                // 天空図の中心からベース上の点の角度
                double centerToPointAngleDeg = app.Geometry.GetAngle(centerPoint, cornerPoint);

                // 角の下から上（屋根位置）までを表す線をリストに追加
                // 同じ座標の点がなければ追加
                if(!IsSamePointInPoints(topCornerPoints, cornerPoint))
                {
                    topCornerPoints.Add(cornerPoint);

                    // 天空図の円の円周上の点
                    Point2d pointOnCircle = new Point2d(centerPoint.X + hemisphereRadius, centerPoint.Y);
                    pointOnCircle = app.Geometry.RotatePoint(pointOnCircle, centerPoint, centerToPointAngleDeg);// 回転移動

                    if(!cornerPoint.Equals(pointOnCircle))
                    {
                        LinePoints cornerLine = new LinePoints(cornerPoint, pointOnCircle);
                        cornerLines.Add(cornerLine);
                    }
                }
            }

            return cornerLines;
        }

        /// <summary>
        ///  baseLine3DListの接続性(端点が同じものを繋いだID情報)更新
        /// </summary>
        /// <param name="baseLine3DList"></param>
        private void UpdateConnectList(List<BaseLine3D> baseLine3DList)
        {
            // 他の線と繋がっているか
            int listCount = baseLine3DList.Count;
            for(int i = 0; i < listCount; i++)
            {
                BaseLine3D baseLine1 = baseLine3DList[i];

                for(int j = listCount - 1; j >= 0; j--)
                {
                    if(i == j)
                        continue;

                    BaseLine3D baseLine2 = baseLine3DList[j];

                    // 繋がる場合
                    if(IsConnectedBaseLine3D(baseLine1, baseLine2))
                    {
                        int baseLineID1 = baseLine1.baseLineID;
                        int baseLineID2 = baseLine2.baseLineID;

                        // baseLine1の繋がる全ての線にbaseLineID2を追加
                        baseLine1.AddConnectID(baseLineID2);
                        foreach(int id in baseLine1.connectLineIDList)
                        {
                            BaseLine3D baseLine3 = baseLine3DList.FirstOrDefault(o => o.baseLineID == id);
                            if(baseLine3 == null)
                                continue;

                            baseLine3.AddConnectID(baseLineID2);
                        }

                        // baseLine2の繋がる全ての線にbaseLineID1を追加
                        baseLine2.AddConnectID(baseLineID1);
                        foreach(int id in baseLine2.connectLineIDList)
                        {
                            BaseLine3D baseLine3 = baseLine3DList.FirstOrDefault(o => o.baseLineID == id);
                            if(baseLine3 == null)
                                continue;

                            baseLine3.AddConnectID(baseLineID1);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 2つのBaseLine3Dが繋がっているか
        /// </summary>
        /// <param name="baseLine1"></param>
        /// <param name="baseLine2"></param>
        /// <returns></returns>
        public bool IsConnectedBaseLine3D(BaseLine3D baseLine1, BaseLine3D baseLine2)
        {
            int baseLineID1 = baseLine1.baseLineID;
            Point2d lineStartPoint1 = baseLine1.line.StartPoint;
            Point2d lineEndPoint1 = baseLine1.line.EndPoint;
            double baseLineStartPointHeight1 = baseLine1.startPointHeight;
            double baseLineEndPointHeight1 = baseLine1.endPointHeight;

            int baseLineID2 = baseLine2.baseLineID;
            Point2d lineStartPoint2 = baseLine2.line.StartPoint;
            Point2d lineEndPoint2 = baseLine2.line.EndPoint;
            double baseLineStartPointHeight2 = baseLine2.startPointHeight;
            double baseLineEndPointHeight2 = baseLine2.endPointHeight;

            if((IsPointEquals(lineStartPoint1, lineStartPoint2) && Math.Abs(baseLineStartPointHeight1 - baseLineStartPointHeight2) <= RcPrecisionConfusion)
                || (IsPointEquals(lineStartPoint1, lineEndPoint2) && Math.Abs(baseLineStartPointHeight1 - baseLineEndPointHeight2) <= RcPrecisionConfusion)
                || (IsPointEquals(lineEndPoint1, lineStartPoint2) && Math.Abs(baseLineEndPointHeight1 - baseLineStartPointHeight2) <= RcPrecisionConfusion)
                || (IsPointEquals(lineEndPoint1, lineEndPoint2) && Math.Abs(baseLineEndPointHeight1 - baseLineEndPointHeight2) <= RcPrecisionConfusion))
            {
                return true;
            }

            return false;
        }

        /// <summary>
        /// 目盛の図形作成
        /// </summary>
        /// <param name="centerPoint"></param>
        /// <param name="hemisphereRadius"></param>
        /// <param name="dueNorthDeg"></param>
        /// <param name="scaleAngleDeg"></param>
        /// <param name="doc"></param>
        /// <param name="currentLayer"></param>
        /// <returns></returns>
        private List<Shape> CreatMemeasureShapeList(Point2d centerPoint, double hemisphereRadius, double dueNorthDeg, double scaleAngleDeg, Document doc, Layer currentLayer)
        {
            List<Shape> measureShapes = new List<Shape>();

            Debug.Assert(hemisphereRadius > 0.0);
            Debug.Assert(dueNorthDeg >= -180.0 && dueNorthDeg <= 180.0);
            Debug.Assert(scaleAngleDeg >= 1.0 && dueNorthDeg <= 90.0);

            // 緯度線
            int circleCount = (int)(90.0 / scaleAngleDeg);
            for(int i = 1; i < circleCount; i++)
            {
                double hAngle = scaleAngleDeg * (double)i;
                double circleRadius = hemisphereRadius * Math.Cos(ToRadian(hAngle));

                CircleShape circleShape = app.ShapeFactory.CreateCircle(centerPoint, circleRadius);

                circleShape.ColorNumber = customSettings.memeasureColorNumber;// 色

                if(i % 2 == 0)
                    circleShape.LinetypeNumber = customSettings.memeasureLineTypeNumber1;// 線種
                else
                    circleShape.LinetypeNumber = customSettings.memeasureLineTypeNumber2;// 線種

                circleShape.LinewidthNumber = customSettings.memeasureLinewidthNumber;// 線幅

                // レイヤのセット
                circleShape.Layer = currentLayer;

                // 図形追加
                measureShapes.Add(circleShape);
            }

            // 経度線
            // 中心から放射状に(一定角度間隔で)円周上までの線を作る
            int lineCount = (int)(360.0 / scaleAngleDeg);
            for(int i = 0; i < lineCount; i++)
            {
                // 測定点（中心点）を通る線の天空図円の交点
                Point2d pointOnCircle = new Point2d(centerPoint.X + hemisphereRadius, centerPoint.Y);
                pointOnCircle = app.Geometry.RotatePoint(pointOnCircle, centerPoint, scaleAngleDeg * (double)i + dueNorthDeg + 90.0);// 回転移動(真北からスタート)

                LineShape lineShape = app.ShapeFactory.CreateLine(centerPoint, pointOnCircle);

                lineShape.ColorNumber = customSettings.memeasureColorNumber;// 色

                if(i % 2 == 0)
                    lineShape.LinetypeNumber = customSettings.memeasureLineTypeNumber1;// 線種
                else
                    lineShape.LinetypeNumber = customSettings.memeasureLineTypeNumber2;// 線種

                lineShape.LinewidthNumber = customSettings.memeasureLinewidthNumber;// 線幅

                // レイヤのセット
                lineShape.Layer = currentLayer;

                // 図形追加
                measureShapes.Add(lineShape);
            }

            return measureShapes;
        }

        /// <summary>
        /// 東西南北方向表記
        /// </summary>
        /// <param name="centerPoint"></param>
        /// <param name="hemisphereRadius"></param>
        /// <param name="dueNorthDeg"></param>
        /// <param name="doc"></param>
        /// <param name="currentLayer"></param>
        /// <returns></returns>
        private List<Shape> CreatEWSNShapeList(Point2d centerPoint, double hemisphereRadius, double dueNorthDeg, Document doc, Layer currentLayer)
        {
            List<Shape> shapes = new List<Shape>();

            // 方向線の長さ
            double lineLength = hemisphereRadius * customSettings.directionLineRatio; // 天空図円の半径に対して
            string[] nwse = new string[] { "N", "W", "S", "E" };// 方向文字配列
            double nwseTextHight = lineLength;// 文字大きさ(方向線の長さと同じに)

            // 北西南東の順番で線と文字
            for(int i = 0; i < 4; i++)
            {
                // 真北から
                double angleDeg = 90.0 * (double)i + dueNorthDeg + 90.0;
                Point2d linePoint1 = new Point2d(centerPoint.X + hemisphereRadius, centerPoint.Y);
                linePoint1 = app.Geometry.RotatePoint(linePoint1, centerPoint, angleDeg);

                Point2d linePoint2 = new Point2d(centerPoint.X + hemisphereRadius + lineLength, centerPoint.Y);
                linePoint2 = app.Geometry.RotatePoint(linePoint2, centerPoint, angleDeg);

                LineShape lineShape = app.ShapeFactory.CreateLine(linePoint1, linePoint2);
                doc.BasicShapeSettings.CopyToShape(lineShape);

                // 色
                if(customSettings.skymap_CircleShapeLineColorNumber >= 0)
                    lineShape.ColorNumber = customSettings.skymap_CircleShapeLineColorNumber;

                // 線種
                lineShape.LinetypeNumber = 1;

                // 線幅
                if(customSettings.skymap_CircleShapeLinewidthNumber >= 0)
                    lineShape.LinewidthNumber = customSettings.skymap_CircleShapeLinewidthNumber;

                // レイヤのセット
                lineShape.Layer = currentLayer;

                // 図形追加
                shapes.Add(lineShape);

                // 東西南北文字
                string directionText = nwse[i];
                double textAngleDeg = 90.0 * (double)i + dueNorthDeg;
                AddTextShape(shapes, directionText, linePoint2, textAngleDeg, nwseTextHight, 0.0, Alignment.BottomMiddle, customSettings.triclinicTextColorNumber, doc, currentLayer);
            }

            return shapes;
        }

        /// <summary>
        /// 太陽軌跡の図形作成
        /// </summary>
        /// <param name="centerPoint"></param>
        /// <param name="hemisphereRadius"></param>
        /// <param name="dueNorthDeg"></param>
        /// <param name="latitudeDeg">観測地点の緯度(deg) </param>
        /// <param name="interval10Minutes"></param>
        /// <param name="drawSeasonsSolarTrajectoryFlg"></param>
        /// <param name="textHight"></param>
        /// <param name="solarSeasonNameShape"></param>
        /// <param name="doc"></param>
        /// <param name="currentLayer"></param>
        /// <returns></returns>
        private List<Shape> CreatsolarTrajectoryShapeList(Point2d centerPoint, double hemisphereRadius, double dueNorthDeg, double latitudeDeg, bool interval10Minutes, bool drawSeasonsSolarTrajectoryFlg, double textHight, List<Shape> solarSeasonNameShape, Document doc, Layer currentLayer)
        {
            List<Shape> solarTrajectoryShapes = new List<Shape>();
            Debug.Assert(solarSeasonNameShape != null && solarSeasonNameShape.Count() == 0);// 「四季の太陽軌跡」文字図形リスト図形リスト

            double latitudeRad = latitudeDeg * (Math.PI / 180d);// 観測地点の緯度(rad)
            int startSolarTime = 8;// 開始真太陽時
            int endSolarTime = 16;
            int intervalTime = 600;// データ取得時間間隔(秒) 600:10分間隔

            double largeMarkerSize = textHight * 0.5;

            // 天空図の円
            CircleShape circleShape = app.ShapeFactory.CreateCircle(centerPoint, hemisphereRadius);
            doc.BasicShapeSettings.CopyToShape(circleShape);

            // ループ回数
            int roopSize = 1;
            // 四季フラグOnのとき
            if(drawSeasonsSolarTrajectoryFlg == true)
                roopSize = 3;

            for(int i = 0; i < roopSize; i++)
            {
                // 日赤緯(冬至:-23.45°、春秋分:0°、夏至:23.45°)
                double solarDeclinationRad = -23.45 * (Math.PI / 180d);// 冬至 日赤緯(rad)
                string solarSeasonName = "冬至 太陽軌跡";
                if(i == 1)
                {
                    solarDeclinationRad = 0.0;// 春秋分 日赤緯(rad) 
                    solarSeasonName = "春秋分 太陽軌跡";
                }
                else if(i == 2)
                {
                    solarDeclinationRad = 23.45 * (Math.PI / 180d);// 夏至 日赤緯(rad) 
                    solarSeasonName = "夏至 太陽軌跡";
                }

                Point2d oldPoint = new Point2d();
                double solarTime = 0.0;
                int counter = 0;
                do
                {
                    // 真太陽時
                    solarTime = (double)startSolarTime + ((double)(intervalTime * counter) / 3600.0);

                    // 太陽高度を取得(rad)
                    double solarHeight = RCShadowDataCalculator.GetSolarHeight(latitudeRad, solarDeclinationRad, solarTime);

                    // 太陽方位角を取得(rad)
                    double solarDirection = RCShadowDataCalculator.GetSolarDirection(latitudeRad, solarDeclinationRad, solarTime, solarHeight); 

                    // 水平(xプラス)方向を南として考える
                    double a = hemisphereRadius * Math.Cos(solarHeight);
                    Point2d pointA = new Point2d(centerPoint.X + a, centerPoint.Y);
                    double solarDirectionDeg = solarDirection * (180d / Math.PI);// (rad -> Deg)
                    pointA = app.Geometry.RotatePoint(pointA, centerPoint, solarDirectionDeg);

                    // 最初のループ
                    if(counter == 0)
                    {
                        // 最初の1点目の点(大)
                        AddPointMarkerShape(solarTrajectoryShapes, pointA, PointMarkerType.FilledCircle, largeMarkerSize, customSettings.sunPointMarkerLColorNumber, doc, currentLayer);

                        // 最初の時間の文字(8時)
                        string outputText = startSolarTime.ToString();
                        Point2d textLocatePoint = new Point2d(pointA.X - textHight, pointA.Y);

                        // 文字図形を作成してリストに追加
                        AddTextShape(solarTrajectoryShapes, outputText, textLocatePoint, 90.0, textHight, 0.0, Alignment.MiddleMiddle, customSettings.triclinicTextColorNumber, doc, currentLayer);
                    }
                    else
                    {
                        LineShape lineShape = app.ShapeFactory.CreateLine(oldPoint, pointA);
                        doc.BasicShapeSettings.CopyToShape(lineShape);

                        // 色
                        if(customSettings.sunLineColorNumber >= 0)
                            lineShape.ColorNumber = customSettings.sunLineColorNumber;

                        // 線種
                        lineShape.LinetypeNumber = 1;

                        // 線幅
                        lineShape.LinewidthNumber = 1;

                        // レイヤのセット
                        lineShape.Layer = currentLayer;

                        solarTrajectoryShapes.Add(lineShape);

                        // 10分間隔で点を配置する場合
                        if(interval10Minutes == true && counter % 6 != 0)
                        {
                            // 点図形(小)
                            AddPointMarkerShape(solarTrajectoryShapes, pointA, PointMarkerType.FilledCircle, largeMarkerSize * 0.5, customSettings.sunPointMarkerSColorNumber, doc, currentLayer);
                        }
                        else if(counter % 6 == 0)
                        {
                            // 点図形(大)
                            AddPointMarkerShape(solarTrajectoryShapes, pointA, PointMarkerType.FilledCircle, largeMarkerSize, customSettings.sunPointMarkerLColorNumber, doc, currentLayer);

                            // endSolarTimeの文字(16時)
                            if(Math.Abs(solarTime - (double)endSolarTime) <= RcPrecisionConfusion)
                            {
                                // 16 時
                                string outputText = endSolarTime.ToString();
                                Point2d textLocatePoint = new Point2d(pointA.X - textHight, pointA.Y);

                                // 文字図形を作成してリストに追加
                                AddTextShape(solarTrajectoryShapes, outputText, textLocatePoint, 90.0, textHight, 0.0, Alignment.MiddleMiddle, customSettings.triclinicTextColorNumber, doc, currentLayer);

                                //「四季の太陽軌跡」文字
                                double textOffset = textHight * 1.5;
                                Point2d textLocatePoint2 = new Point2d(pointA.X, pointA.Y + textOffset);// 配置位置初期値
                                // 16時の太陽の位置の点から上に垂直に延びる半無限線
                                HalfInfiniteLineShape halfInfiniteLineShape = app.ShapeFactory.CreateHalfInfiniteLine(pointA, 90.0);
                                // 天空図円との交点から少し離した位置に置く
                                IntersectionPoint[] intersectionPoints = circleShape.GetIntersectionPoint(halfInfiniteLineShape);
                                if(intersectionPoints.Length > 0)
                                    textLocatePoint2.Y = intersectionPoints[0].PointOnShape1.Point.Y + textOffset;

                                // 文字図形を作成してリストに追加
                                AddTextShape(solarSeasonNameShape, solarSeasonName, textLocatePoint2, 90.0, textHight, 0.0, Alignment.MiddleLeft, customSettings.triclinicTextColorNumber, doc, currentLayer);
                            }
                        }
                    }

                    oldPoint = pointA;

                    // カウントアップ
                    counter++;

                } while(solarTime + (double)intervalTime / 3600.0 * 0.5 < (double)endSolarTime);// 0.5計算の誤差考慮
            }

            // 方角に合わせて回転&鏡像
            RotateAndMirrorShapes(solarTrajectoryShapes, centerPoint, dueNorthDeg + 270.0, dueNorthDeg + 270.0);

            // 方角に合わせて回転(「四季の太陽軌跡」文字図形リスト)
            foreach(Shape shape in solarSeasonNameShape)
                shape.Rotate(centerPoint, dueNorthDeg + 270.0);

            return solarTrajectoryShapes;
        }

        /// <summary>
        /// 位置番号を生成更新
        /// </summary>
        /// <param name="baseLine3DList"></param>
        /// <param name="overhangBaseLine3DList"></param>
        private void UpdateBaseLine3DPositionNumber(List<BaseLine3D> baseLine3DList, List<BaseLine3D> overhangBaseLine3DList)
        {
            int positionNumber = 1;
            foreach(BaseLine3D baseLine3D in baseLine3DList)
            {
                positionNumber = baseLine3D.SetPositionNumber(positionNumber);
            }

            foreach(BaseLine3D overhangBaseLine3D in overhangBaseLine3DList)
            {
                positionNumber = overhangBaseLine3D.SetPositionNumber(positionNumber);
            }
        }

        /// <summary>
        /// 正射影の出力図形作成
        /// </summary>
        /// <param name="orthographicProjectionShapeList"></param>
        /// <param name="overhangOrthographicProjectionShapeList"></param>
        /// <param name="cornerLines"></param>
        /// <param name="doc"></param>
        /// <param name="currentLayer"></param>
        /// <returns></returns>
        private List<Shape> CreateOrthographicProjectionShapeList(List<OrthographicProjectionShape> orthographicProjectionShapeList,
            List<OrthographicProjectionShape> overhangOrthographicProjectionShapeList, List<LinePoints> cornerLines, Document doc, Layer currentLayer)
        {
            List<Shape> shapes = new List<Shape>();

            // 正射影の線の図形
            foreach(OrthographicProjectionShape orthographicProjectionShape in orthographicProjectionShapeList)
            {
                PolylineShape polylineShape = orthographicProjectionShape.polylineShape;
                doc.BasicShapeSettings.CopyToShape(polylineShape);

                // レイヤのセット
                polylineShape.Layer = currentLayer;

                // 色
                if(customSettings.orthographicProjection_LineColorNumber >= 0)
                    polylineShape.ColorNumber = customSettings.orthographicProjection_LineColorNumber;

                // 線種
                polylineShape.LinetypeNumber = 1;

                // 線幅
                if(customSettings.orthographicProjection_LinewidthNumber >= 0)
                    polylineShape.LinewidthNumber = customSettings.orthographicProjection_LinewidthNumber;

                // 図形追加
                shapes.Add(polylineShape);
            }

            // オーバーハング対応（建物の底の部分）
            foreach(OrthographicProjectionShape orthographicProjectionShape in overhangOrthographicProjectionShapeList)
            {
                PolylineShape polylineShape = orthographicProjectionShape.polylineShape;
                doc.BasicShapeSettings.CopyToShape(polylineShape);

                // レイヤのセット
                polylineShape.Layer = currentLayer;

                // 色
                if(customSettings.orthographicProjection_LineColorNumber >= 0)
                    polylineShape.ColorNumber = customSettings.orthographicProjection_LineColorNumber;

                // 線種
                polylineShape.LinetypeNumber = 1;

                // 線幅
                if(customSettings.orthographicProjection_LinewidthNumber >= 0)
                    polylineShape.LinewidthNumber = customSettings.orthographicProjection_LinewidthNumber;

                // 図形追加
                shapes.Add(polylineShape);
            }


            // 建物の角の下から上（屋根位置）までを表す線
            foreach(LinePoints cornerLine in cornerLines)
            {
                LineShape lineShape = app.ShapeFactory.CreateLine(cornerLine.point1, cornerLine.point2);

                doc.BasicShapeSettings.CopyToShape(lineShape);

                // 図形追加

                // レイヤのセット
                lineShape.Layer = currentLayer;

                // 色
                if(customSettings.orthographicProjection_LineColorNumber >= 0)
                    lineShape.ColorNumber = customSettings.orthographicProjection_LineColorNumber;

                // 線種
                lineShape.LinetypeNumber = 1;

                // 線幅
                if(customSettings.orthographicProjection_LinewidthNumber >= 0)
                    lineShape.LinewidthNumber = customSettings.orthographicProjection_LinewidthNumber;

                // 追加
                shapes.Add(lineShape);
            }

            return shapes;
        }

        /// <summary>
        /// 天空図の円図形作成
        /// </summary>
        /// <param name="centerPoint"></param>
        /// <param name="hemisphereRadius"></param>
        /// <param name="doc"></param>
        /// <param name="currentLayer"></param>
        /// <returns></returns>
        private List<Shape> CreateSkymapCircleShapeList(Point2d centerPoint, double hemisphereRadius, Document doc, Layer currentLayer)
        {
            List<Shape> skymapCircleShapeList = new List<Shape>();

            // 天空図の円
            CircleShape circleShape = app.ShapeFactory.CreateCircle(centerPoint, hemisphereRadius);
            doc.BasicShapeSettings.CopyToShape(circleShape);

            // レイヤのセット
            circleShape.Layer = currentLayer;

            // 色
            if(customSettings.skymap_CircleShapeLineColorNumber >= 0)
                circleShape.ColorNumber = customSettings.skymap_CircleShapeLineColorNumber;

            // 線種
            circleShape.LinetypeNumber = 1;

            // 線幅
            if(customSettings.skymap_CircleShapeLinewidthNumber >= 0)
                circleShape.LinewidthNumber = customSettings.skymap_CircleShapeLinewidthNumber;

            skymapCircleShapeList.Add(circleShape);

            return skymapCircleShapeList;
        }

        /// <summary>
        /// 天空率等の基本情報図形作成
        /// </summary>
        /// <param name="samplingPointNumber"></param>
        /// <param name="centerPoint"></param>
        /// <param name="hemisphereRadius"></param>
        /// <param name="textHight"></param>
        /// <param name="scaleText"></param>
        /// <param name="latitudeDeg"></param>
        /// <param name="locationHeight"></param>
        /// <param name="triclinicMode"></param>
        /// <param name="triclinicSkyRateAttribute"></param>
        /// <param name="drawNWSEFlg"></param>
        /// <param name="onlySkyRate"></param>
        /// <param name="doc"></param>
        /// <param name="currentLayer"></param>
        /// <param name="infoShapesBndBox"></param>
        /// <returns></returns>
        private List<Shape> CreateSkyRateBasicInfoShapes(int samplingPointNumber, Point2d centerPoint, double hemisphereRadius, double textHight, string scaleText,
                                                    double latitudeDeg, double locationHeight, TriclinicMode triclinicMode, TriclinicSkyRateAttribute triclinicSkyRateAttribute, bool drawNWSEFlg, bool onlySkyRate,
                                                    Document doc, Layer currentLayer, out BndBox infoShapesBndBox)
        {
            List<Shape> infoShapes = new List<Shape>();
            infoShapesBndBox = null;
            
            // 位置移動(測定円の下に置く)
            Point2d locatePoint = new Point2d(centerPoint.X, centerPoint.Y);
            if(!onlySkyRate)
            {
                locatePoint.X = centerPoint.X;
                locatePoint.Y = centerPoint.Y - hemisphereRadius - textHight * 2.0;
                if(drawNWSEFlg)// 天空図に方角表記を付けている場合は少し下げる
                    locatePoint.Y -= (hemisphereRadius * customSettings.directionLineRatio + textHight);
            }

            double lineSpacing = textHight * 0.5;// 行間隔

            string outputText = "";
            int lineCount = 0;
           
            // 測定点番号
            if(samplingPointNumberCheckBox.Checked)
            {
                if(lineCount > 0)
                    locatePoint.Y -= (textHight + lineSpacing);

                string numberText = samplingPointNumber.ToString();
                outputText = "測定点 " + customSettings.samplingNumberHeadText + numberText;
                AddTextShape(infoShapes, outputText, locatePoint, 0.0, textHight, 0.0, Alignment.TopMiddle, customSettings.triclinicTextColorNumber, doc, currentLayer);
                lineCount++;
            }

            // 天空率のみを表示でなければ
            if(!onlySkyRate)
            {
                // 目盛間隔
                if(scaleText != "")
                {
                    if(lineCount > 0)
                        locatePoint.Y -= (textHight + lineSpacing);

                    string scaleComboBoxSelectedText2 = scaleText == "" ? "0°" : scaleText;
                    outputText = "目盛間隔 " + scaleComboBoxSelectedText2;
                    AddTextShape(infoShapes, outputText, locatePoint, 0.0, textHight, 0.0, Alignment.TopMiddle, customSettings.triclinicTextColorNumber, doc, currentLayer);
                    lineCount++;
                }

                // 緯度(太陽軌跡を表示Onのとき)
                if(solarTrajectoryCheckBox.Checked)
                {
                    if(lineCount > 0)
                        locatePoint.Y -= (textHight + lineSpacing);

                    string latitudeStr = DegreeToString(latitudeDeg, 2);
                    outputText = "緯度 = " + latitudeStr;
                    AddTextShape(infoShapes, outputText, locatePoint, 0.0, textHight, 0.0, Alignment.TopMiddle, customSettings.triclinicTextColorNumber, doc, currentLayer);
                    lineCount++;
                }

                // 正射影 測定面高さ
                if(lineCount > 0)
                    locatePoint.Y -= (textHight + lineSpacing);

                double locationHeight2 = Math.Round(locationHeight, addInSettings.skymapLengthValueDecimals, MidpointRounding.AwayFromZero);// 四捨五入
                string skymapLengthValueStringFormatParamf = String.Format("{{0:f{0}}}", addInSettings.skymapLengthValueDecimals);
                string locationHeightStr = String.Format(skymapLengthValueStringFormatParamf, locationHeight2);// ex.小数点以下3桁まで0.000
                outputText = "正射影 測定面高さ = " + locationHeightStr + " m";
                AddTextShape(infoShapes, outputText, locatePoint, 0.0, textHight, 0.0, Alignment.TopMiddle, customSettings.triclinicTextColorNumber, doc, currentLayer);
                lineCount++;
            }

            // 水平円射影面積、天空率
            if(circularProjectionAreaCheckBox.Checked || skyRateCheckBox.Checked)
            {
                double skyMapCircleArea = hemisphereRadius * hemisphereRadius * Math.PI;// 円の面積(初期値)
                skyMapCircleArea = GetSkymapRoundValue(skyMapCircleArea, triclinicMode, addInSettings.skymapAreaValueDecimals);// 三斜求積モードに合わせた値(「切り捨て」or 「切り上げ」)の取得
                double circularProjectionArea = 0.0;
                double skyRate = 0.0;

                if(triclinicSkyRateAttribute != null)
                {
                    // 測定円面積
                    skyMapCircleArea = triclinicSkyRateAttribute.skyMapCircleArea;

                    // 水平円射影面積
                    circularProjectionArea = triclinicSkyRateAttribute.circularProjectionArea;
                    skyRate = triclinicSkyRateAttribute.skyRate;
                }

                // 水平円射影面積
                if(circularProjectionAreaCheckBox.Checked && !onlySkyRate)
                {
                    if(lineCount > 0)
                        locatePoint.Y -= (textHight + lineSpacing);

                    // 円の面積
                    double skyMapCircleArea2 = skyMapCircleArea;
                    string skymapAreaValueStringFormatParamf = String.Format("{{0:f{0}}}", addInSettings.skymapAreaValueDecimals);
                    string skyMapCircleAreaStr = String.Format(skymapAreaValueStringFormatParamf, skyMapCircleArea2);

                    // 水平円射影面積(円の面積, 建築物投影面積)
                    double circularProjectionArea2 = circularProjectionArea;
                    string circularProjectionAreaStr = String.Format(skymapAreaValueStringFormatParamf, circularProjectionArea2);
                    outputText = "水平円射影面積 " + skyMapCircleAreaStr + " , " + circularProjectionAreaStr;
                    AddTextShape(infoShapes, outputText, locatePoint, 0.0, textHight, 0.0, Alignment.TopMiddle, customSettings.triclinicTextColorNumber, doc, currentLayer);
                    lineCount++;
                }

                // 天空率
                if(skyRateCheckBox.Checked || onlySkyRate)
                {
                    if(lineCount > 0)
                        locatePoint.Y -= (textHight + lineSpacing);

                    double skyRate2 = skyRate;
                    string skymapRateValueStringFormatParamf = String.Format("{{0:f{0}}}", addInSettings.skyRateValueDecimals);
                    string skyRateStr = String.Format(skymapRateValueStringFormatParamf, skyRate2);
                    outputText = customSettings.skyRateHeadText + skyRateStr + " %";
                    AddTextShape(infoShapes, outputText, locatePoint, 0.0, textHight, 0.0, Alignment.TopMiddle, customSettings.triclinicTextColorNumber, doc, currentLayer);
                    lineCount++;
                }
            }

            // 全図形範囲取得
            infoShapesBndBox = GetShapesBndBox(infoShapes);

            return infoShapes;
        }

        /// <summary>
        /// 図形リスト中の全図形範囲取得
        /// </summary>
        /// <param name="shapes"></param>
        /// <returns></returns>
        private BndBox GetShapesBndBox(List<Shape> shapes)
        {
            BndBox shapesBndBox = null;
            foreach(Shape shape in shapes)
            {
                Point2d[] shapeBndingBox = shape.GetBoundingBox();
                if(shapesBndBox == null)
                {
                    shapesBndBox = new BndBox(shapeBndingBox[0], shapeBndingBox[1]);
                }
                else
                {
                    shapesBndBox.Add(shapeBndingBox[0]);
                    shapesBndBox.Add(shapeBndingBox[1]);
                }
            }

            return shapesBndBox;
        }

        /// <summary>
        /// 三斜求積表の図形作成
        /// </summary>
        /// <param name="scaleText"></param>
        /// <param name="locationHeight"></param>
        /// <param name="centerPoint"></param>
        /// <param name="hemisphereRadius"></param>
        /// <param name="maxSeparateAngleText"></param>
        /// <param name="triclinicTriangleList"></param>
        /// <param name="triclinicSkyRateAttribute"></param>
        /// <param name="skyMapTableLocateMode"></param>
        /// <param name="locatePoint"></param>
        /// <param name="textHight"></param>
        /// <param name="forRubberBand"></param>
        /// <param name="doc"></param>
        /// <param name="currentLayer"></param>
        /// <returns></returns>
        private List<Shape> CreateQuadratureTableShapes(string scaleText, double locationHeight, Point2d centerPoint, double hemisphereRadius, string maxSeparateAngleText, List<TriclinicTriangle> triclinicTriangleList, TriclinicSkyRateAttribute triclinicSkyRateAttribute,
                                                      SkyMapTableLocateMode skyMapTableLocateMode, Point2d locatePoint, double textHight, bool forRubberBand, Document doc, Layer currentLayer)
        {
            List<Shape> shapes = new List<Shape>();
            string outputText = "";
            string triangleNumberKakkoStart = "";
            string triangleNumberKakkoEnd = "";

            // 目盛間隔
            if(scaleText != "")
            {
                string scaleComboBoxSelectedText2 = scaleText == "" ? "0°" : scaleText;
                outputText = "目盛間隔 " + scaleComboBoxSelectedText2;
            }

            // 正射影 測定面高さ
            if(outputText != "")
                outputText += Environment.NewLine;// 改行

            double locationHeight2 = Math.Round(locationHeight, addInSettings.skymapLengthValueDecimals, MidpointRounding.AwayFromZero);// 四捨五入
            string skymapLengthValueStringFormatParamf = String.Format("{{0:f{0}}}", addInSettings.skymapLengthValueDecimals);
            string locationHeightStr = String.Format(skymapLengthValueStringFormatParamf, locationHeight2);
            outputText += "正射影 測定面高さ = " + locationHeightStr + " m";

            // 天空図の半径
            outputText += Environment.NewLine;// 改行
            double hemisphereRadius2 = Math.Round(hemisphereRadius, addInSettings.hemisphereRadiusDecimals, MidpointRounding.AwayFromZero);// 四捨五入 整数表示
            string hemisphereRadiusStringFormatParamf = String.Format("{{0:f{0}}}", addInSettings.hemisphereRadiusDecimals);
            string hemisphereRadiusStr = String.Format(hemisphereRadiusStringFormatParamf, hemisphereRadius2);
            outputText += "天空図半径(R) = " + hemisphereRadiusStr + "(実寸mm)";

            // 求積タイプと三斜計算最大分割角度 カッコ文字のタイプもセット
            outputText += Environment.NewLine;// 改行
            if(triclinicSkyRateAttribute.triclinicMode == TriclinicMode.TriclinicMode_Plan)
            {
                outputText += customSettings.plan_TriclinicTitle_Standard;
                triangleNumberKakkoStart = customSettings.plan_TriangleNumberKakkoStart;
                triangleNumberKakkoEnd = customSettings.plan_TriangleNumberKakkoEnd;
            }
            else if(triclinicSkyRateAttribute.triclinicMode == TriclinicMode.TriclinicMode_Standard)
            {
                outputText += customSettings.standard_TriclinicTitle;
                triangleNumberKakkoStart = customSettings.standard_TriangleNumberKakkoStart;
                triangleNumberKakkoEnd = customSettings.standard_TriangleNumberKakkoEnd;
            }

            outputText += " 三斜計算    最大分割角 = " + maxSeparateAngleText;

            // リストに追加
            double lineSpacing = textHight * 0.5;
            TextShape textShape = AddTextShape(shapes, outputText, locatePoint, 0.0, textHight, lineSpacing, Alignment.TopLeft, customSettings.triclinicTextColorNumber, doc, currentLayer);// 位置は表を作成した後調整しなおす

            // 文字範囲
            Point2d[] textShapeBoundingBox = textShape.GetBoundingBox();
            BndBox infoTextBoundingBox1 = new BndBox(textShapeBoundingBox[0], textShapeBoundingBox[1]);

            // 表
            // 番号順にソートしておく（念のため）
            triclinicTriangleList.Sort((a, b) => Math.Sign(a.triangleNumber - b.triangleNumber));
            int listCount = triclinicTriangleList.Count;

            // セル内での文字オフセット
            double textOffsetH = textHight * 0.25;
            double textOffsetW = textHight * 0.5;

            // 表のサイズ
            double cellHight = textHight + textOffsetH * 2.0;// セル高さ
            int itemNameRows = 2;// 項目行に使う行数(最初の項目名と最後の合計の行)
            double tableHight = cellHight * (double)(listCount + itemNameRows);// 表の高さ、項目行と合計値の行分多くする
            double tableWidth = textHight * customSettings.tableWidthTextScale;// 表の幅
            int clumnCount = 4;// (列の数)
            double cellWidth = tableWidth / (double)clumnCount;// セル幅

            // 表の左上座標
            double tableOffsetH = textHight * 0.5;
            double tableOffsetW = textHight * 4.0;
            Point2d tableLocatePointTL = new Point2d();
            // 表の配置が下・下のとき
            if(skyMapTableLocateMode == SkyMapTableLocateMode.SkyMapTableLocateMode_LowerLower)
            {
                tableLocatePointTL = new Point2d(locatePoint.X - tableWidth * 0.5, locatePoint.Y - infoTextBoundingBox1.GetHeight() - tableOffsetH); // X座標はlocatePointが表の中央になるように、Y座標はquadratureTableBndBoxPointsの左下の座標を元に
            }
            // 表の配置が右・下のとき
            else if(skyMapTableLocateMode == SkyMapTableLocateMode.SkyMapTableLocateMode_RightLower)
            {
                tableLocatePointTL = new Point2d(centerPoint.X + hemisphereRadius + tableOffsetW, centerPoint.Y + (infoTextBoundingBox1.GetHeight() + tableOffsetH + tableHight) * 0.5); // X座標は表の左側が円の右側になるように、Y座標は情報文字と表の中央
            }
            // 任意の場合
            else if(skyMapTableLocateMode == SkyMapTableLocateMode.SkyMapTableLocateMode_Free)
            {
                tableLocatePointTL = locatePoint;
            }
            else
            {
                Debug.Assert(false);
            }

            // 情報文字1(表の上)の配置位置確定
            double infoTextIndent = textHight;
            textShape.Point = new Point2d(tableLocatePointTL.X + infoTextIndent, tableLocatePointTL.Y + tableOffsetH + infoTextBoundingBox1.GetHeight());
 
            // 縦線
            for(int i = 0; i < clumnCount + 1; i++)
            {
                double lineLength = tableHight;
                if(i > 0 && i < 3)
                    lineLength -= cellHight;

                Point2d lineStartPoint = new Point2d(tableLocatePointTL.X + cellWidth * (double)i, tableLocatePointTL.Y);
                Point2d lineEndPoint = new Point2d(lineStartPoint.X, tableLocatePointTL.Y - lineLength);

                LineShape lineShape = app.ShapeFactory.CreateLine(lineStartPoint, lineEndPoint);

                doc.BasicShapeSettings.CopyToShape(lineShape);

                // 図形追加
                // レイヤのセット
                lineShape.Layer = currentLayer;

                // 色
                if(customSettings.tableLineColorNumber >= 0)
                    lineShape.ColorNumber = customSettings.tableLineColorNumber;

                // 線種
                lineShape.LinetypeNumber = 1;

                // 線幅
                if(customSettings.tableLinewidthNumber >= 0)
                    lineShape.LinewidthNumber = customSettings.tableLinewidthNumber;

                // 追加
                shapes.Add(lineShape);
            }

            // 横線
            for(int i = 0; i < listCount + itemNameRows + 1; i++)
            {
                double lineLength = tableWidth;
 
                Point2d lineStartPoint = new Point2d(tableLocatePointTL.X, tableLocatePointTL.Y - cellHight * (double)i);
                Point2d lineEndPoint = new Point2d(tableLocatePointTL.X + lineLength, lineStartPoint.Y);

                LineShape lineShape = app.ShapeFactory.CreateLine(lineStartPoint, lineEndPoint);

                doc.BasicShapeSettings.CopyToShape(lineShape);

                // 図形追加
                // レイヤのセット
                lineShape.Layer = currentLayer;

                // 色
                if(customSettings.tableLineColorNumber >= 0)
                    lineShape.ColorNumber = customSettings.tableLineColorNumber;

                // 線種
                lineShape.LinetypeNumber = 1;

                // 線幅
                if(customSettings.tableLinewidthNumber >= 0)
                    lineShape.LinewidthNumber = customSettings.tableLinewidthNumber;

                // 追加
                shapes.Add(lineShape);
            }

            // 表内の文字
            if(!forRubberBand)
            {
                double totalTriclinicTriangleArea = 0.0;
                string[] clumnNames = new string[] { "三角形 No.", "底辺(mm)", "高さ(mm)", "面積(mm2)" }; // 項目行の文字
                Debug.Assert(clumnNames.Count() == clumnCount);

                for(int i = 0; i < listCount + itemNameRows; i++)
                {
                    Point2d cellTextPoint = new Point2d(tableLocatePointTL.X, tableLocatePointTL.Y - cellHight * (double)(i + 1) + textOffsetH);// 配置点初期値
                    string cellText = "";
                    Alignment textAlignment = Alignment.BottomMiddle;

                    TriclinicTriangle triclinicTriangle = null;
                    if(i > 0 && i < listCount + 1)
                        triclinicTriangle = triclinicTriangleList[i - 1];

                    for(int j = 0; j < clumnCount; j++)
                    {
                        if(j == 0)
                        {
                            textAlignment = Alignment.BottomMiddle;
                            cellTextPoint.X = tableLocatePointTL.X + cellWidth * 0.5;
                        }
                        else
                        {
                            textAlignment = Alignment.BottomRight;
                            cellTextPoint.X = tableLocatePointTL.X + cellWidth * (double)(j + 1) - textOffsetW;
                        }

                        // 項目行の文字
                        if(i == 0)
                        {
                            cellText = clumnNames[j];
                        }
                        // 三斜三角形情報
                        else if(i < listCount + 1)
                        {
                            // 三角形 No.
                            if(j == 0)
                            {
                                cellText = triangleNumberKakkoStart + triclinicTriangle.triangleNumber.ToString() + triangleNumberKakkoEnd;
                            }
                            // 底辺(mm)
                            else if(j == 1)
                            {
                                double itemValue = triclinicTriangle.baselineLength;
                                string skymapTriangleLengthValueStringFormatParamf = String.Format("{{0:f{0}}}", addInSettings.skymapTriangleLengthValueDecimals);
                                cellText = String.Format(skymapTriangleLengthValueStringFormatParamf, itemValue);
                            }
                            // 高さ(mm)
                            else if(j == 2)
                            {
                                double itemValue = triclinicTriangle.hight;
                                string skymapTriangleLengthValueStringFormatParamf = String.Format("{{0:f{0}}}", addInSettings.skymapTriangleLengthValueDecimals);
                                cellText = String.Format(skymapTriangleLengthValueStringFormatParamf, itemValue);
                            }
                            // 面積(mm2)
                            else if(j == 3)
                            {
                                double itemValue = triclinicTriangle.triangleArea;
                                string skymapAreaValueStringFormatParamf = String.Format("{{0:f{0}}}", addInSettings.skymapAreaValueDecimals);
                                cellText = String.Format(skymapAreaValueStringFormatParamf, itemValue);

                                // 合計値更新 文字列をdouble型に変換
                                double.TryParse(cellText, out double heigtriangleAreaht2);
                                totalTriclinicTriangleArea += heigtriangleAreaht2;
                            }

                        }
                        // 合計の行
                        else
                        {
                            if(j == 2)
                            {
                                cellText = "三斜面積合計";
                            }
                            else if(j == 3)
                            {
                                string skymapAreaValueStringFormatParamf = String.Format("{{0:f{0}}}", addInSettings.skymapAreaValueDecimals);
                                cellText = String.Format(skymapAreaValueStringFormatParamf, totalTriclinicTriangleArea);
                            }
                        }


                        if(cellText == "")
                            continue;

                        // 文字追加
                        AddTextShape(shapes, cellText, cellTextPoint, 0.0, textHight, 0.0, textAlignment, customSettings.triclinicTextColorNumber, doc, currentLayer);
                    }
                }
            }

            // 表の下の情報
            // 扇形角度
            double sectorAngleDeg = triclinicSkyRateAttribute.sectorAngleDeg;
            string skymapAngleValueStringFormatParamf2 = String.Format("{{0:f{0}}}", addInSettings.skymapAngleValueDecimals);
            string sectorAngleDegStr = String.Format(skymapAngleValueStringFormatParamf2, sectorAngleDeg);
            outputText = "扇形面積(扇形中心角 = " + sectorAngleDegStr + "°)";

            Point2d textLocatePoint = new Point2d(tableLocatePointTL.X + infoTextIndent, tableLocatePointTL.Y - tableHight - tableOffsetH - textHight);

            // リストに追加
            textShape = AddTextShape(shapes, outputText, textLocatePoint, 0.0, textHight, 0.0, Alignment.BottomLeft, customSettings.triclinicTextColorNumber, doc, currentLayer);

            // 扇形面積の値文字
            double sectorArea = triclinicSkyRateAttribute.sectorArea;
            string skymapAreaValueStringFormatParamf2 = String.Format("{{0:f{0}}}", addInSettings.skymapAreaValueDecimals);
            string sectorAreaStr = String.Format(skymapAreaValueStringFormatParamf2, sectorArea);
            outputText = sectorAreaStr;
            textLocatePoint.X = tableLocatePointTL.X + tableWidth - textOffsetW;
            textLocatePoint.Y = textShape.Point.Y;// 左側に追加した文字と同じ
            // リストに追加
            textShape = AddTextShape(shapes, outputText, textLocatePoint, 0.0, textHight, 0.0, Alignment.BottomRight, customSettings.triclinicTextColorNumber, doc, currentLayer);

            // 水平円射影面積 = 扇形面積 - 三斜面積合計
            outputText = "扇形面積 - 三斜面積合計 = 水平円射影面積";
            textLocatePoint.X = tableLocatePointTL.X + infoTextIndent;
            textLocatePoint.Y = textShape.Point.Y - lineSpacing - textHight;// 前に追加した文字の1行下
            // リストに追加
            textShape = AddTextShape(shapes, outputText, textLocatePoint, 0.0, textHight, 0.0, Alignment.BottomLeft, customSettings.triclinicTextColorNumber, doc, currentLayer);

            // 水平円射影面積の値文字
            double circularProjectionArea = triclinicSkyRateAttribute.circularProjectionArea;
            string circularProjectionAreaStr = String.Format(skymapAreaValueStringFormatParamf2, circularProjectionArea);
            outputText = circularProjectionAreaStr;
            textLocatePoint.X = tableLocatePointTL.X + tableWidth - textOffsetW;
            textLocatePoint.Y = textShape.Point.Y;// 左側に追加した文字と同じ
            // リストに追加
            textShape = AddTextShape(shapes, outputText, textLocatePoint, 0.0, textHight, 0.0, Alignment.BottomRight, customSettings.triclinicTextColorNumber, doc, currentLayer);

            // 天空図円面積
            outputText = "天空図円面積";
            textLocatePoint.X = tableLocatePointTL.X + infoTextIndent;
            textLocatePoint.Y = textShape.Point.Y - lineSpacing - textHight;// 前に追加した文字の1行下
            // リストに追加
            textShape = AddTextShape(shapes, outputText, textLocatePoint, 0.0, textHight, 0.0, Alignment.BottomLeft, customSettings.triclinicTextColorNumber, doc, currentLayer);

            // 水平円射影面積の値文字
            double skyMapCircleArea = triclinicSkyRateAttribute.skyMapCircleArea;
            string skyMapCircleAreaStr = String.Format(skymapAreaValueStringFormatParamf2, skyMapCircleArea);
            outputText = skyMapCircleAreaStr;
            textLocatePoint.X = tableLocatePointTL.X + tableWidth - textOffsetW;
            textLocatePoint.Y = textShape.Point.Y;// 左側に追加した文字と同じ
            // リストに追加
            textShape = AddTextShape(shapes, outputText, textLocatePoint, 0.0, textHight, 0.0, Alignment.BottomRight, customSettings.triclinicTextColorNumber, doc, currentLayer);

            // "三斜計算"
            outputText = "三斜計算";
            textLocatePoint.X = tableLocatePointTL.X + infoTextIndent;
            textLocatePoint.Y = textShape.Point.Y - lineSpacing - textHight;// 前に追加した文字の1行下
            // リストに追加
            textShape = AddTextShape(shapes, outputText, textLocatePoint, 0.0, textHight, 0.0, Alignment.BottomLeft, customSettings.triclinicTextColorNumber, doc, currentLayer);

            // 三斜計算 天空率文字
            double skyRate = triclinicSkyRateAttribute.skyRate;
            string skymapRateValueStringFormatParamf2 = String.Format("{{0:f{0}}}", addInSettings.skyRateValueDecimals);
            string skyRateStr = String.Format(skymapRateValueStringFormatParamf2, skyRate);
            outputText = "(" + skyMapCircleAreaStr + " - " + circularProjectionAreaStr + ") / " + skyMapCircleAreaStr + " * 100.0 = " + skyRateStr;

            textLocatePoint.X = tableLocatePointTL.X + infoTextIndent;
            textLocatePoint.Y = textShape.Point.Y - lineSpacing - textHight;// 前に追加した文字の1行下
            // リストに追加
            textShape = AddTextShape(shapes, outputText, textLocatePoint, 0.0, textHight, 0.0, Alignment.BottomLeft, customSettings.triclinicTextColorNumber, doc, currentLayer);

            // 天空率
            outputText = customSettings.skyRateHeadText + skyRateStr + " %";
            textLocatePoint.X = tableLocatePointTL.X + tableWidth;
            textLocatePoint.Y = textShape.Point.Y - lineSpacing - textHight;// 前に追加した文字の1行下
            // リストに追加 (基準は右下)
            textShape = AddTextShape(shapes, outputText, textLocatePoint, 0.0, textHight, 0.0, Alignment.BottomRight, customSettings.triclinicTextColorNumber, doc, currentLayer);

            return shapes;
        }

     
        /// <summary>
        /// 建物位置確認表作成
        /// </summary>
        /// <param name="scaleText"></param>
        /// <param name="positionDataItemList"></param>
        /// <param name="locationHeight"></param>
        /// <param name="hemisphereRadius"></param>
        /// <param name="triclinicMode"></param>
        /// <param name="skyMapTableLocateMode"></param>
        /// <param name="locatePoint"></param>
        /// <param name="textHight"></param>
        /// <param name="forRubberBand"></param>
        /// <param name="doc"></param>
        /// <param name="currentLayer"></param>
        /// <returns></returns>
        private List<Shape> CreatePositionTableShapes(string scaleText, List<PositionDataItem> positionDataItemList, double locationHeight, double hemisphereRadius, TriclinicMode triclinicMode,
                                                SkyMapTableLocateMode skyMapTableLocateMode, Point2d locatePoint, double textHight, bool forRubberBand, Document doc, Layer currentLayer)
        {
            List<Shape> shapes = new List<Shape>();
            string outputText = "";
            string positionNumberKakkoStart = "";
            string positionNumberKakkoEnd = "";

            Debug.Assert(positionDataItemList != null);

            // 目盛間隔
            if(scaleText != "")
            {
                string scaleComboBoxSelectedText2 = scaleText == "" ? "0°" : scaleText;
                outputText = "目盛間隔 " + scaleComboBoxSelectedText2;
            }

            // 正射影 測定面高さ
            if(outputText != "")
                outputText += Environment.NewLine;// 改行

            double locationHeight2 = Math.Round(locationHeight, addInSettings.skymapLengthValueDecimals, MidpointRounding.AwayFromZero);
            string skymapLengthValueStringFormatParamf = String.Format("{{0:f{0}}}", addInSettings.skymapLengthValueDecimals);
            string locationHeightStr = String.Format(skymapLengthValueStringFormatParamf, locationHeight2);//小数点以下3桁まで0.000
            outputText += "正射影 測定面高さ = " + locationHeightStr + " m";

            // 天空図の半径
            outputText += Environment.NewLine;// 改行
            double hemisphereRadius2 = Math.Round(hemisphereRadius, addInSettings.hemisphereRadiusDecimals, MidpointRounding.AwayFromZero);// 四捨五入 整数表示
            string hemisphereRadiusStringFormatParamf = String.Format("{{0:f{0}}}", addInSettings.hemisphereRadiusDecimals);
            string hemisphereRadiusStr = String.Format(hemisphereRadiusStringFormatParamf, hemisphereRadius2);
            outputText += "天空図半径(R) = " + hemisphereRadiusStr + "(実寸mm)";

            // 建物位置確認表
            outputText += Environment.NewLine;// 改行
            string subTitle = "";
            if(triclinicMode == TriclinicMode.TriclinicMode_Standard)
            {
                subTitle = customSettings.standard_TriclinicTitle;
                positionNumberKakkoStart = customSettings.standard_PositionNumberKakkoStart;
                positionNumberKakkoEnd = customSettings.standard_PositionNumberKakkoEnd;
            }
            else if(triclinicMode == TriclinicMode.TriclinicMode_Plan)
            {
                subTitle = customSettings.plan_TriclinicTitle_Standard;
                positionNumberKakkoStart = customSettings.plan_PositionNumberKakkoStart;
                positionNumberKakkoEnd = customSettings.plan_PositionNumberKakkoEnd;
            }

            outputText += "建物位置確認表" + "  " + subTitle;

            // リストに追加
            double lineSpacing = textHight * 0.5;
            TextShape textShape = AddTextShape(shapes, outputText, locatePoint, 0.0, textHight, lineSpacing, Alignment.TopLeft, customSettings.triclinicTextColorNumber, doc, currentLayer);// 位置は表を作成した後調整しなおす

            // 文字範囲
            Point2d[] textShapeBoundingBox = textShape.GetBoundingBox();
            BndBox infoTextBoundingBox1 = new BndBox(textShapeBoundingBox[0], textShapeBoundingBox[1]);

            // 表に表示させるデータリスト作成
            //List<PositionDataItem> positionDataItemList = CreatePositionDataItemList(dueNorthDeg, samplingPoint, hemisphereRadius, baseLine3DList, overhangBaseLine3DList);
            int listCount = positionDataItemList.Count;

            // セル内での文字オフセット
            double textOffsetH = textHight * 0.25;
            double textOffsetW = textHight * 0.5;

            // 表のサイズ
            double cellHight = textHight + textOffsetH * 2.0;// セル高さ
            int itemNameRows = 3;// 最初の項目行に使う行数
            double tableHight = cellHight * (double)(listCount + itemNameRows);// 表の高さ、項目行と合計値の行分多くする
            double tableWidth = textHight * customSettings.tableWidthTextScale;// 表の幅
            int clumnCount = 6;// (列の数)
            double cellWidth = tableWidth / (double)clumnCount;// セル幅

            // 表の左上座標
            double tableOffsetH = textHight * 0.5;
            double tableOffsetW = textHight * 2.0;
            Point2d tableLocatePointTL = new Point2d();
            // 表の配置が下・下、右・下のとき
            if(skyMapTableLocateMode == SkyMapTableLocateMode.SkyMapTableLocateMode_LowerLower || skyMapTableLocateMode == SkyMapTableLocateMode.SkyMapTableLocateMode_RightLower)
            {
                tableLocatePointTL = new Point2d(locatePoint.X - tableWidth * 0.5, locatePoint.Y - infoTextBoundingBox1.GetHeight() - tableOffsetH); // X座標はlocatePointが表の中央になるように、Y座標はquadratureTableBndBoxPointsの左下の座標を元に
            }
            // 任意の場合
            else
            {
                tableLocatePointTL = locatePoint;
            }

            // 情報文字1(表の上)の配置位置確定
            double infoTextIndent = textHight;
            textShape.Point = new Point2d(tableLocatePointTL.X + infoTextIndent, tableLocatePointTL.Y + tableOffsetH + infoTextBoundingBox1.GetHeight());

            // 縦線
            for(int i = 0; i < clumnCount + 1; i++)
            {
                double lineStartPointY = tableLocatePointTL.Y;
                if(i == 2 || i == 4 || i == 5)
                    lineStartPointY -= cellHight;

                Point2d lineStartPoint = new Point2d(tableLocatePointTL.X + cellWidth * (double)i, lineStartPointY);
                Point2d lineEndPoint = new Point2d(lineStartPoint.X, tableLocatePointTL.Y - tableHight);

                LineShape lineShape = app.ShapeFactory.CreateLine(lineStartPoint, lineEndPoint);

                doc.BasicShapeSettings.CopyToShape(lineShape);

                // 図形追加
                // レイヤのセット
                lineShape.Layer = currentLayer;

                // 色
                if(customSettings.tableLineColorNumber >= 0)
                    lineShape.ColorNumber = customSettings.tableLineColorNumber;

                // 線種
                lineShape.LinetypeNumber = 1;

                // 線幅
                if(customSettings.tableLinewidthNumber >= 0)
                    lineShape.LinewidthNumber = customSettings.tableLinewidthNumber;

                // 追加
                shapes.Add(lineShape);
            }

            // 横線
            for(int i = 0; i < listCount + itemNameRows + 1; i++)
            {
                double lineStartPointX = tableLocatePointTL.X;
                if(i == 1)
                    lineStartPointX += cellWidth;
                else if(i == 2)
                    continue;

                Point2d lineStartPoint = new Point2d(lineStartPointX, tableLocatePointTL.Y - cellHight * (double)i);
                Point2d lineEndPoint = new Point2d(tableLocatePointTL.X + tableWidth, lineStartPoint.Y);

                LineShape lineShape = app.ShapeFactory.CreateLine(lineStartPoint, lineEndPoint);

                doc.BasicShapeSettings.CopyToShape(lineShape);

                // 図形追加
                // レイヤのセット
                lineShape.Layer = currentLayer;

                // 色
                if(customSettings.tableLineColorNumber >= 0)
                    lineShape.ColorNumber = customSettings.tableLineColorNumber;

                // 線種
                lineShape.LinetypeNumber = 1;

                // 線幅
                if(customSettings.tableLinewidthNumber >= 0)
                    lineShape.LinewidthNumber = customSettings.tableLinewidthNumber;

                // 追加
                shapes.Add(lineShape);
            }

            if(!forRubberBand)
            {
                // 項目行の文字
                string[] clumnNames1 = new string[] { "位置", "配置図", "", "", "天空図", "" };
                string[] clumnNames2 = new string[] { customSettings.samplingNumberHeadText, "距離", "高さ", "方位角", "仰角:h", "R*cos(h)"};// customSettings.samplingNumberHeadText:"No."
                string[] clumnNames3 = new string[] { "", "実寸(m)", "実寸(m)", "(°)", "(°)", "実寸(mm)" };
                Debug.Assert(clumnNames1.Count() == clumnCount && clumnNames2.Count() == clumnCount && clumnNames3.Count() == clumnCount);

                for(int i = 0; i < listCount + itemNameRows; i++)
                {
                    Point2d cellTextPoint = new Point2d(tableLocatePointTL.X, tableLocatePointTL.Y - cellHight * (double)(i + 1) + textOffsetH);// 配置点初期値
                    string cellText = "";
                    Alignment textAlignment = Alignment.BottomMiddle;

                    // 項目名
                    if(i == 0)
                    {
                        textAlignment = Alignment.MiddleMiddle;
                        for(int j = 0; j < clumnCount; j++)
                        {
                            cellText = clumnNames1[j];
                            if(cellText == "")
                                continue;

                            if(j == 0)
                            {
                                cellTextPoint.Y = tableLocatePointTL.Y - cellHight;
                                cellTextPoint.X = tableLocatePointTL.X + cellWidth * 0.5;
                            }
                            else if(j == 1)
                            {
                                cellTextPoint.X = tableLocatePointTL.X + cellWidth * (double)(j + 1);
                                cellTextPoint.Y = tableLocatePointTL.Y - cellHight * (double)(i + 1) + cellHight * 0.5;
                            }
                            else if(j == 4)
                            {
                                cellTextPoint.X = tableLocatePointTL.X + cellWidth * (double)(j + 1) - cellWidth * 0.5;
                                cellTextPoint.Y = tableLocatePointTL.Y - cellHight * (double)(i + 1) + cellHight * 0.5;
                            }

                            // 文字追加
                            AddTextShape(shapes, cellText, cellTextPoint, 0.0, textHight, 0.0, textAlignment, customSettings.triclinicTextColorNumber, doc, currentLayer);
                        }
                    }
                    else if(i == 1)
                    {
                        cellTextPoint.Y = tableLocatePointTL.Y - cellHight * (double)(i + 1) + cellHight * 0.5;
                        for(int j = 0; j < clumnCount; j++)
                        {
                            cellText = clumnNames2[j];
                            if(cellText == "")
                                continue;

                            if(j == 0)
                            {
                                textAlignment = Alignment.TopMiddle;
                                cellTextPoint.X = tableLocatePointTL.X + cellWidth * 0.5;
                            }
                            else
                            {
                                textAlignment = Alignment.MiddleRight;
                                cellTextPoint.X = tableLocatePointTL.X + cellWidth * (double)(j + 1) - textOffsetW;
                            }

                            // 文字追加
                            AddTextShape(shapes, cellText, cellTextPoint, 0.0, textHight, 0.0, textAlignment, customSettings.triclinicTextColorNumber, doc, currentLayer);
                        }
                    }
                    else if(i == 2)
                    {
                        textAlignment = Alignment.MiddleRight;
                        cellTextPoint.Y = tableLocatePointTL.Y - cellHight * (double)(i + 1) + cellHight * 0.5;
                        for(int j = 0; j < clumnCount; j++)
                        {
                            cellText = clumnNames3[j];
                            if(cellText == "")
                                continue;

                            cellTextPoint.X = tableLocatePointTL.X + cellWidth * (double)(j + 1) - textOffsetW;

                            // 文字追加
                            AddTextShape(shapes, cellText, cellTextPoint, 0.0, textHight, 0.0, textAlignment, customSettings.triclinicTextColorNumber, doc, currentLayer);
                        }
                    }
                    // データー
                    else
                    {
                        Debug.Assert(i >= itemNameRows);
                        PositionDataItem positionDataItem = positionDataItemList[i - itemNameRows];

                        for(int j = 0; j < clumnCount; j++)
                        {
                            int number = 0;
                            double itemValue = 0.0;

                            // 位置No.
                            if(j == 0)
                            {
                                number = positionDataItem.Number;
                            }
                            // 距離（配置図）
                            else if(j == 1)
                            {
                                itemValue = positionDataItem.Distance;
                            }
                            // 高さ（配置図）
                            else if(j == 2)
                            {
                                itemValue = positionDataItem.Hight;
                            }
                            // 方位角（天空図）
                            else if(j == 3)
                            {
                                itemValue = positionDataItem.AngleDeg;
                            }
                            // 仰角:h（天空図）
                            else if(j == 4)
                            {
                                itemValue = positionDataItem.AngleDegH;
                            }
                            // R*Cos(h) （天空図）
                            else if(j == 5)
                            {
                                itemValue = positionDataItem.DistanceH;
                            }

                            if(j == 0)
                            {
                                cellTextPoint.Y = tableLocatePointTL.Y - cellHight * (double)(i + 1) + textOffsetH;
                                cellTextPoint.X = tableLocatePointTL.X + cellWidth * 0.5;

                                cellText = positionNumberKakkoStart + number.ToString() + positionNumberKakkoEnd;

                                textAlignment = Alignment.BottomMiddle;
                            }
                            else
                            {
                                cellTextPoint.Y = tableLocatePointTL.Y - cellHight * (double)(i + 1) + textOffsetH;
                                cellTextPoint.X = tableLocatePointTL.X + cellWidth * (double)(j + 1) - textOffsetW;

                                if(j == 3 || j == 4)
                                {
                                    // 角度
                                    double itemValue2 = Math.Round(itemValue, addInSettings.skymapAngleValueDecimals, MidpointRounding.AwayFromZero);// 四捨五入
                                    string skymapAngleValueStringFormatParamf2 = String.Format("{{0:f{0}}}", addInSettings.skymapAngleValueDecimals);
                                    cellText = String.Format(skymapAngleValueStringFormatParamf2, itemValue2);
                                }
                                else
                                {
                                    // 長さ
                                    double itemValue2 = Math.Round(itemValue, addInSettings.skymapLengthValueDecimals, MidpointRounding.AwayFromZero);// 四捨五入
                                    string skymapLengthValueStringFormatParamf2 = String.Format("{{0:f{0}}}", addInSettings.skymapLengthValueDecimals);
                                    cellText = String.Format(skymapLengthValueStringFormatParamf2, itemValue2);
                                }

                                textAlignment = Alignment.BottomRight;
                            }

                            // 文字追加
                            AddTextShape(shapes, cellText, cellTextPoint, 0.0, textHight, 0.0, textAlignment, customSettings.triclinicTextColorNumber, doc, currentLayer);
                        }
                    }
                }
            }


            return shapes;
        }

        /// <summary>
        /// 位置表データアイテムリスト作成
        /// </summary>
        /// <param name="dueNorthDeg"></param>
        /// <param name="samplingPoint"></param>
        /// <param name="hemisphereRadius"></param>
        /// <param name="baseLine3DList"></param>
        /// <param name="overhangBaseLine3DList"></param>
        /// <returns></returns>
        private List<PositionDataItem> CreatePositionDataItemList(double dueNorthDeg, Point2d samplingPoint, double hemisphereRadius, List<BaseLine3D> baseLine3DList, List<BaseLine3D> overhangBaseLine3DList)
        {
            List<PositionDataItem> positionDataItemList = new List<PositionDataItem>();

            // BaseLine3D リスト（オーバーハングのと合わせる）
            List <BaseLine3D> baseLine3DList2 = new List<BaseLine3D>();
            baseLine3DList2.AddRange(baseLine3DList);
            baseLine3DList2.AddRange(overhangBaseLine3DList);

            // 配置番号が設定されていて天空図に描画するものに絞る
            List<BaseLine3D> baseLine3DList3 = baseLine3DList2.FindAll(n => n.IsPositionNumber() == true && (n.isDrawStartPointSidePositionNumberAtSkymap == true || n.isDrawEndPointSidePositionNumberAtSkymap == true));
            
            // 配置番号でソート
            baseLine3DList3.Sort((a, b) => Math.Sign(a.startPointSidePositionNumber - b.startPointSidePositionNumber));// 始点側だけで比べていい

            // 始点側と終点側のデータを作成
            foreach(BaseLine3D baseLine3D in baseLine3DList3)
            {
                if(!baseLine3D.IsPositionNumber())
                    continue;

                // 始点側
                if(baseLine3D.isDrawStartPointSidePositionNumberAtSkymap == true)
                {
                    int number = baseLine3D.startPointSidePositionNumber;// 番号
                    Point2d baseLinePoint = baseLine3D.line.StartPoint;// ベース線の端点の座標
                    double baseLineHight = baseLine3D.startPointHeight;// ベース線の端点の高さ

                    // データ作成
                    PositionDataItem positionDataItem = CreatePositionDataItem(dueNorthDeg, samplingPoint, hemisphereRadius, number, baseLinePoint, baseLineHight);
                    if(positionDataItem == null)
                        continue;

                    // 直前のデータが前の番号で内容が同じ場合は直前のデータを削除                    
                    int positionDataItemListCount = positionDataItemList.Count();
                    if(positionDataItemListCount > 0)
                    {
                        PositionDataItem lastPositionDataItem = positionDataItemList.Last();
                        if(lastPositionDataItem.Number == positionDataItem.Number - 1 && Math.Abs(positionDataItem.Distance - lastPositionDataItem.Distance) <= RcPrecisionConfusion && Math.Abs(positionDataItem.Hight - lastPositionDataItem.Hight) <= RcPrecisionConfusion)
                            positionDataItemList.RemoveAt(positionDataItemListCount - 1);
                    }

                    // リストに追加
                    positionDataItemList.Add(positionDataItem);
                }

                // 終点側
                if(baseLine3D.isDrawEndPointSidePositionNumberAtSkymap == true)
                {
                    int number = baseLine3D.endPointSidePositionNumber;// 番号
                    Point2d baseLinePoint = baseLine3D.line.EndPoint;// ベース線の端点の座標
                    double baseLineHight = baseLine3D.endPointHeight;// ベース線の端点の高さ

                    // データ作成
                    PositionDataItem positionDataItem = CreatePositionDataItem(dueNorthDeg, samplingPoint, hemisphereRadius, number, baseLinePoint, baseLineHight);
                    if(positionDataItem == null)
                        continue;

                    // リストに追加
                    positionDataItemList.Add(positionDataItem);
                }
            }

            // 最初と最後が同じ内容の場合は最後を削除
            int listCount = positionDataItemList.Count();
            if(listCount > 1)
            {
                PositionDataItem positionDataItem1 = positionDataItemList.First();
                PositionDataItem positionDataItem2 = positionDataItemList.Last();
                if(Math.Abs(positionDataItem1.Distance - positionDataItem2.Distance) <= RcPrecisionConfusion && Math.Abs(positionDataItem1.Hight - positionDataItem2.Hight) <= RcPrecisionConfusion)
                    positionDataItemList.RemoveAt(listCount - 1);
            }

            return positionDataItemList;
        }

        /// <summary>
        /// 位置表データアイテム作成
        /// </summary>
        /// <param name="dueNorthDeg"></param>
        /// <param name="samplingPoint"></param>
        /// <param name="hemisphereRadius"></param>
        /// <param name="number"></param>
        /// <param name="baseLinePoint"></param>
        /// <param name="baseLineHight"></param>
        /// <returns></returns>
        private PositionDataItem CreatePositionDataItem(double dueNorthDeg, Point2d samplingPoint, double hemisphereRadius, int number, Point2d baseLinePoint, double baseLineHight)
        {
            PositionDataItem positionDataItem = null;

            if(!GetOrthographicProjectionDistanceAndAngle(samplingPoint, hemisphereRadius, baseLinePoint, baseLineHight, out double centerToOrthographicProjectionDistance, out double centerToPointAngleDeg))
                return null;

            // 測定点からベース線の端点までの距離(実寸m)
            double fromSamplingPointDistance = app.Geometry.GetDistance(samplingPoint, baseLinePoint) / 1000.0;

            // 高さ(実寸m)
            double hight = baseLineHight / 1000.0;

            // 天空図の中心からベース上の点の角度
            double angleDeg = centerToPointAngleDeg;

            // 天空図の中心点から正射影の点までの距離(R*Cos(h))
            double distanceH = centerToOrthographicProjectionDistance;

            // 仰角:h
            Debug.Assert(hemisphereRadius > Precision.Confusion);
            double cosh = distanceH / hemisphereRadius;
            double angleDegH = app.Geometry.RadianToDegree(Math.Acos(cosh));// °

            positionDataItem = new PositionDataItem(number, fromSamplingPointDistance, hight, angleDeg, angleDegH, distanceH);

            return positionDataItem;
        }

        /// <summary>
        /// リストに指定した設定の文字の文字図形を追加
        /// </summary>
        /// <param name="shapes"></param>
        /// <param name="str"></param>
        /// <param name="locatePoint"></param>
        /// <param name="angle"></param>
        /// <param name="textHight"></param>
        /// <param name="lineSpacing"></param>
        /// <param name="alignment"></param>
        /// <param name="colorNumber"></param>
        /// <param name="doc"></param>
        /// <param name="currentLayer"></param>
        private TextShape AddTextShape(List<Shape> shapes, string str, Point2d locatePoint, double angle, double textHight, double lineSpacing, Alignment alignment, int colorNumber, Document doc, Layer currentLayer)
        {
            TextShape textShape = app.ShapeFactory.CreateText(str, locatePoint, angle);
            doc.TextSettings.CopyToShape(textShape);

            // 文字高さセット
            textShape.FontHeight = textHight;

            // 向きは横書き固定
            textShape.DirectionVertical = false;

            // 行間をセット
            textShape.LineSpacing = lineSpacing;

            // 配置基準位置
            textShape.Alignment = alignment;

            // 色
            if(colorNumber >= 0)
                textShape.ColorNumber = colorNumber;

            // レイヤのセット
            textShape.Layer = currentLayer;

            // リストに図形の追加
            shapes.Add(textShape);

            return textShape;
        }

        /// <summary>
        /// リストに指定した設定の点図形を追加
        /// </summary>
        /// <param name="shapes"></param>
        /// <param name="locatePoint"></param>
        /// <param name="pointMarkerType"></param>
        /// <param name="size"></param>
        /// <param name="colorNumber"></param>
        /// <param name="doc"></param>
        /// <param name="currentLayer"></param>
        /// <returns></returns>
        private PointMarkerShape AddPointMarkerShape(List<Shape> shapes, Point2d locatePoint, PointMarkerType pointMarkerType, double size, int colorNumber, Document doc, Layer currentLayer)
        {
            PointMarkerShape pointMarkerShape = app.ShapeFactory.CreatePointMarker(locatePoint);
            doc.PointMarkerSettings.CopyToShape(pointMarkerShape);

            // タイプ
            pointMarkerShape.MarkerType = pointMarkerType;

            // サイズ
            pointMarkerShape.Size = size;

            // 色
            if(colorNumber >= 0)
                pointMarkerShape.ColorNumber = colorNumber;

            // レイヤのセット
            pointMarkerShape.Layer = currentLayer;

            // リストに図形の追加
            shapes.Add(pointMarkerShape);

            return pointMarkerShape;
        }


        /// <summary>
        /// 測定点が建物内部か
        /// </summary>
        /// <param name="centerPoint"></param>
        /// <param name="hemisphereRadius"></param>
        /// <param name="baseLine3DList"></param>
        /// <param name="orthographicProjectionShapeList"></param>
        /// <returns></returns>
        private bool IsPointInBuilding(Point2d centerPoint, double hemisphereRadius, List<BaseLine3D> baseLine3DList, List<OrthographicProjectionShape> orthographicProjectionShapeList)
        {
            // 測定点からX+方向に延びる水平な線が正射影の線と交わる場合、
            // 反対側のX -方向に延びる線が同じ正射影の繋がった線と交点を持つ場合は内部点と判断する

            Point2d point1 = new Point2d(centerPoint.X + hemisphereRadius, centerPoint.Y);
            LineShape lineShape1 = app.ShapeFactory.CreateLine(centerPoint, point1);
            List<IntersectionPointInfo> intersectionPointInfoList = new List<IntersectionPointInfo>();
            GetIntersectionPointInfo(intersectionPointInfoList, lineShape1, orthographicProjectionShapeList, 1);

            Point2d point2 = new Point2d(centerPoint.X - hemisphereRadius, centerPoint.Y);
            LineShape lineShape2 = app.ShapeFactory.CreateLine(centerPoint, point2);
            List<IntersectionPointInfo> oppositesideIntersectionPointInfoList = new List<IntersectionPointInfo>();
            GetIntersectionPointInfo(oppositesideIntersectionPointInfoList, lineShape2, orthographicProjectionShapeList, 1);

            int intersectionPointInfoListCount = intersectionPointInfoList.Count;
            int oppositesideIntersectionPointInfoListCount = oppositesideIntersectionPointInfoList.Count;

            for(int i = 0; i < intersectionPointInfoListCount; i++)
            {
                IntersectionPointInfo intersectionPointInfo = intersectionPointInfoList[i];
                int baseLineID = intersectionPointInfo.baseLineID;

                for(int j = 0; j < oppositesideIntersectionPointInfoListCount; j++)
                {
                    int baseLineID0 = oppositesideIntersectionPointInfoList[j].baseLineID;
                    BaseLine3D baseLine0 = baseLine3DList.FirstOrDefault(o => o.baseLineID == baseLineID0);
                    if(baseLine0 == null)
                        continue;

                    if(baseLine0.IsConnectID(baseLineID))
                        return true;
                }
            }

            return false;
        }

        /// <summary>
        /// 建物範囲のハッチング図形作成
        /// </summary>
        /// <param name="centerPoint"></param>
        /// <param name="hemisphereRadius"></param>
        /// <param name="radialIntervalAngleDeg">放射状に描く線の間隔(10.0以下で、且つ360を割ったとき商が偶数で余りのない値)</param>
        /// <param name="baseLine3DList"></param>
        /// <param name="orthographicProjectionShapeList"></param>
        /// <param name="overhangOrthographicProjectionShapeList"></param>
        /// <param name="isCenterPointInside">測定点が建物の内側フラグ</param>
        /// <param name=" isIinsideTopRoof">建物内部の上が屋根フラグ</param>
        /// <param name="doc"></param>
        /// <param name="currentLayer"></param>
        /// <returns></returns>
        private List<Shape> CreateHatchShapes(Point2d centerPoint, double hemisphereRadius, double radialIntervalAngleDeg, List<BaseLine3D> baseLine3DList, List<OrthographicProjectionShape> orthographicProjectionShapeList, List<OrthographicProjectionShape> overhangOrthographicProjectionShapeList,
                                            bool isCenterPointInside, bool isIinsideTopRoof, Document doc, Layer currentLayer)
        {
            int hatchLineCount = (int)(360.0 / radialIntervalAngleDeg);
            Debug.Assert(hatchLineCount % 2 == 0);

            bool isOverhang = overhangOrthographicProjectionShapeList.Count > 0 ? true : false;// オーバーハングしているところがあるか

            List<IntersectionPointInfoListSet> intersectionPointInfoListSetList = new List<IntersectionPointInfoListSet>();

            // 中心から放射状に(一定角度間隔で)円周上まで描いたとき正射影の線との交点を１つだけ持つ線をその交点から円周上まで描く
            for(int i = 0; i < hatchLineCount; i++)
            {
                // 測定点（中心点）を通る線の天空図円の交点
                Point2d pointOnCircle = new Point2d(centerPoint.X + hemisphereRadius, centerPoint.Y);
                pointOnCircle = app.Geometry.RotatePoint(pointOnCircle, centerPoint, radialIntervalAngleDeg * (double)i);// 回転移動

                LineShape hatchLineShape = app.ShapeFactory.CreateLine(centerPoint, pointOnCircle);

                // 正射影の線と交差するものが建物の内を表すとしてハッチングの線に用いる
                // 交点情報収集（ハッチング線が全ての正射影の線と持つ交点のパラメータや交点、オーバーハングしている場合の線のタイプを中心点から近い順に並べる）
                List<IntersectionPointInfo> intersectionPointInfoList = new List<IntersectionPointInfo>();
                GetIntersectionPointInfo(intersectionPointInfoList, hatchLineShape, orthographicProjectionShapeList, 1);
                // オーバーハングしているところがある場合
                if(isOverhang)
                    GetIntersectionPointInfo(intersectionPointInfoList, hatchLineShape, overhangOrthographicProjectionShapeList, -1);

                // 交点パラメータの小さい順に並び替え
                intersectionPointInfoList.Sort((a, b) => Math.Sign(a.parameter - b.parameter));

                IntersectionPointInfoListSet intersectionPointInfoListSet = new IntersectionPointInfoListSet(hatchLineShape, intersectionPointInfoList);
                intersectionPointInfoListSetList.Add(intersectionPointInfoListSet);
            }

            // ハッチングの線で正射影の線で切れる部分があればそこで線を切り分ける
            List<Shape> hatchShapes = new List<Shape>();// 図形リスト
            for(int i = 0; i < intersectionPointInfoListSetList.Count; i++)
            {
                List<Shape> cutHatchLineShapes = GetCutHatchLineShapes(intersectionPointInfoListSetList, i, isCenterPointInside, isIinsideTopRoof, baseLine3DList);
                hatchShapes.AddRange(cutHatchLineShapes);
            }

            // 線の色とレイヤセット
            foreach(LineShape lineShape in hatchShapes)
            {
                // 色
                doc.BasicShapeSettings.CopyToShape(lineShape);
                if(customSettings.hatchColorNumber >= 0)
                    lineShape.ColorNumber = customSettings.hatchColorNumber;// ハッチングの線の色

                // 線種
                lineShape.LinetypeNumber = 1;

                // 線幅
                if(customSettings.hatchLinewidthNumber >= 0)
                    lineShape.LinewidthNumber = customSettings.hatchLinewidthNumber;

                // レイヤのセット
                lineShape.Layer = currentLayer;
            }

            return hatchShapes;
        }

        /// <summary>
        /// 交点情報収集
        /// 指定した線が全ての正射影の線と持つ交点のパラメータや交点、オーバーハングしている場合の線のタイプを取得
        /// </summary>
        /// <param name="intersectionPointInfoList"></param>
        /// <param name="lineShape"></param>
        /// <param name="orthographicProjectionShapeList"></param>
        /// <param name="crossLineType">交点を持つ相手の線のタイプ(正射影の上の線:1、下の線:-1)</param>
        private void GetIntersectionPointInfo(List<IntersectionPointInfo> intersectionPointInfoList, LineShape lineShape, List<OrthographicProjectionShape> orthographicProjectionShapeList, int crossLineType)
        {
            Debug.Assert(intersectionPointInfoList != null);
            Debug.Assert(crossLineType == -1 || crossLineType == 1);

            foreach(OrthographicProjectionShape orthographicProjectionShape in orthographicProjectionShapeList)
            {
                PolylineShape polylineShape = orthographicProjectionShape.polylineShape;
                int baseLineID = orthographicProjectionShape.baseLineID;
                IntersectionPoint[] intersectionPoints = lineShape.GetIntersectionPoint(polylineShape);
                int pointCount = intersectionPoints.Length;
                for(int i = 0; i < pointCount; i++)
                {
                    IntersectionPoint intersectionPoint = intersectionPoints[i];
                    double parameter = intersectionPoint.PointOnShape1.Parameter;
                    Point2d point = intersectionPoint.PointOnShape1.Point;
                    IntersectionPointInfo intersectionPointInfo = new IntersectionPointInfo(parameter, point, baseLineID, crossLineType);
                    intersectionPointInfoList.Add(intersectionPointInfo);
                }
            }
        }

        /// <summary>
        /// 指定したインデックスの線の正射影との交点情報を取得
        /// </summary>
        /// <param name="intersectionPointInfoListSetList"></param>
        /// <param name="listIndex"></param>
        /// <param name="lineShape"></param>
        /// <param name="intersectionPointInfoList"></param>
        /// <param name="oppositesideIntersectionPointInfoList"></param>
        private void GetIntersectionPointInfoList(List<IntersectionPointInfoListSet> intersectionPointInfoListSetList, int listIndex,
            out LineShape lineShape, out List<IntersectionPointInfo> intersectionPointInfoList, out List<IntersectionPointInfo> oppositesideIntersectionPointInfoList)
        {
            int intersectionPointInfoListSetListCount = intersectionPointInfoListSetList.Count;
            Debug.Assert(listIndex < intersectionPointInfoListSetListCount);
            Debug.Assert(intersectionPointInfoListSetListCount % 2 == 0);
            lineShape = intersectionPointInfoListSetList[listIndex].lineShape;
            intersectionPointInfoList = intersectionPointInfoListSetList[listIndex].intersectionPointInfoList;

            // 指定した線の反対側の線のインデックス(オーバーハングしていて測定点が内部点だった場合に必要)
            int oppositesideIndex = listIndex + intersectionPointInfoListSetListCount / 2;
            if(oppositesideIndex >= intersectionPointInfoListSetListCount)
                oppositesideIndex -= intersectionPointInfoListSetListCount;

            oppositesideIntersectionPointInfoList = intersectionPointInfoListSetList[oppositesideIndex].intersectionPointInfoList;
        }

        /// <summary>
        /// 正射影の線で切れる部分があればそこで線を切り分ける
        /// </summary>
        /// <param name="intersectionPointInfoListSetList"></param>
        /// <param name="listIndex"></param>
        /// <param name="insideFlg"></param>
        /// <param name="isIinsideTopRoof"></param>
        /// <param name="baseLine3DList"></param>
        /// <returns></returns>
        private List<Shape> GetCutHatchLineShapes(List<IntersectionPointInfoListSet> intersectionPointInfoListSetList, int listIndex, bool insideFlg, bool isIinsideTopRoof, List<BaseLine3D> baseLine3DList)
        {
            // 指定したインデックスの線の正射影との交点情報を取得
            LineShape hatchLineShape = null;
            List<IntersectionPointInfo> intersectionPointInfoList = null;
            List<IntersectionPointInfo> oppositesideIntersectionPointInfoList = null;
            GetIntersectionPointInfoList(intersectionPointInfoListSetList, listIndex, out hatchLineShape, out intersectionPointInfoList, out oppositesideIntersectionPointInfoList);

            // 交点が構造物の上(+)の線と下(-)の線の順番を見る +- +-, +-+- +-, または++-- ++など、0になった後に再び+になるところがあればそこまでは線を切り離す。
            // 測定点が内部の場合、測定点の上に屋根がなければ、その部分にハッチングは掛けない。屋根ならハッチングを掛ける。

            List<Shape> cutHatchLineShapes = new List<Shape>();// 図形リスト

            Point2d startPoint = hatchLineShape.StartPoint;
            Point2d endPoint = hatchLineShape.EndPoint;
            int intersectionPointInfoListCount = intersectionPointInfoList.Count;

            bool hatchTopRoof = false;
            if(insideFlg && isIinsideTopRoof)
                hatchTopRoof = true;

            // 交点ある場合
            int sumCrossLineTypeValue = 0;
            bool continuousFlg = false;// 線の始点終点を確定せず続けるフラグ
            for(int i = 0; i < intersectionPointInfoListCount; i++)
            {
                bool isLast = i == intersectionPointInfoListCount - 1 ? true : false;

                // 中心(元の線の始点)に最も近い点から円周上までの線をハッチングの線にする
                IntersectionPointInfo intersectionPointInfo = intersectionPointInfoList[i];
                int baseLineID = intersectionPointInfo.baseLineID;
                int crossLineTypeValue = intersectionPointInfo.crossLineType;// 交点を持つ線のタイプ値
                Point2d thisPoint = intersectionPointInfoList[i].point;

                // ここで0から+に切り替わる場合
                if(sumCrossLineTypeValue == 0 && crossLineTypeValue == 1)
                {
                    if(i == 0 && hatchTopRoof)
                    {

                    }
                    else if(!continuousFlg)
                    {
                        startPoint = thisPoint;
                    }

                    if(isLast)
                    {
                        endPoint = hatchLineShape.EndPoint;
                        LineShape cutHatchLineShape = app.ShapeFactory.CreateLine(startPoint, endPoint);
                        // ハッチングの線追加
                        cutHatchLineShapes.Add(cutHatchLineShape);
                    }
                }
                // ここで+から0になる場合
                else if(sumCrossLineTypeValue == 1 && crossLineTypeValue == -1)
                {
                    // この線までのbaseLineIDのと同じIDの線(繋がっているIDも含む)が
                    // この先の交点の線のIDと同じになるものがある場合はここで終点を決めずに進める
                    continuousFlg = false;

                    for(int j = i - 1; j >= 0; j--)
                    {
                        int baseLineID0 = intersectionPointInfoList[j].baseLineID;
                        BaseLine3D baseLine0 = baseLine3DList.FirstOrDefault(o => o.baseLineID == baseLineID0);
                        if(baseLine0 == null)
                            continue;

                        for(int k = i + 1; k < intersectionPointInfoListCount; k++)
                        {
                            if(baseLine0.IsConnectID(intersectionPointInfoList[k].baseLineID))
                            {
                                continuousFlg = true;
                                break;
                            }
                        }

                        if(continuousFlg)
                            break;
                    }

                    // 測定点が内部ときは、反対側の線の交点でも状態を見る
                    if(hatchTopRoof && !continuousFlg)
                    {
                        int oppositesideIntersectionPointInfoListCount = oppositesideIntersectionPointInfoList.Count;
                        for(int j = 0; j < oppositesideIntersectionPointInfoListCount; j++)
                        {
                            int baseLineID0 = oppositesideIntersectionPointInfoList[j].baseLineID;
                            BaseLine3D baseLine0 = baseLine3DList.FirstOrDefault(o => o.baseLineID == baseLineID0);
                            if(baseLine0 == null)
                                continue;

                            for(int k = i + 1; k < intersectionPointInfoListCount; k++)
                            {
                                if(baseLine0.IsConnectID(intersectionPointInfoList[k].baseLineID))
                                {
                                    continuousFlg = true;
                                    break;
                                }
                            }

                            if(continuousFlg)
                                break;
                        }
                    }

                    if(!continuousFlg)
                    {
                        endPoint = thisPoint;
                        LineShape cutHatchLineShape = app.ShapeFactory.CreateLine(startPoint, endPoint);
                        // ハッチングの線追加
                        cutHatchLineShapes.Add(cutHatchLineShape);
                    }
                }
                else if(sumCrossLineTypeValue > 0 && isLast)
                {
                    endPoint = hatchLineShape.EndPoint;
                    LineShape cutHatchLineShape = app.ShapeFactory.CreateLine(startPoint, endPoint);
                    // ハッチングの線追加
                    cutHatchLineShapes.Add(cutHatchLineShape);
                }

                sumCrossLineTypeValue += crossLineTypeValue;
            }

            return cutHatchLineShapes;
        }

        /// <summary>
        /// 天空率属性値計算取得
        /// </summary>
        /// <param name="triclinicMode"></param>
        /// <param name="hemisphereRadius"></param>
        /// <param name="triclinicTriangleList"></param>
        /// <returns></returns>
        private TriclinicSkyRateAttribute GetSkyRate(TriclinicMode triclinicMode, double hemisphereRadius, List<TriclinicTriangle> triclinicTriangleList)
        {
            double skyMapRadius = hemisphereRadius; // 天空図半径

            double sectorAngleDeg = 0.0;            // 扇形中心角
            double totalTriclinicTriangleArea = 0.0;// 三斜面積合計

            foreach(TriclinicTriangle triclinicTriangle in triclinicTriangleList)
            {
                // 中心角度
                Point2d centerPoint = triclinicTriangle.centerPoint;
                Point2d hightlineTopPoint = triclinicTriangle.hightlineTopPoint;
                Point2d baselinePoint = triclinicTriangle.baselinePoint;
                Debug.Assert(!IsPointEquals(centerPoint, hightlineTopPoint) && !IsPointEquals(centerPoint, baselinePoint));

                // 2点間角度
                double centerAngleDeg = app.Geometry.GetAngle(centerPoint, hightlineTopPoint, baselinePoint);
                Debug.Assert(centerAngleDeg <= 360.0);
                if(centerAngleDeg > 180.0)
                    centerAngleDeg = 360.0 - centerAngleDeg;

                // 扇形中心角加算
                sectorAngleDeg += centerAngleDeg;

                // 面積加算
                totalTriclinicTriangleArea += triclinicTriangle.triangleArea;
            }

            TriclinicSkyRateAttribute triclinicSkyRateAttribute = new TriclinicSkyRateAttribute(triclinicMode, skyMapRadius, sectorAngleDeg, totalTriclinicTriangleArea, addInSettings);
            return triclinicSkyRateAttribute;
        }

        /// <summary>
        /// 位置を表す文字作成
        /// </summary>
        /// <param name="baseLine3DList"></param>
        /// <param name="orthographicProjectionShapeList"></param>
        /// <param name="triangleInfoList"></param>
        /// <param name="triclinicMode"></param>
        /// <param name="drawPositionNumberText"></param>
        /// <param name="drawAllPointNumber"></param>
        /// <param name="optionShapes"></param>
        /// <param name="doc"></param>
        /// <param name="currentLayer"></param>
        /// <param name="positionNumberTextHight"></param>
        /// <param name="textHight"></param>
        /// <returns></returns>
        private List<PositionNumberShape> CreateSkymapPositionTexts(List<BaseLine3D> baseLine3DList, List<OrthographicProjectionShape> orthographicProjectionShapeList, List<BaseTriangleInfo> triangleInfoList, TriclinicMode triclinicMode, bool drawPositionNumberText, bool drawAllPointNumber, List<PositionNumberShape> optionShapes, Document doc, Layer currentLayer, double positionNumberTextHight, double textHight)
        {
            List<PositionNumberShape> positionShapeList = new List<PositionNumberShape>();

            string startKakko = "";
            string endKakko = "";
            if(triclinicMode == TriclinicMode.TriclinicMode_Standard)
            {
                startKakko = customSettings.standard_PositionNumberKakkoStart;
                endKakko = customSettings.standard_PositionNumberKakkoEnd;
            }
            else if(triclinicMode == TriclinicMode.TriclinicMode_Plan)
            {
                startKakko = customSettings.plan_PositionNumberKakkoStart;
                endKakko = customSettings.plan_PositionNumberKakkoEnd;
            }

            // 図面上の元の線の頂点に位置を表す文字を出力する
            if(drawPositionNumberText)
            {
                List<Shape> tempArrayShapeList = new List<Shape>();
                foreach(BaseLine3D baseLine in baseLine3DList)
                {
                    // 始点側
                    if(baseLine.startPointSidePositionNumber > 0)
                    {
                        string outputText = startKakko + baseLine.startPointSidePositionNumber.ToString() + endKakko;
                        Point2d textLocatePoint = baseLine.line.StartPoint;

                        // 文字図形を作成してリストに追加
                        TextShape textShape = AddTextShape(tempArrayShapeList, outputText, textLocatePoint, 0.0, positionNumberTextHight, 0.0, Alignment.BottomMiddle, customSettings.triclinicTextColorNumber, doc, currentLayer);
                        
                        PositionNumberShape positionNumberShape = new PositionNumberShape();
                        positionNumberShape.Number = baseLine.startPointSidePositionNumber;
                        positionNumberShape.Shapes.Add(textShape);
                        optionShapes.Add(positionNumberShape);
                    }

                    // 終点側(始点側と高さが異なる場合に始点側と異なる番号が設定されている)
                    if(baseLine.endPointSidePositionNumber > 0 && baseLine.startPointSidePositionNumber != baseLine.endPointSidePositionNumber)
                    {
                        string outputText = startKakko + baseLine.endPointSidePositionNumber.ToString() + endKakko;
                        Point2d textLocatePoint = baseLine.line.EndPoint;

                        // 文字図形を作成してリストに追加
                        TextShape textShape = AddTextShape(tempArrayShapeList, outputText, textLocatePoint, 0.0, positionNumberTextHight, 0.0, Alignment.BottomMiddle, customSettings.triclinicTextColorNumber, doc, currentLayer);

                        PositionNumberShape positionNumberShape = new PositionNumberShape();
                        positionNumberShape.Number = baseLine.endPointSidePositionNumber;
                        positionNumberShape.Shapes.Add(textShape);
                        optionShapes.Add(positionNumberShape);
                    }
                }
            }

            // 正射影の頂点位置
            foreach(OrthographicProjectionShape orthographicProjectionShape in orthographicProjectionShapeList)
            {
                PolylineShape polylineShape = orthographicProjectionShape.polylineShape;

                Point2d[] polylinePoints = ((PolylineShape)polylineShape).Points;
                int polylinePointCount = polylinePoints.Length;
                Debug.Assert(polylinePointCount > 1);
                Point2d polylineStartPoint = polylinePoints.First();
                Point2d polylineEndPoint = polylinePoints.Last();

                int baseLineID = orthographicProjectionShape.baseLineID;
                BaseLine3D baseLine = baseLine3DList.FirstOrDefault(o => o.baseLineID == baseLineID);
                Debug.Assert(baseLine != null);
                int startSideNumber = baseLine.startPointSidePositionNumber;
                int endSideNumber = baseLine.endPointSidePositionNumber;

                bool startSideNumberTextOutFlg = false;
                bool endSideNumberTextOutFlg = false;
                // 全ての配置位置を出力する場合
                if(drawAllPointNumber)
                {
                    startSideNumberTextOutFlg = true;
                    if(endSideNumber > 0 && startSideNumber != endSideNumber)
                        endSideNumberTextOutFlg = true;
                }
                // 全ての配置位置を出力しない場合
                else
                {
                    // 元となる三角形に使われた線の頂点が一致するものだけ出力する
                    foreach(BaseTriangleInfo baseTriangleInfo in triangleInfoList)
                    {
                        // 始点側の番号
                        if(IsPointEquals(polylineStartPoint, baseTriangleInfo.vertexPoint1) || IsPointEquals(polylineStartPoint, baseTriangleInfo.vertexPoint2))
                        {
                            startSideNumberTextOutFlg = true;
                        }

                        // 終点側の番号が始点側と異なる場合
                        if(endSideNumber > 0 && startSideNumber != endSideNumber)
                        {
                            if(IsPointEquals(polylineEndPoint, baseTriangleInfo.vertexPoint1) || IsPointEquals(polylineEndPoint, baseTriangleInfo.vertexPoint2))
                            {
                                 endSideNumberTextOutFlg = true;
                            }
                        }

                        if(startSideNumberTextOutFlg || endSideNumberTextOutFlg)
                            break;
                    }
                }

                if(startSideNumberTextOutFlg)
                {
                    string outputText = startKakko + startSideNumber.ToString() + endKakko;
                    Point2d textLocatePoint = polylineStartPoint;

                    PositionNumberShape positionNumberShape = new PositionNumberShape();
                    positionNumberShape.Number = startSideNumber;

                    // 文字図形を作成してリストに追加
                    TextShape textShape = AddTextShape(positionNumberShape.Shapes, outputText, textLocatePoint, 0.0, textHight, 0.0, Alignment.BottomMiddle, customSettings.triclinicTextColorNumber, doc, currentLayer);

                    // 位置に点図形も
                    int positionPointMarkerColorNumber = customSettings.positionPointMarkerColorNumber;
                    // 色を文字と同じ場合
                    if(customSettings.positionPointMarkerColorNumber == -1)
                        positionPointMarkerColorNumber = textShape.ColorNumber;

                    AddPointMarkerShape(positionNumberShape.Shapes, textLocatePoint, customSettings.positionPointMarkerType, textHight * 0.2, positionPointMarkerColorNumber, doc, currentLayer);

                    positionShapeList.Add(positionNumberShape);
                }

                if(endSideNumberTextOutFlg)
                {
                    string outputText = startKakko + endSideNumber.ToString() + endKakko;
                    Point2d textLocatePoint = polylineEndPoint;

                    PositionNumberShape positionNumberShape = new PositionNumberShape();
                    positionNumberShape.Number = endSideNumber;

                    // 文字図形を作成してリストに追加
                    TextShape textShape = AddTextShape(positionNumberShape.Shapes, outputText, textLocatePoint, 0.0, textHight, 0.0, Alignment.BottomMiddle, customSettings.triclinicTextColorNumber, doc, currentLayer);

                    // 位置に点図形も
                    int positionPointMarkerColorNumber = customSettings.positionPointMarkerColorNumber;
                    // 色を文字と同じ場合
                    if(customSettings.positionPointMarkerColorNumber == -1)
                        positionPointMarkerColorNumber = textShape.ColorNumber;

                    AddPointMarkerShape(positionNumberShape.Shapes, textLocatePoint, customSettings.positionPointMarkerType, textHight * 0.2, positionPointMarkerColorNumber, doc, currentLayer);

                    positionShapeList.Add(positionNumberShape);
                }

                // 天空図に番号を描画するかフラグをセット
                baseLine.isDrawStartPointSidePositionNumberAtSkymap = startSideNumberTextOutFlg;
                baseLine.isDrawEndPointSidePositionNumberAtSkymap = endSideNumberTextOutFlg;
            }

            return positionShapeList;
        }

        /// <summary>
        /// 三斜三角形の元となる三角形のリストを作成
        /// </summary>
        /// <param name="orthographicProjectionShapes">正射影の線(ポリライン)リスト</param>
        /// <param name="centerPoint"></param>
        /// <param name="hemisphereRadius"></param>
        /// <returns></returns>
        private List<BaseTriangleInfo> CreateBaseTriangleInfoList(List<OrthographicProjectionShape> orthographicProjectionShapes, Point2d centerPoint, double hemisphereRadius)
        {
            List<OrthographicProjectionShape> orthographicProjectionShapes2 = orthographicProjectionShapes;

            // 正射影の線が第一象限と第四象限の両方にある場合はまたがらないように回転させて考えて、三角形を作成するときに戻す
            // (角度が小さい順に三角形の辺を求めていくので、0度をまたがっていいると順番が決めにくいため)
            double rotateAngleDeg = 0.0;
            if(IsPointIn14Area(orthographicProjectionShapes, centerPoint))
            {
                rotateAngleDeg = 180.0;
                orthographicProjectionShapes2 = RotateOrthographicProjectionShapes(orthographicProjectionShapes, centerPoint, rotateAngleDeg);
            }

            // 三角形の辺の線（天空図の中心から伸びる線）リスト
            List<TriangleLine> triangleLines = new List<TriangleLine>();

            // 正射影の線同士の交点から円の中心点とを結ぶ線をリストに追加する
            for(int i = 0; i < orthographicProjectionShapes2.Count; i++)
            {
                OrthographicProjectionShape orthographicProjectionShape1 = orthographicProjectionShapes2[i];

                // 正射影の線のID == 正射影の線元になった建物の線のインデックス
                int baseLineID1 = orthographicProjectionShape1.baseLineID;

                // 正射影の線連続線図形
                PolylineShape polylineShape1 = orthographicProjectionShape1.polylineShape;

                for(int j = 0; j < orthographicProjectionShapes2.Count; j++)
                {
                    if(i == j)
                        continue;

                    // 正射影の線連続線図形
                    PolylineShape polylineShape2 = orthographicProjectionShapes2[j].polylineShape;

                    IntersectionPoint[] intersectionPoints = polylineShape1.GetIntersectionPoint(polylineShape2);
                    for(int k = 0; k < intersectionPoints.Length; k++)
                    {
                        Point2d pointOnShape1Point = intersectionPoints[k].PointOnShape1.Point;
                        TriangleLine triangleBaseLine = new TriangleLine(baseLineID1, centerPoint, pointOnShape1Point);
                        triangleLines.Add(triangleBaseLine);
                    }
                }
            }

            // 中心角度の0.0 ～ 360.0 の順に並び替える
            triangleLines.Sort((a, b) => Math.Sign(a.lineAngleDeg - b.lineAngleDeg));

            // それぞれの線について全ての正射影の線と途中で交差することなく、一つの正射影の線上にのみ交点があるものだけを残す。
            // 建物の角（正射影の線の端点）から円の中心に向かう線で途中に他の正射影の線と交点があるものは除く。
            for(int i = triangleLines.Count - 1; i >= 0; i--)
            {
                // LineShape 作成
                double lineAngleDeg = triangleLines[i].lineAngleDeg;// 底辺の角度(水平角度)
                Point2d lineEndPoint = triangleLines[i].vertexPoint;
                LineShape triangleLineShape = app.ShapeFactory.CreateLine(centerPoint, lineEndPoint);

                bool removeLineFlg = false;

                // 全ての正射影の線との交点を調べる
                foreach(OrthographicProjectionShape orthographicProjectionShape in orthographicProjectionShapes2)
                {
                    PolylineShape polylineShape = orthographicProjectionShape.polylineShape;
                    IntersectionPoint[] intersectionPoints = triangleLineShape.GetIntersectionPoint(polylineShape);
                    if(intersectionPoints.Length == 0)
                        continue;

                    // 終点以外で交点がある線は除く
                    for(int j = 0; j < intersectionPoints.Length; j++)
                    {
                        Point2d pointOnShape1Point = intersectionPoints[j].PointOnShape1.Point;

                        if(!IsPointEquals(lineEndPoint, pointOnShape1Point))
                        {
                            removeLineFlg = true;
                            break;
                        }
                    }

                    if(removeLineFlg)
                        break;
                }

                if(removeLineFlg)
                {
                    triangleLines.RemoveAt(i);
                    continue;
                }
            }

            // メモ:----------------------------------------------------------------------------------------------------------------------------
            // 上の処理で残った建物の角（正射影の線の端点）から円の中心に向かう線は同じ角度の線が複数本の場合がある
            // 線にはどの正射影の線の端点から出る線かのIDを持たせてある。、1本目と2本目の線（共に天空図の中心を通る）は
            // 同じIDの正射影の線の端点で結ばれて三角形を成す。2本の場合はそのまま使うが、1本のところでは、
            // 一つはそのままだか、もう一つは円の中心から円周上までの線が最初に正射影の線（端点ではないところ）と交わる点までの線を使う
            // ---------------------------------------------------------------------------------------------------------------------------------
            // 例：triangleLines 内の線が以下の12個あった場合。
            // 配列の順番,線の角度,元の正射影の線のID
            // 1  104.21    6----(1-4が対)
            // 2  104.21    5   |
            // 3  113.63    7   |
            // 4  113.63    6----
            // 5  121.21    2----(5-8が対)
            // 6  121.21    1   |
            // 7  136.30    3--------(7-9が対)
            // 8  136.30    2---|   |
            // 9  153.86    3--------
            // 10 153.86    4
            // 11 155.78    9
            // 12 155.78   10
            // 3つの対ができ、中心点を頂点とした三角形が3つできる。
            // 対がなかった線については、線を延長させて他の正射影の線につなげて三角形を作る

            // 三角形リスト
            bool toFirstIndex = IsDifferent3AnglesLines(triangleLines);// 線を探すときに最初に戻って探すかどうかフラグ(違う角度の線が3本以上ある場合に探す)
            List<BaseTriangleInfo> triangleInfoList = new List<BaseTriangleInfo>();
            List<int> usedTriangleLinesIndexs = new List<int>();
            double lastSelectedLineAngleDeg = -360.0;// 最後に使った1本目の線の角度(-360.0:初期値)
            for(int i = 0; i < triangleLines.Count; i++)
            {
                // 既に対として使った線なら次の線に進む
                if(usedTriangleLinesIndexs.Contains(i))
                    continue;

                // 前の三角形の1本目の線の角度と同じなら次の線に進む
                if(lastSelectedLineAngleDeg >= 0.0 && IsEqualAngleDeg(lastSelectedLineAngleDeg, triangleLines[i].lineAngleDeg))
                    continue;

                int thisIndex = i;
                int selectedLineIndex = -1;
                int selectedOtherLineIndex = -1;
                Point2d vertexPoint1 = new Point2d(0.0, 0.0);
                Point2d vertexPoint2 = new Point2d(0.0, 0.0);
                int baseLineID = -1;// 頂点1と頂点2の間になる正射影の線のID

                // 三角形の2辺がリスト内の線で決まるか
                bool stat = GetFirstAngleLineIndex(triangleLines, thisIndex, toFirstIndex, out selectedLineIndex, out selectedOtherLineIndex);
                if(stat)
                {
                    vertexPoint1 = triangleLines[selectedLineIndex].vertexPoint;
                    vertexPoint2 = triangleLines[selectedOtherLineIndex].vertexPoint;
                    baseLineID = triangleLines[selectedLineIndex].baseLineID;

                    // 使ったインデックスを保管
                    usedTriangleLinesIndexs.Add(selectedLineIndex);
                    usedTriangleLinesIndexs.Add(selectedOtherLineIndex);

                    // 次のiの準備
                    int nextAngleLineIndex = GetNextChangeAngleFirstIndex(triangleLines, thisIndex, false);
                    // 次のi
                    if(selectedOtherLineIndex < selectedLineIndex)
                        i = triangleLines.Count - 1;// 最初の線と組んだのでここまで
                    else
                        i = nextAngleLineIndex - 1; // for文でカウントアップするので-1しておく
                }
                else
                {
                    // 三角形の1本目の候補の線のインデックスリスト
                    List<int> sameAngleLine1IndexList = GetSameAngleLineIndexList(triangleLines, thisIndex, toFirstIndex);
                    if(sameAngleLine1IndexList.Count > 0)
                    {
                        // 2本目と成り得る線
                        int nextAngleLineIndex = GetNextChangeAngleFirstIndex(triangleLines, thisIndex, toFirstIndex);
                        if(nextAngleLineIndex >= 0)
                        {
                            // 1本目と2本目の角度
                            double lineAngleDeg1 = triangleLines[thisIndex].lineAngleDeg;
                            double lineAngleDeg2 = triangleLines[nextAngleLineIndex].lineAngleDeg;

                            // 三角形の2本目の候補の線のインデックスリスト
                            List<int> sameAngleLine2IndexList = GetSameAngleLineIndexList(triangleLines, nextAngleLineIndex, toFirstIndex);

                            // baseLineIDリストの線と交点を持つか
                            Point2d crossPoint1;
                            Point2d crossPoint2;
                            // 2本目の線を延長させて交点を持つか
                            if(GetCrossPoint(centerPoint, hemisphereRadius, lineAngleDeg2, orthographicProjectionShapes2, triangleLines, sameAngleLine1IndexList, out crossPoint1, out selectedOtherLineIndex))
                            {
                                vertexPoint1 = triangleLines[selectedOtherLineIndex].vertexPoint;
                                vertexPoint2 = crossPoint1;
                                baseLineID = triangleLines[selectedOtherLineIndex].baseLineID;

                                // 使ったインデックスを保管
                                usedTriangleLinesIndexs.Add(selectedOtherLineIndex);
                                selectedLineIndex = sameAngleLine2IndexList.First();// sameAngleLine2IndexListの一番前
                                usedTriangleLinesIndexs.Add(selectedLineIndex);
                            }
                            // 1本目の線を延長して交点も持つか
                            else
                            {
                                if(GetCrossPoint(centerPoint, hemisphereRadius, lineAngleDeg1, orthographicProjectionShapes2, triangleLines, sameAngleLine2IndexList, out crossPoint2, out selectedOtherLineIndex))
                                {
                                    vertexPoint1 = crossPoint2;
                                    vertexPoint2 = triangleLines[selectedOtherLineIndex].vertexPoint;
                                    baseLineID = triangleLines[selectedOtherLineIndex].baseLineID;

                                    // 使ったインデックスを保管
                                    selectedLineIndex = sameAngleLine1IndexList.Last();// sameAngleLine1IndexListの一番後ろ
                                    usedTriangleLinesIndexs.Add(selectedLineIndex);
                                    usedTriangleLinesIndexs.Add(selectedOtherLineIndex);
                                }
                                // 1本目と2本目の両方を延長して交点を持つか
                                else
                                {
                                    int usedBaseLineID = -1;
                                    if(GetCrossPoint2(centerPoint, hemisphereRadius, orthographicProjectionShapes2, triangleLines, lineAngleDeg1, sameAngleLine1IndexList,
                                        lineAngleDeg2, sameAngleLine2IndexList, out crossPoint1, out crossPoint2, out usedBaseLineID))
                                    {
                                        vertexPoint1 = crossPoint1;
                                        vertexPoint2 = crossPoint2;

                                        baseLineID = usedBaseLineID;

                                        // 使ったインデックスを保管
                                        selectedLineIndex = sameAngleLine1IndexList.Last();     // sameAngleLine1IndexListの一番後ろ
                                        selectedOtherLineIndex = sameAngleLine2IndexList.Last();// sameAngleLine2IndexListの一番前
                                        usedTriangleLinesIndexs.Add(selectedLineIndex);
                                        usedTriangleLinesIndexs.Add(selectedOtherLineIndex);
                                    }
                                }
                            }
                        }
                    }
                }

                // 頂点1と頂点2の間になる正射影の線のIDがあった場合に三角形を作成
                if(baseLineID > 0)
                {
                    // 回転を戻す
                    if(rotateAngleDeg != 0.0)
                    {
                        vertexPoint1 = app.Geometry.RotatePoint(vertexPoint1, centerPoint, -rotateAngleDeg);
                        vertexPoint2 = app.Geometry.RotatePoint(vertexPoint2, centerPoint, -rotateAngleDeg);
                    }

                    // 三角形の作成
                    BaseTriangleInfo baseTriangleInfo = new BaseTriangleInfo(centerPoint, vertexPoint1, vertexPoint2, baseLineID);

                    // リストに追加
                    triangleInfoList.Add(baseTriangleInfo);

                    // 1本目の線の角度保持
                    lastSelectedLineAngleDeg = triangleLines[thisIndex].lineAngleDeg;
                }
                else
                {
                    lastSelectedLineAngleDeg = -360.0;// 使われた線なし
                }
            }

            return triangleInfoList;
        }

        /// <summary>
        /// 三斜求積図(三斜三角形リスト)作成
        /// </summary>
        /// <param name="triclinicMode"></param>
        /// <param name="triangleInfoList"></param>
        /// <param name="orthographicProjectionShapeList"></param>
        /// <param name="maxSeparateAngleDeg">最大分割角度</param>
        /// <param name="centerPoint"></param>
        /// <param name="hemisphereRadius"></param>
        /// <returns></returns>
        private List<TriclinicTriangle> CreateTriclinicTriangleList(TriclinicMode triclinicMode, List<BaseTriangleInfo> triangleInfoList, List<OrthographicProjectionShape> orthographicProjectionShapeList, double maxSeparateAngleDeg, Point2d centerPoint, double hemisphereRadius)
        {
            // 三斜三角形リスト
            List<TriclinicTriangle> triclinicTriangleList = new List<TriclinicTriangle>();

            // 多角形近似
            // 天空図の中心位置の三角形の角度が最大分割角度以上ならそれ未満になるように分割する。
            int totalTriclinicTriangleCounter = 0;
            int triangleInfoListCount = triangleInfoList.Count;
            for(int i = 0; i < triangleInfoListCount; i++)
            {
                BaseTriangleInfo baseTriangleInfo = triangleInfoList[i];
                int baseLineID = baseTriangleInfo.baseLineID;

                // 正射影の線のID == 正射影の線の元になった建物の線のインデックス
                OrthographicProjectionShape orthographicProjectionShape = orthographicProjectionShapeList.FirstOrDefault(o => o.baseLineID == baseLineID);
                // 正射影の線連続線図形
                Debug.Assert(orthographicProjectionShape != null);
                PolylineShape polylineShape = orthographicProjectionShape.polylineShape;

                Debug.Assert(IsPointEquals(centerPoint, baseTriangleInfo.centerPoint));
                Point2d baseTriangleVertexPoint1 = baseTriangleInfo.vertexPoint1;
                Point2d baseTriangleVertexPoint2 = baseTriangleInfo.vertexPoint2;

                // 測定点Cを中心とした2点間角度
                double angleDeg = app.Geometry.GetAngle(centerPoint, baseTriangleVertexPoint1, baseTriangleVertexPoint2);
                Debug.Assert(angleDeg <= 360.0);
                if(angleDeg > 180.0)
                    angleDeg = 360.0 - angleDeg;

                // 最大分割角度以内に分けたときの分割数(JCBA方式基準では「天空率の三射求積の分割角度は最大で10度」とある)
                int separateCount = (int)(angleDeg / maxSeparateAngleDeg);// (int):doubleの小数点以下は切り捨て
                if(angleDeg % maxSeparateAngleDeg > 0.0 || separateCount == 0)
                    separateCount++;

                Debug.Assert(separateCount > 0);
                double triclinicTriangleCenterAngleDeg = angleDeg / ((double)(separateCount));
                double bisectAngleDeg = triclinicTriangleCenterAngleDeg * 0.5;// 三斜三角形の中心点の位置の角の二等分角度
                double baseTriangleStartSideLineAngleDeg = app.Geometry.GetAngle(centerPoint, baseTriangleVertexPoint1);

                for(int j = 0; j < separateCount; j++)
                {
                    // 三斜三角形の中心点から天空図円方向に延びる1つ目の線の角度
                    double triclinicTriangleStartSideLineAngleDeg = baseTriangleStartSideLineAngleDeg + triclinicTriangleCenterAngleDeg * (double)j;
                    // 三斜三角形の中心点から天空図円方向に延びる2つ目の線の角度
                    double triclinicTriangleEndSideLineAngleDeg = triclinicTriangleStartSideLineAngleDeg + triclinicTriangleCenterAngleDeg;

                    // 三斜三角形の中心点から延長した線1(適合建物用のときは交点が円の外になる場合があるので半無限線で求める)
                    HalfInfiniteLineShape triangleHalfInfiniteLineShape1Ex = app.ShapeFactory.CreateHalfInfiniteLine(centerPoint, triclinicTriangleStartSideLineAngleDeg);

                    // 三斜三角形の中心点から延長した線2
                    HalfInfiniteLineShape triangleHalfInfiniteLineShape2Ex = app.ShapeFactory.CreateHalfInfiniteLine(centerPoint, triclinicTriangleEndSideLineAngleDeg);

                    Point2d triangleVertexPoint1 = new Point2d();
                    Point2d triangleVertexPoint2 = new Point2d();

                    // ●基準建物用（適合建物用）
                    // 外接近似　三斜の三角形は、天空図の中心点を頂点の1つとして、他の2点は建物の正射影の線との交点をの間にある正射影の線に対して外接する線との交点まで伸ばした点とする。
                    // そのため、正射影の面積（扇形の面積 － 三斜の面積）は小さめに計算され、天空率が大きくなります(外接三角形で近似し結果の小数は切り上げ)
                    if(triclinicMode == TriclinicMode.TriclinicMode_Standard)
                    {
                        // 三斜三角形の中心点の位置の角の二等分線(長さは天空図円半径)
                        double bisectLineAngleDeg = triclinicTriangleStartSideLineAngleDeg + bisectAngleDeg;
                        HalfInfiniteLineShape bisectHalfInfiniteLine = app.ShapeFactory.CreateHalfInfiniteLine(centerPoint, bisectLineAngleDeg);

                        // 正射影の線との交点
                        // 交点の位置でのpolylineShapeの角度と同じ角度の線がpolylineShapeに外接する線と考える
                        IntersectionPoint[] intersectionPoints = bisectHalfInfiniteLine.GetIntersectionPoint(polylineShape);
                        if(intersectionPoints.Length != 1)
                        {
                            if(j == 0)
                                break;
                            else
                                goto ERRLABEL;
                        }

                        Point2d bisectPointOnPolyline = intersectionPoints[0].PointOnShape2.Point;

                        // 交点のある位置の線分と同じ角度の無限線を作成
                        InfiniteLineShape tempInfiniteLineShape = null;
                        LineShape[] lines = polylineShape.GetSplitLineShapes();
                        for(int k = 0; k < lines.Length; k++)
                        {
                            if(lines[k].IsPointOnShape(bisectPointOnPolyline))
                            {
                                Point2d point1 = lines[k].StartPoint;
                                double lineAngleDeg = lines[k].Angle;
                                tempInfiniteLineShape = app.ShapeFactory.CreateInfiniteLine(point1, lineAngleDeg);
                                break;
                            }
                        }

                        // tempInfiniteLineShape と 線1,線2それぞれの交点が三斜三角形の頂点になる。もう一つは中心点
                        if(tempInfiniteLineShape == null)
                            goto ERRLABEL;

                        IntersectionPoint[] intersectionPoints1 = triangleHalfInfiniteLineShape1Ex.GetIntersectionPoint(tempInfiniteLineShape);
                        if(intersectionPoints1.Length != 1)
                            goto ERRLABEL;

                        triangleVertexPoint1 = intersectionPoints1[0].PointOnShape1.Point;

                        IntersectionPoint[] intersectionPoints2 = triangleHalfInfiniteLineShape2Ex.GetIntersectionPoint(tempInfiniteLineShape);
                        if(intersectionPoints2.Length != 1)
                            goto ERRLABEL;

                        triangleVertexPoint2 = intersectionPoints2[0].PointOnShape1.Point;
                    }
                    // ●計画建物用
                    // 内接近似　三斜の三角形は、天空図の中心点を頂点の1つとして、他の2点は建物の正射影の線との交点とする。
                    // 正射影の面積（扇形の面積 － 三斜の面積）が大きめに計算され、天空率は小さくなる。(内接三角形で近似し結果の小数は切り捨て)　(計画建物に有利にならないための安全策)
                    else if(triclinicMode == TriclinicMode.TriclinicMode_Plan)
                    {
                        IntersectionPoint[] intersectionPoints1 = triangleHalfInfiniteLineShape1Ex.GetIntersectionPoint(polylineShape);
                        if(intersectionPoints1.Length != 1)
                        {
                            if(j == 0)
                                break;
                            else
                                goto ERRLABEL;
                        }

                        triangleVertexPoint1 = intersectionPoints1[0].PointOnShape1.Point;

                        IntersectionPoint[] intersectionPoints2 = triangleHalfInfiniteLineShape2Ex.GetIntersectionPoint(polylineShape);
                        if(intersectionPoints2.Length != 1)
                        {
                            if(j == 0)
                                break;
                            else
                                goto ERRLABEL;
                        }

                        triangleVertexPoint2 = intersectionPoints2[0].PointOnShape1.Point;
                    }

                    // 三斜三角形作成
                    LineShape triangleLineShape1 = app.ShapeFactory.CreateLine(centerPoint, triangleVertexPoint1);// 中心点が始点になるように作る
                    LineShape triangleLineShape2 = app.ShapeFactory.CreateLine(centerPoint, triangleVertexPoint2);// 中心点が始点になるように作る
                    totalTriclinicTriangleCounter++;
                    TriclinicTriangle triclinicTriangle = new TriclinicTriangle(totalTriclinicTriangleCounter, triangleLineShape1, triangleLineShape2, triclinicMode, addInSettings);// 各線の始点は同じで天空図の中心点になる
                    triclinicTriangleList.Add(triclinicTriangle);
                }
            }

            return triclinicTriangleList;

        // エラーの場合
        ERRLABEL:
            triclinicTriangleList.Clear();// リストをクリア
            return triclinicTriangleList;
        }

        /// <summary>
        /// 三角形描画用図形作成（チェック用）
        /// </summary>
        /// <param name="triangleInfoList"></param>
        /// <param name="hemisphereRadius"></param>
        /// <param name="isCenterPointInside"></param>
        /// <param name="doc"></param>
        /// <param name="currentLayer"></param>
        /// <returns></returns>
        private List<Shape> CreateTriangleInfoListShapeList(List<BaseTriangleInfo> triangleInfoList, double hemisphereRadius, bool isCenterPointInside, Document doc, Layer currentLayer)
        {
            int triangleColorNumber = 6;
            int triangleLinewidthNumber = 5;

            int triangleInfoListCount = triangleInfoList.Count;
            List<Shape> triangleInfoListArrayList = new List<Shape>();
            if(triangleInfoListCount == 0)
                return triangleInfoListArrayList;

            // 三角形描画のための線図形の作成
            Point2d centerPoint = triangleInfoList[0].centerPoint;// 中心点（他も同じなので先頭のものを使う）
            for(int i = 0; i < triangleInfoListCount; i++)
            {
                BaseTriangleInfo triangleInfo = triangleInfoList[i];

                LineShape[] lineShape = new LineShape[3];
                // 三角形の各辺の線
                lineShape[0] = app.ShapeFactory.CreateLine(triangleInfo.centerPoint, triangleInfo.vertexPoint1);
                lineShape[1] = app.ShapeFactory.CreateLine(triangleInfo.vertexPoint1, triangleInfo.vertexPoint2);
                lineShape[2] = app.ShapeFactory.CreateLine(triangleInfo.vertexPoint2, triangleInfo.centerPoint);
                
                for(int j = 0; j < 3; j++)
                {
                    doc.BasicShapeSettings.CopyToShape(lineShape[j]);

                    // レイヤのセット
                    lineShape[j].Layer = currentLayer;

                    // 色
                    lineShape[j].ColorNumber = triangleColorNumber;

                    // 線幅
                    lineShape[j].LinewidthNumber = triangleLinewidthNumber;

                    // 追加
                    triangleInfoListArrayList.Add(lineShape[j]);
                }
            }

            return triangleInfoListArrayList;
        }

        /// <summary>
        /// 三斜求積図(三斜三角形)描画図形リスト作成
        /// </summary>
        /// <param name="triclinicTriangleList"></param>
        /// <param name="triclinicMode"></param>
        /// <param name="hemisphereRadius"></param>
        /// <param name="isCenterPointInside"></param>
        /// <param name="doc"></param>
        /// <param name="currentLayer"></param>
        /// <param name="textHight"></param>
        /// <returns></returns>
        private List<Shape> CreateTriclinicTriangleShapeList(List<TriclinicTriangle> triclinicTriangleList, TriclinicMode triclinicMode, double hemisphereRadius, bool isCenterPointInside, Document doc, Layer currentLayer, double textHight)
        {
            //int triangleColorNumber = customSettings.triangleColorNumber;
            //int triangleHightColorNumber = customSettings.triangleHightColorNumber;

            List<Shape> triclinicTriangleShapeList = new List<Shape>();
            if(triclinicTriangleList.Count == 0)
                return triclinicTriangleShapeList;

            List<double> triangleLineAngleDegs = new List<double>();// 三斜三角形の辺の角度配列
            Point2d centerPoint = triclinicTriangleList[0].centerPoint;// 中心点（他も同じなので先頭のものを使う）

            // 三斜三角形描画のための線図形の作成
            int triclinicTriangleListCount = triclinicTriangleList.Count;
            for(int i = 0; i < triclinicTriangleListCount; i++)
            {
                TriclinicTriangle triclinicTriangle = triclinicTriangleList[i];

                triangleLineAngleDegs.Add(triclinicTriangle.baselineAngleDeg);
                triangleLineAngleDegs.Add(triclinicTriangle.otherlineAngleDeg);

                LineShape[] lineShape = new LineShape[4];
                // 三角形の各辺の線
                lineShape[0] = app.ShapeFactory.CreateLine(triclinicTriangle.centerPoint, triclinicTriangle.hightlineTopPoint);
                lineShape[1] = app.ShapeFactory.CreateLine(triclinicTriangle.hightlineTopPoint, triclinicTriangle.baselinePoint);
                lineShape[2] = app.ShapeFactory.CreateLine(triclinicTriangle.baselinePoint, triclinicTriangle.centerPoint);
                // 高さを表す線
                lineShape[3] = app.ShapeFactory.CreateLine(triclinicTriangle.hightlineTopPoint, triclinicTriangle.hightlinePointOnBaseline);

                for(int j = 0; j< 4; j++)
                {
                    doc.BasicShapeSettings.CopyToShape(lineShape[j]);

                    // レイヤのセット
                    lineShape[j].Layer = currentLayer;

                    // 色・線幅
                    if(j == 3)
                    {
                        if(customSettings.triangleHightColorNumber >= 0)
                            lineShape[j].ColorNumber = customSettings.triangleHightColorNumber;

                        if(customSettings.triangleHightLinewidthNumber >= 0)
                            lineShape[j].LinewidthNumber = customSettings.triangleHightLinewidthNumber;

                    }
                    else
                    {
                        if(customSettings.triangleColorNumber >= 0)
                            lineShape[j].ColorNumber = customSettings.triangleColorNumber;

                        if(customSettings.triangleLinewidthNumber >= 0)
                            lineShape[j].LinewidthNumber = customSettings.triangleLinewidthNumber;
                    }

                    // 線種
                    lineShape[j].LinetypeNumber = 1;

                    // 追加
                    triclinicTriangleShapeList.Add(lineShape[j]);
                }
            }

            // 三斜三角形の最初と最後の線も追加
            if(!isCenterPointInside)
            {
                double minAngleDeg = triangleLineAngleDegs.Min();
                double maxAngleDeg = triangleLineAngleDegs.Max();

                // 三斜三角形の辺の角度が第一象限と第四象限の両方にある場合は180度反転して最初と最後の線を求める
                if(minAngleDeg < 90.0 && maxAngleDeg >= 270.0)
                {
                    var triangleLineAngleDegs2 = triangleLineAngleDegs.Select(x => x + (x < 180.0 ? 180.0 : -180.0));

                    minAngleDeg = triangleLineAngleDegs2.Min();
                    maxAngleDeg = triangleLineAngleDegs2.Max();

                    // 戻す
                    minAngleDeg += minAngleDeg >= 180.0 ? -180.0 : 180.0;
                    maxAngleDeg += maxAngleDeg >= 180.0 ? -180.0 : 180.0;
                }

                // 最初と最後の線の図形を作成
                Point2d line0EndPoint = new Point2d(centerPoint.X + hemisphereRadius, centerPoint.Y);
                Point2d minAngleDegLinePoint = app.Geometry.RotatePoint(line0EndPoint, centerPoint, minAngleDeg);
                Point2d maxnAngleDegLinePoint = app.Geometry.RotatePoint(line0EndPoint, centerPoint, maxAngleDeg);
                LineShape minAngleDegLineShape = app.ShapeFactory.CreateLine(centerPoint, minAngleDegLinePoint);
                LineShape maxAngleDegLineShape = app.ShapeFactory.CreateLine(centerPoint, maxnAngleDegLinePoint);

                // 設定のコピー
                doc.BasicShapeSettings.CopyToShape(minAngleDegLineShape);
                doc.BasicShapeSettings.CopyToShape(maxAngleDegLineShape);

                // レイヤのセット
                minAngleDegLineShape.Layer = currentLayer;
                maxAngleDegLineShape.Layer = currentLayer;

                // 色
                if(customSettings.triangleColorNumber >= 0)
                {
                    minAngleDegLineShape.ColorNumber = customSettings.triangleColorNumber;
                    maxAngleDegLineShape.ColorNumber = customSettings.triangleColorNumber;
                }

                // 線種
                minAngleDegLineShape.LinetypeNumber = 1;
                maxAngleDegLineShape.LinetypeNumber = 1;

                // 線幅
                if(customSettings.triangleLinewidthNumber >= 0)
                {
                    minAngleDegLineShape.LinewidthNumber = customSettings.triangleLinewidthNumber;
                    maxAngleDegLineShape.LinewidthNumber = customSettings.triangleLinewidthNumber;
                }

                // 追加
                triclinicTriangleShapeList.Add(minAngleDegLineShape);
                triclinicTriangleShapeList.Add(maxAngleDegLineShape);
            }

            // 三斜三角形の番号描画
            string startKakko = "";
            string endKakko = "";
            if(triclinicMode == TriclinicMode.TriclinicMode_Standard)
            {
                startKakko = customSettings.standard_TriangleNumberKakkoStart;
                endKakko = customSettings.standard_TriangleNumberKakkoEnd;
            }
            else if(triclinicMode == TriclinicMode.TriclinicMode_Plan)
            {
                startKakko = customSettings.plan_TriangleNumberKakkoStart;
                endKakko = customSettings.plan_TriangleNumberKakkoEnd;
            }

            foreach(TriclinicTriangle triclinicTriangle in triclinicTriangleList)
            {
                int triclinicTriangleNumber = triclinicTriangle.triangleNumber;
                string outputText = startKakko + triclinicTriangleNumber + endKakko;

                // 天空図の中心位置で作成して後で整える
                TextShape textShape = app.ShapeFactory.CreateText(outputText, centerPoint, 0.0);
                doc.TextSettings.CopyToShape(textShape);

                textShape.DirectionVertical = false;// 横書き
                //double baselineLength = triclinicTriangle.baselineLength;

                // 三角形の中心点以外の2点を結ぶ線の中央を通る線の角度と中心からそこまでx%の位置に配置する
                Point2d baselinePoint = triclinicTriangle.baselinePoint;
                Point2d hightlineTopPoint = triclinicTriangle.hightlineTopPoint;
                Point2d betweenPoint = app.Geometry.GetMiddlePoint(baselinePoint, hightlineTopPoint);
                double positionAngleDeg = Get2PointsAngle(centerPoint, betweenPoint);
                double postDistance = Get2PointsDistance(centerPoint, betweenPoint) * 0.85;

                // 配置点
                Point2d textLocatePoint = new Point2d(centerPoint.X + postDistance, centerPoint.Y);
                textLocatePoint = app.Geometry.RotatePoint(textLocatePoint, centerPoint, positionAngleDeg); // 回転移動
                textShape.Point = textLocatePoint;

                textShape.FontHeight = textHight; // 文字高さセット
                textShape.Alignment = Alignment.MiddleMiddle;// 配置点：中中

                // レイヤのセット
                textShape.Layer = currentLayer;

                // 色
                if(customSettings.triclinicTextColorNumber >= 0)
                    textShape.ColorNumber = customSettings.triclinicTextColorNumber;

                triclinicTriangleShapeList.Add(textShape);
            }

            return triclinicTriangleShapeList;
        }

        /// <summary>
        /// 正射影の線が第一象限と第四象限の両方にあるか
        /// （端点だけで見ている）
        /// </summary>
        /// <param name="orthographicProjectionShapes"></param>
        /// <param name="centerPoint"></param>
        /// <returns></returns>
        private bool IsPointIn14Area(List<OrthographicProjectionShape> orthographicProjectionShapes, Point2d centerPoint)
        {
            bool isPointIn1Area = false;// 第一象限に点が存在フラグ
            bool isPointIn4Area = false;// 第四象限に点が存在フラグ

            foreach(OrthographicProjectionShape orthographicProjectionShape in orthographicProjectionShapes)
            {
                // 正射影の線連続線図形
                PolylineShape polylineShape = orthographicProjectionShape.polylineShape;

                // 交点があった場合
                // 連続線で点列が複数あるが始点と終点をそれぞれ中心点と成す角度の大きいほうを探す
                Point2d[] polylinePoints = ((PolylineShape)polylineShape).Points;
                Debug.Assert(polylinePoints.Length > 1);
                Point2d polylineStartPoint = polylinePoints.First();
                Point2d polylineEndPoint = polylinePoints.Last();

                double angleDeg1 = Get2PointsAngle(centerPoint, polylineStartPoint);
                double angleDeg2 = Get2PointsAngle(centerPoint, polylineEndPoint);

                // 第一象限に点が存在
                if(angleDeg1 < 90.0 || angleDeg2 < 90.0)
                    isPointIn1Area = true;

                // 第四象限に点が存在
                if(angleDeg1 >= 270.0 || angleDeg2 >= 270.0)
                    isPointIn4Area = true;

                if(isPointIn1Area && isPointIn4Area)
                    return true;
            }

            return false;
        }

        /// <summary>
        /// 正射影の線リストを回転
        /// </summary>
        /// <param name="orthographicProjectionShapes"></param>
        /// <param name="centerPoint"></param>
        /// <param name="rotateAngleDeg"></param>
        /// <returns></returns>
        private List<OrthographicProjectionShape> RotateOrthographicProjectionShapes(List<OrthographicProjectionShape> orthographicProjectionShapes, Point2d centerPoint, double rotateAngleDeg)
        {
            // 新しい正射影の線のリスト
            List<OrthographicProjectionShape> newOrthographicProjectionShapes = new List<OrthographicProjectionShape>();

            foreach(OrthographicProjectionShape orthographicProjectionShape in orthographicProjectionShapes)
            {
                // 正射影の線連続線図形
                int baseLineID = orthographicProjectionShape.baseLineID;
                PolylineShape polylineShape = orthographicProjectionShape.polylineShape;

                // 回転させた新しい連続線
                PolylineShape newPolylineShape = (PolylineShape)polylineShape.Clone();
                newPolylineShape.Transform(centerPoint, 1.0, 1.0, rotateAngleDeg, centerPoint);
                OrthographicProjectionShape newOrthographicProjectionShape = new OrthographicProjectionShape(baseLineID, newPolylineShape);

                // リストに追加
                newOrthographicProjectionShapes.Add(newOrthographicProjectionShape);
            }
             
            return newOrthographicProjectionShapes;
        }


        /// <summary>
        /// 指定した中心点を通る線が正射影の線と交差するときの交点取得
        /// 交点を持つ線ともう一方の三角形の辺になる線のインデックスも取得
        /// </summary>
        /// <param name="centerPoint"></param>
        /// <param name="radius"></param>
        /// <param name="lineAngleDeg"></param>
        /// <param name="orthographicProjectionShapes"></param>
        /// <param name="triangleLines"></param>
        /// <param name="otherLineIndexList"></param>
        /// <param name="crossPoint"></param>
        /// <param name="selectedOtherLineIndex"></param>
        /// <returns></returns>
        private bool GetCrossPoint(Point2d centerPoint, double radius, double lineAngleDeg, List<OrthographicProjectionShape> orthographicProjectionShapes, 
            List<TriangleLine> triangleLines, List<int> otherLineIndexList, out Point2d crossPoint, out int selectedOtherLineIndex)
        {
            crossPoint = new Point2d(0.0, 0.0);
            selectedOtherLineIndex = -1;

            Point2d pointOnCircle = new Point2d(centerPoint.X + radius, centerPoint.Y);
            pointOnCircle = app.Geometry.RotatePoint(pointOnCircle, centerPoint, lineAngleDeg);// 回転移動
            LineShape lineShape = app.ShapeFactory.CreateLine(centerPoint, pointOnCircle);// 延長した線

            // 候補となる同じ角度の線に関連づいているIDの正射影の線との交点で、最も測定点（中心点）に近い点を求める
            List<IntersectionPointInfo2> intersectionPointInfo2List = new List<IntersectionPointInfo2>();
            for(int i = 0; i < otherLineIndexList.Count; i++)
            {
                int lineIndex = otherLineIndexList[i];
                int baseLineID = triangleLines[lineIndex].baseLineID;

                // 正射影の線のID == 正射影の線の元になった建物の線のインデックス
                OrthographicProjectionShape orthographicProjectionShape = orthographicProjectionShapes.FirstOrDefault(o => o.baseLineID == baseLineID);
                // 正射影の線連続線図形
                if(orthographicProjectionShape != null)
                {
                    PolylineShape polylineShape = orthographicProjectionShape.polylineShape;
                    IntersectionPoint[] intersectionPoints = lineShape.GetIntersectionPoint(polylineShape);
                    if(intersectionPoints.Length > 0)
                    {
                        IntersectionPointInfo2 intersectionPointInfo2 = new IntersectionPointInfo2(intersectionPoints[0].PointOnShape1.Parameter, lineIndex);// 交点は1つとして
                        intersectionPointInfo2List.Add(intersectionPointInfo2);
                    }
                }
            }

            // 交点があった場合
            if(intersectionPointInfo2List.Count > 0)
            {
                // 中心に最も近い交点を返す
                // パラメータの小さい順に並び替え
                intersectionPointInfo2List.Sort((a, b) => Math.Sign(a.parameter - b.parameter));

                double parameter = intersectionPointInfo2List[0].parameter;
                crossPoint = lineShape.GetPointAtParameter(parameter);
                selectedOtherLineIndex = intersectionPointInfo2List[0].index;

                return true;
            }

            return false;
        }


        /// <summary>
        /// 1本目として指定した角度の線リストの中でbaseLineIDが一つだけ（他に同じbaseLineIDを持ったものがない）ものと
        /// 2本目として指定した角度の線リストの中でbaseLineIDが一つだけ（他に同じbaseLineIDを持ったものがない）ものが
        /// 同じ正射影の線と交点を持つ場合、それぞれの交点を返す
        /// </summary>
        /// <param name="centerPoint"></param>
        /// <param name="radius"></param>
        /// <param name="orthographicProjectionShapes"></param>
        /// <param name="triangleLines"></param>
        /// <param name="lineAngleDeg1"></param>
        /// <param name="sameAngleLine1IndexList"></param>
        /// <param name="lineAngleDeg2"></param>
        /// <param name="sameAngleLine2IndexList"></param>
        /// <param name="crossPoint1"></param>
        /// <param name="crossPoint2"></param>
        /// <param name="usedBaseLineID"></param>
        /// <returns></returns>
        private bool GetCrossPoint2(Point2d centerPoint, double radius, List<OrthographicProjectionShape> orthographicProjectionShapes, List<TriangleLine> triangleLines,
                                    double lineAngleDeg1, List<int> sameAngleLine1IndexList, double lineAngleDeg2, List<int> sameAngleLine2IndexList,
                                    out Point2d crossPoint1, out Point2d crossPoint2, out int usedBaseLineID)
        {
            crossPoint1 = new Point2d(0.0, 0.0);
            crossPoint2 = new Point2d(0.0, 0.0);
            usedBaseLineID = -1;

            Point2d pointOnCircle1 = new Point2d(centerPoint.X + radius, centerPoint.Y);
            pointOnCircle1 = app.Geometry.RotatePoint(pointOnCircle1, centerPoint, lineAngleDeg1);
            LineShape lineShape1 = app.ShapeFactory.CreateLine(centerPoint, pointOnCircle1);

            Point2d pointOnCircle2 = new Point2d(centerPoint.X + radius, centerPoint.Y);
            pointOnCircle2 = app.Geometry.RotatePoint(pointOnCircle2, centerPoint, lineAngleDeg2);
            LineShape lineShape2 = app.ShapeFactory.CreateLine(centerPoint, pointOnCircle2);

            List<IntersectionPointInfo3> intersectionPointInfo3List1 = GetIntersectionPointInfo3List(lineShape1, orthographicProjectionShapes, triangleLines, lineAngleDeg1, sameAngleLine1IndexList);
            if(intersectionPointInfo3List1.Count == 0)
                return false;

            List<IntersectionPointInfo3> intersectionPointInfo3List2 = GetIntersectionPointInfo3List(lineShape2, orthographicProjectionShapes, triangleLines, lineAngleDeg2, sameAngleLine2IndexList);
            if(intersectionPointInfo3List2.Count == 0)
                return false;

            for(int i = 0; i < intersectionPointInfo3List1.Count; i++)
            {
                int baseLineID1 = intersectionPointInfo3List1[i].baseLineID;

                // baseLineID1と同じIDのを持つ交点情報を取得(Linq)
                IntersectionPointInfo3 IntersectionPointInfo3_2 = intersectionPointInfo3List2.FirstOrDefault(o => o.baseLineID == baseLineID1);
                if(IntersectionPointInfo3_2 == null)
                    continue;

                double parameter1 = intersectionPointInfo3List1[i].parameter;
                crossPoint1 = lineShape1.GetPointAtParameter(parameter1);

                double parameter2 = IntersectionPointInfo3_2.parameter;
                crossPoint2 = lineShape2.GetPointAtParameter(parameter2);

                usedBaseLineID = baseLineID1;

                return true;
            }

            return false;
        }

        private List<IntersectionPointInfo3> GetIntersectionPointInfo3List(LineShape lineShape, List<OrthographicProjectionShape> orthographicProjectionShapes,
                                                                                List<TriangleLine> triangleLines, double lineAngleDeg, List<int> sameAngleLineIndexList)
        {
            List<IntersectionPointInfo3> intersectionPointInfo3List = new List<IntersectionPointInfo3>();

            for(int i = 0; i < sameAngleLineIndexList.Count; i++)
            {
                int lineIndex = sameAngleLineIndexList[i];
                int baseLineID = triangleLines[lineIndex].baseLineID;

                // baseLineIDと同じIDの線の数取得(Linq)
                int sameIDCount = triangleLines.Count(o => o.baseLineID == baseLineID);
                if(sameIDCount == 1)
                {
                    int orthographicProjectionShapesCount = orthographicProjectionShapes.Count();
                    for(int j = 0; j < orthographicProjectionShapesCount; j++)
                    {
                        PolylineShape polylineShape = orthographicProjectionShapes[j].polylineShape;
                        IntersectionPoint[] intersectionPoints = lineShape.GetIntersectionPoint(polylineShape);
                        if(intersectionPoints.Length > 0)
                        {
                            int orthographicProjectionShapeBaseLineID = orthographicProjectionShapes[j].baseLineID;
                            IntersectionPointInfo3 intersectionPointInfo3 = new IntersectionPointInfo3(intersectionPoints[0].PointOnShape1.Parameter, lineIndex, orthographicProjectionShapeBaseLineID);// 交点は1つとして
                            intersectionPointInfo3List.Add(intersectionPointInfo3);
                        }
                    }
                }
            }

            // 交点があった場合
            if(intersectionPointInfo3List.Count > 0)
            {
                // 中心に最も近い交点を返す
                // パラメータの小さい順に並び替え
                intersectionPointInfo3List.Sort((a, b) => Math.Sign(a.parameter - b.parameter));
            }

            return intersectionPointInfo3List;
        }


        /// <summary>
        /// 三角形の1本目の辺となる角度線のインデックス取得
        /// </summary>
        /// <param name="triangleLines"></param>
        /// <param name="thisIndex">最初にその線に成り得る線のインデックス</param>
        /// <param name="toFirstIndex">初めからも探すフラグ</param>
        /// <param name="lineIndex1">1本目の線のインデックス</param>
        /// <param name="lineIndex2">2本目の線のインデックス</param>
        /// <returns></returns>
        private bool GetFirstAngleLineIndex(List<TriangleLine> triangleLines, int thisIndex, bool toFirstIndex, out int lineIndex1, out int lineIndex2)
        {
            lineIndex1 = -1;
            lineIndex2 = -1;
            
            Debug.Assert(thisIndex < triangleLines.Count);

            // 1本目の辺となる角度の線のリスト
            List<int> sameAngleLineIndexList1 = GetSameAngleLineIndexList(triangleLines, thisIndex, toFirstIndex);

            // 2本目の辺となる角度の最初の線のインデックス
            int nextChangeAngleFirstIndex = GetNextChangeAngleFirstIndex(triangleLines, thisIndex, toFirstIndex);
            if(nextChangeAngleFirstIndex < 0)
                return false;

            // 2本目の辺となる角度の線のリスト
            List<int> sameAngleLineIndexList2 = GetSameAngleLineIndexList(triangleLines, nextChangeAngleFirstIndex, toFirstIndex);

            // 1本目に成り得る角度の線のbaseLineIDと2本目に成り得る角度のbaseLineID見て一致すものがあったとき、1本目の線のインデックスを返す
            for(int i = 0; i < sameAngleLineIndexList1.Count; i++)
            {
                lineIndex1 = sameAngleLineIndexList1[i];
                int baseLineID1 = triangleLines[lineIndex1].baseLineID;

                for(int j = 0; j < sameAngleLineIndexList2.Count; j++)
                {
                    lineIndex2 = sameAngleLineIndexList2[j];
                    int baseLineID2 = triangleLines[lineIndex2].baseLineID;

                    // 同じbaseLineIDを持っている
                    if(baseLineID1 == baseLineID2)
                        return true;
                }
            }

            return false;
        }

        /// <summary>
        /// 指定したインデックスの角度と同じ角度の線のインデックスリスト取得
        /// </summary>
        /// <param name="triangleLines"></param>
        /// <param name="thisIndex"></param>
        /// <param name="thisIndexIncludeFlg">指定したインデックスも含めるかどうか</param>
        /// <returns></returns>
        private List<int> GetSameAngleLineIndexList(List<TriangleLine> triangleLines, int thisIndex, bool thisIndexIncludeFlg)
        {
            Debug.Assert(thisIndex < triangleLines.Count);

            double lineAngleDeg1 = triangleLines[thisIndex].lineAngleDeg;

            List<int> sameAngleLineIndexList = new List<int>();

            for(int i = 0; i < triangleLines.Count; i++)
            {
                if(!thisIndexIncludeFlg && i == thisIndex)
                    continue;

                double lineAngleDeg2 = triangleLines[i].lineAngleDeg;
                if(IsEqualAngleDeg(lineAngleDeg1, lineAngleDeg2))
                    sameAngleLineIndexList.Add(i);
            }

            return sameAngleLineIndexList;
        }

        /// <summary>
        /// 指定したインデックスの線の角度と次に異なる角度になる最初の線のインデックスを取得
        /// </summary>
        /// <param name="triangleLines"></param>
        /// <param name="thisIndex"></param>
        /// <param name="toFirstIndex">初めからも探すフラグ</param>
        /// <returns></returns>
        private int GetNextChangeAngleFirstIndex(List<TriangleLine> triangleLines, int thisIndex, bool toFirstIndex)
        {
            int triangleLinesCount = triangleLines.Count;
            Debug.Assert(thisIndex < triangleLinesCount);
            double lineAngleDeg1 = triangleLines[thisIndex].lineAngleDeg;

            for(int i = thisIndex + 1; i < triangleLinesCount; i++)
            {
                double lineAngleDeg2 = triangleLines[i].lineAngleDeg;
                if(!IsEqualAngleDeg(lineAngleDeg1, lineAngleDeg2))
                    return i;
            }

            if(toFirstIndex)
            {
                for(int i = 0; i < thisIndex; i++)
                {
                    double lineAngleDeg2 = triangleLines[i].lineAngleDeg;
                    if(!IsEqualAngleDeg(lineAngleDeg1, lineAngleDeg2))
                        return i;
                }
            }

            return -1;
        }

        /// <summary>
        /// 指定したリスト中の線が3種類以上の角度を持っているか
        /// (指定するリスト中の線の角度は昇順になっていること)
        /// </summary>
        /// <param name="triangleLines"></param>
        /// <returns></returns>
        private bool IsDifferent3AnglesLines(List<TriangleLine> triangleLines)
        {
            int triangleLinesCount = triangleLines.Count;
            if(triangleLinesCount <= 2)
                return false;

            int differentCounter = 1;
            double oldLineAngleDeg = triangleLines[0].lineAngleDeg;

            for(int i = 1; i < triangleLinesCount; i++)
            {
                double lineAngleDeg = triangleLines[i].lineAngleDeg;
                if(!IsEqualAngleDeg(oldLineAngleDeg, lineAngleDeg))
                    differentCounter++;

                if(differentCounter >= 3)
                    return true;

                oldLineAngleDeg = lineAngleDeg;
            }

            return false;
        }

        /// <summary>
        /// 回転＆鏡像移動
        /// </summary>
        /// <param name="ewsnShapes"></param>
        /// <param name="centerPoint"></param>
        /// <param name="rotateAngle"></param>
        /// <param name="mirrorLineAngle"></param>
        public void RotateAndMirrorShapes(List<Shape> ewsnShapes, Point2d centerPoint, double rotateAngle, double mirrorLineAngle)
        {
            Point2d mirrorPoint1 = centerPoint;
            Point2d mirrorPoint2 = new Point2d(centerPoint.X + 100.0, centerPoint.Y);// 長さ100の線を境界に鏡像移動
            mirrorPoint2 = app.Geometry.RotatePoint(mirrorPoint2, centerPoint, mirrorLineAngle);

            foreach(Shape shape in ewsnShapes)
            {
                // 回転
                shape.Rotate(centerPoint, rotateAngle);

                // 鏡像移動
                shape.Mirror(mirrorPoint1, mirrorPoint2);
            }
        }

        /// <summary>
        /// 枠の図形作成
        /// </summary>
        /// <param name="drawing"></param>
        /// <param name="frameLTPoint">枠の左上角座標</param>
        /// <param name="circleRadius"></param>
        /// <param name="cellWidth"></param>
        /// <param name="frameType"> 1:1段, 2:左[]付1段, 3:2段, 4:左[]付2段</param>
        /// <param name="locateSize"></param>
        /// <param name="circleCenterPoint">最初の天空図円を配置する中心点</param>
        /// <returns></returns>
        private List<Shape> CreateFrameShapes(Drawing drawing, Point2d frameLTPoint, double circleRadius, double cellWidth, int frameType, int locateSize, out Point2d circleCenterPoint)
        {
            Debug.Assert(frameType >= 1 && frameType <= 4);

            List<Shape> shapes = new List<Shape>();

            // セルの高さ
            double cellHeight = circleRadius * 2.0 * 1.65;

            // ヘッドセルの高さ（2段の場合）
            double headCellHeight = 0.0;
            // frameType == 3:2段、4:左[] 付2段の場合
            bool isHeadCell = false;
            if(frameType == 3 || frameType == 4)
            {
                headCellHeight = cellHeight * 0.15;
                isHeadCell = true;
            }

            // 左側にタイトルのセルの幅
            double titleCellWidth = 0.0;
            // frameType == 2:左[] 付1段、4:左[] 付2段の場合
            bool isTitleCell = false;
            if(frameType == 2 || frameType == 4)
            {
                titleCellWidth = cellHeight * 0.15;
                isTitleCell = true;
            }

            // 上の枠線と天空図円の間
            double topMargin = circleRadius * 2.0 * 0.2;

            // 枠の配置点(左上角)から1つ目の天空図円の中心点までの相対距離
            double relativeX = titleCellWidth + cellWidth * 0.5;
            double relativeY = -(headCellHeight + topMargin + circleRadius);

            // 1つ目の天空図円の中心点座標
            circleCenterPoint = new Point2d(frameLTPoint.X + relativeX, frameLTPoint.Y + relativeY);

            Document doc = drawing.Document;
            Layer currentLayer = doc.LayerTable.CurrentLayer;

            // 横線
            double lineStartPointX = frameLTPoint.X;
            double lineEndPointX = lineStartPointX + titleCellWidth + cellWidth * (double)locateSize;
            int hLineSize = isHeadCell ? 3 : 2;
            for(int i = 0; i < hLineSize; i++)
            {
                double linePointY = frameLTPoint.Y - (double)i * cellHeight;

                // ヘッドセルがある場合
                if(isHeadCell && i >= 1)
                    linePointY += (cellHeight - headCellHeight);

                Point2d lineStartPoint = new Point2d(lineStartPointX, linePointY);
                Point2d lineEndPoint = new Point2d(lineEndPointX, linePointY);

                LineShape lineShape = app.ShapeFactory.CreateLine(lineStartPoint, lineEndPoint);
                // 既定値の設定セット
                doc.BasicShapeSettings.CopyToShape(lineShape);

                // レイヤのセット
                lineShape.Layer = currentLayer;

                // 色
                if(customSettings.frameLineColorNumber >= 0)
                    lineShape.ColorNumber = customSettings.frameLineColorNumber;

                // 線種
                lineShape.LinetypeNumber = 1;

                // 線幅
                if(customSettings.frameLinewidthNumber >= 0)
                    lineShape.LinewidthNumber = customSettings.frameLinewidthNumber;

                // 追加
                shapes.Add(lineShape);
            }

            // 縦線
            double lineStartPointY = frameLTPoint.Y;
            double lineEndPointY = frameLTPoint.Y - headCellHeight - cellHeight;
            int vLineSize = isTitleCell ? locateSize + 2 : locateSize + 1;
            for(int i = 0; i < vLineSize; i++)
            {
                double linePointX = frameLTPoint.X + (double)i * cellWidth;

                // タイトルセル有りの場合
                if(isTitleCell && i > 0)
                    linePointX -= (cellWidth - titleCellWidth);

                Point2d lineStartPoint = new Point2d(linePointX, lineStartPointY);
                Point2d lineEndPoint = new Point2d(linePointX, lineEndPointY);

                LineShape lineShape = app.ShapeFactory.CreateLine(lineStartPoint, lineEndPoint);
                // 既定値の設定セット
                doc.BasicShapeSettings.CopyToShape(lineShape);

                // レイヤのセット
                lineShape.Layer = currentLayer;

                // 色
                if(customSettings.frameLineColorNumber >= 0)
                    lineShape.ColorNumber = customSettings.frameLineColorNumber;

                // 線種
                lineShape.LinetypeNumber = 1;

                // 線幅
                if(customSettings.frameLinewidthNumber >= 0)
                    lineShape.LinewidthNumber = customSettings.frameLinewidthNumber;

                // 追加
                shapes.Add(lineShape);
            }

            return shapes;
        }
    }
}