﻿using RootPro.RootProCAD;
using RootPro.RootProCAD.Command;
using RootPro.RootProCAD.Geometry;
using RootPro.RootProCAD.UI;
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Windows.Forms;
using static RCAddInShadowDrawing.DockingBarUserControl;

/// <summary>
/// 日影図作成プログラム
/// 概要はDockingBarUserControl.cs参照
/// ここは、主に指定点の日影時間を計算するための処理コードを書いている
/// 
/// ※使用許諾
/// ・このアドインを実行した結果についての一切の責任はお使いになられる方にあります。
/// ・このアドインは、プログラムソース自体を改変・流用することが可能です。
/// 　また、このプログラムソースを元に独自のアドインやソフトウェアを作成し、無償・有償を問わず第三者に配布することも認めます。
/// 　ただし、改変・流用する場合は、株式会社ルートプロ(RootPro Corporation)作成の
/// 　当プログラムソースを参考にしていることを第三者にわかるように明記してください。
/// 　
/// 　Copyright @ 2019 RootPro Co.,Ltd. (Japan)
/// 
/// </summary>

namespace RCAddInShadowDrawing
{
    partial class DockingBarUserControl
    {
        // 点指定した点の影情報リスト
        private SortedList<int, PinpointShadowAreaInfo> pinpointShadowAreaInfoList = new SortedList<int, PinpointShadowAreaInfo>();

        /// <summary>
        /// 指定点開始No.の値が変わったときのイベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void numberingNumericUpDown_ValueChanged(object sender, EventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 点指定 ボタンクリック
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void markDotButton_Click(object sender, EventArgs e)
        {
             app.CommandManager.CancelCurrentCommand();

             // コマンドの実行
             app.CommandManager.ExecuteCommand(ShadowDrawingMarkDotCommandName);
        }

        /// <summary>
        /// 計算して結果を配置 ボタンクリック
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void locateComputationButton_Click(object sender, EventArgs e)
        {
            if(!EditControlChk())
                return;

             app.CommandManager.CancelCurrentCommand();

            // 点指定の点の情報更新
            if(!UpdatePinpointShadowAreaInfo())
                return;

             // コマンドの実行
            app.CommandManager.ExecuteCommand(ShadowDrawingLocateComputationCommandName);
        }


        /// <summary>
        /// 点指定コマンドマウス移動イベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void CommandMarkDot_MouseMove(Object sender, CommandMouseMoveEventArgs e)
        {
             Command command = (Command)sender;
            // Command.GetTemporaryParameterItemByID を使用してコマンドの一時的なパラメータアイテムを取得します。
            // 一時的なパラメータアイテムからは、すでにパラメータの入力が確定している場合は確定している入力値が、
            // 入力が確定していない場合はマウスの現在位置などの一時的な値が取得できます。

            // 一時パラメータを取得する
            PointParameterItem pointParameterItem = (PointParameterItem)command.GetTemporaryParameterItemByID(locatePointParameterId);
            if(pointParameterItem == null)
                return;

            if(pointParameterItem.IsEmpty)
                return;

            // 配置点
            Point2d locatePoint = pointParameterItem.Point;

            //ドキュメントの取得
            Document doc = app.ActiveDocument;
            Drawing drawing = doc.CurrentDrawing;
            Point2d orgPoint = new Point2d(0.0, 0.0);

            // 図形の作成
            if(rubberBandShapes == null)
            {
                // 配置点はとりあえず原点で作成する
                // ラバーバンドを作ってしまうので後で座標だけ変える
                Shape[] shapes = CreateShapesForCommandMarkDot(command, drawing, orgPoint);
                if(shapes == null)
                    return;

                rubberBandShapes = new List<Shape>();
                rubberBandShapes.AddRange(shapes);
            }

            List<Shape> rubberBandShapes2 = new List<Shape>();
            foreach(Shape shape in rubberBandShapes)
            {
                // マウスの位置に移動
                Shape shape2 = shape.Clone();
                shape2.Transform(orgPoint, 1.0, 1.0, 0.0, locatePoint);
                rubberBandShapes2.Add(shape2);
            }

            try
            {
                // ラバーバンド描画
                command.DrawShapeRubberBand(rubberBandShapes2.ToArray());
            }
            catch
            {
                throw;
            }

        }

        /// <summary>
        /// 点指定コマンドパラメータ変更イベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void CommandMarkDot_ParameterChanged(Object sender, CommandParameterChangedEventArgs e)
        {
            //ラバーバンド表示を削除する
            ClearRubberBand();

            Command command = (Command)sender;

            //ドキュメントの取得
            Document doc = app.ActiveDocument;
            Drawing drawing = doc.CurrentDrawing;

            Point2d locatePoint = new Point2d(0.0, 0.0);
            PointParameterItem pointParameterItem = (PointParameterItem)command.GetParameterItemByID(locatePointParameterId);
            if(pointParameterItem == null)
                return;

            if(pointParameterItem.IsEmpty)
                return;

            locatePoint = pointParameterItem.Point;

            // 図形の作成
            Shape[] shapes = CreateShapesForCommandMarkDot(command, drawing, locatePoint);
            if(shapes == null)
            {
                e.CommandState = CommandState.Continue;
                return;
            }

            // Undoの開始
            doc.UndoManager.BeginUndoUnit();
 
            ArrayList addGroupShapeArray = new ArrayList();     // グループ化するための図形リスト

            for(int i = 0; i < shapes.Length; i++)
            {
                Shape shape = shapes[i];

                // レイヤのセット
                Layer currentLayer = doc.LayerTable.CurrentLayer;
                shape.Layer = currentLayer;

                // 追加
                Shape addShape = drawing.Shapes.Add(shape);

                // グループ
                addGroupShapeArray.Add(addShape);
            }

            // グループ化
            if(addGroupShapeArray.Count > 0)
            {
                Shape shapeGroup = drawing.Shapes.AddDrawGroup((Shape[])addGroupShapeArray.ToArray(typeof(Shape)), false);
            }

            // Undo の終了
            doc.UndoManager.EndUndoUnit();

            // コマンド実行
            e.CommandState = CommandState.Execute;

            // 開始No.更新
            numberingNumericUpDown.UpButton();

            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 点指定コマンドで描き出す図形作成
        /// </summary>
        /// <param name="command"></param>
        /// <param name="drawing"></param>
        /// <param name="locatePoint"></param>
        /// <returns></returns>
        private Shape[] CreateShapesForCommandMarkDot(Command command, Drawing drawing, Point2d locatePoint)
        {
            string strValue = numberingNumericUpDown.Text;
            if(strValue == "")
                return null;

            Document doc = drawing.Document;

            ArrayList shapes = new ArrayList();// 戻り値の図形リスト
            string str = customSettings.markDotNumberHeadText + strValue;

            TextShape text = app.ShapeFactory.CreateText(str, locatePoint, 0.0);
            doc.TextSettings.CopyToShape(text);

            // 属性の追加
            ShapeAttributeCollection attributes = text.Attributes;
            attributes.Add(attributesName, attributesValueMarkDot);// 点指定の点であることを表す属性

            text.DirectionVertical = false;// 横書き
            text.FontHeight = GetTextFontHeight();// 文字高さセット
            text.Alignment = Alignment.BottomLeft;// 配置点：左下
            shapes.Add(text);
 
            // 点図形
            PointMarkerShape pointMarkerShape = app.ShapeFactory.CreatePointMarker(locatePoint);
            doc.PointMarkerSettings.CopyToShape(pointMarkerShape);
            pointMarkerShape.MarkerType = PointMarkerType.FilledCircle;// 点タイプ●
            pointMarkerShape.Size = text.FontHeight * 0.2;// 大きさ文字の20%
            shapes.Add(pointMarkerShape);

            return (Shape[])shapes.ToArray(typeof(Shape));
        }

        /// <summary>
        /// 計算して結果を配置コマンドマウス移動イベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void CommandLocateComputation_MouseMove(Object sender, CommandMouseMoveEventArgs e)
        {
            Command command = (Command)sender;
            // Command.GetTemporaryParameterItemByID を使用してコマンドの一時的なパラメータアイテムを取得します。
            // 一時的なパラメータアイテムからは、すでにパラメータの入力が確定している場合は確定している入力値が、
            // 入力が確定していない場合はマウスの現在位置などの一時的な値が取得できます。

            // グラフの付加の取得
            ListParameterItem listParameterItem = (ListParameterItem)command.GetTemporaryParameterItemByID(graphModeId);
            if(listParameterItem == null)
                return;

            if(listParameterItem.IsEmpty)
                return;

            bool withGraph = false;
            if(listParameterItem.SelectedIndex == 1)
                withGraph = true;

            // 配置点パラメータを取得する
            PointParameterItem pointParameterItem = (PointParameterItem)command.GetTemporaryParameterItemByID(locatePointParameterId);
            if(pointParameterItem == null)
                return;

            if(pointParameterItem.IsEmpty)
                return;

            // 配置点
            Point2d locatePoint = pointParameterItem.Point;

            //ドキュメントの取得
            Document doc = app.ActiveDocument;
            Drawing drawing = doc.CurrentDrawing;
            Point2d orgPoint = new Point2d(0.0, 0.0);

            // 図形の作成
            if(rubberBandShapes == null)
            {
                // 配置点はとりあえず原点で作成する
                // ラバーバンドを作ってしまうので後で座標だけ変える
                Shape[] shapes = CreateShapesForCommandLocateComputation(orgPoint, withGraph);
                if(shapes == null)
                    return;

                rubberBandShapes = new List<Shape>();
                rubberBandShapes.AddRange(shapes);
            }

            List<Shape> rubberBandShapes2 = new List<Shape>();
            foreach(Shape shape in rubberBandShapes)
            {
                // マウスの位置に移動
                Shape shape2 = shape.Clone();
                shape2.Transform(orgPoint, 1.0, 1.0, 0.0, locatePoint);
                rubberBandShapes2.Add(shape2);
            }

            try
            {
                // ラバーバンド描画
                command.DrawShapeRubberBand(rubberBandShapes2.ToArray());
            }
            catch
            {
                throw;
            }
        }

        /// <summary>
        /// 計算して結果を配置コマンドパラメータ変更イベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void CommandLocateComputation_ParameterChanged(Object sender, CommandParameterChangedEventArgs e)
        {
            Command command = (Command)sender;

            //ラバーバンド表示を削除する
            ClearRubberBand();

            //ドキュメントの取得
            Document doc = app.ActiveDocument;
            Drawing drawing = doc.CurrentDrawing;

            // グラフの付加の取得
            ListParameterItem listParameterItem = (ListParameterItem)command.GetParameterItemByID(graphModeId);
            if(listParameterItem == null)
                return;

            if(listParameterItem.IsEmpty)
                return;

            bool withGraph = false;
            if(listParameterItem.SelectedIndex == 1)
                withGraph = true;

            Point2d locatePoint = new Point2d(0.0, 0.0);
            PointParameterItem pointParameterItem = (PointParameterItem)command.GetParameterItemByID(locatePointParameterId);
            if(pointParameterItem == null)
                return;

            if(pointParameterItem.IsEmpty)
                return;

            locatePoint = pointParameterItem.Point;

            // 図形の作成
            Shape[] shapes = CreateShapesForCommandLocateComputation(locatePoint, withGraph);
            if(shapes == null)
            {
                e.CommandState = CommandState.Continue;
                return;
            }

            // Undoの開始
            doc.UndoManager.BeginUndoUnit();
 
            ArrayList addGroupShapeArray = new ArrayList();     // グループ化するための図形リスト

            for(int i = 0; i < shapes.Length; i++)
            {
                Shape shape = shapes[i];

                // レイヤのセット
                Layer currentLayer = doc.LayerTable.CurrentLayer;
                shape.Layer = currentLayer;

                // 追加
                Shape addShape = drawing.Shapes.Add(shape);

                // グループ
                addGroupShapeArray.Add(addShape);
            }

            // グループ化
            if(addGroupShapeArray.Count > 0)
            {
                Shape shapeGroup = drawing.Shapes.AddDrawGroup((Shape[])addGroupShapeArray.ToArray(typeof(Shape)), false);
            }

            // Undo の終了
            doc.UndoManager.EndUndoUnit();

            // 「計算して結果を配置」のグラフ付きフラグ保持
            addInSettings.locateComputationWithGraph = withGraph;

            // コマンド終了
            e.CommandState = CommandState.End;

            pinpointShadowAreaInfoList.Clear();

            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 点指定の点の情報更新
        /// </summary>
        /// <returns></returns>
        private bool UpdatePinpointShadowAreaInfo()
        {
            // ウェイトカーソル
            Cursor.Current = Cursors.WaitCursor;
            
            //ドキュメントの取得
            Document doc = app.ActiveDocument;
            //Drawing drawing = doc.CurrentDrawing;

            // 実行できる状態の図面か
            if(!CheckDocument(doc, true))
                return false;

            // 観測地点の緯度
            double latitudeDeg = 0; // (deg)

            // 日赤緯(冬至:-23.45°、春秋分:0°、夏至:23.45°、任意時期)
            double solarDeclinationDeg = 0.0;// 初期値:春秋分(deg)

            // 測定面の高さ
            double locationHeight = 0.0;

            // 真北の角度を探す
            double dueNorthDeg = 0.0;       // 真北の角度(deg)
            int result = DueNorthSearch(doc, out dueNorthDeg);
            if(result != 1)
            {
                if(result == 0)
                    MessageBox.Show("真北が設定されていません。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                else if(result == -1)
                    MessageBox.Show("真北が図面中に複数あります。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);

                return false;
            }

            // 観測地点の緯度、日赤緯、測定面の高さ
            if(!GetShadowBaseData(out latitudeDeg, out solarDeclinationDeg, out locationHeight))
                return false;

            // 高さの文字のリスト取得
            var heightTextShapeList = new List<TextShape>();
            int count = GetWithAttributesTextShapeList(doc, attributesValueZ, heightTextShapeList);
            if(count == 0)
            {
                MessageBox.Show("高さの設定された線がありません。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                return false;
            }

            double maxHeightValue = 0.0;
            if(!ChkHeightTextShapeList(heightTextShapeList, out maxHeightValue))
            {
                MessageBox.Show("高さのデータまたは測定面の高さが正しくありません。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                return false;
            }
 
            // 実行できるレイアウトの図面かチェック
            if(!CheckLayout(doc, heightTextShapeList))
            {
                return false;
            }

            // 高さの「ベース線」(高さが書かれている建物の縁を表す線)リストを取得
            List<List<BaseLine>> baseLinesList = GetHeightBaseLineShapeList(doc, heightTextShapeList);
            if(baseLinesList.Count == 0)
                return false;

            // ベース線からなる多角形のリスト作成
            List<BaseLinePolygon> baseLinesPolygonList = new List<BaseLinePolygon>();
            List<BaseLinePolygon> subBaseLinesPolygonList = null;
            foreach(List<BaseLine> baseLines in baseLinesList)
            {
                subBaseLinesPolygonList = GetBaseLinesPolygonList(doc, locationHeight, baseLines);
                if(subBaseLinesPolygonList.Count > 0)
                    baseLinesPolygonList.AddRange(subBaseLinesPolygonList);
            }

            // 日影データを取得
            double latitudeRad = latitudeDeg * (Math.PI / 180d);
            double solarDeclinationRad = solarDeclinationDeg * (Math.PI / 180d); // 日赤緯(冬至: -23.45°、春秋分: 0°、夏至: 23.45°、任意時期)
            bool h9Toh15 = timeRangeComboBox.SelectedIndex == 1 ? true : false;// true:測定時間が9:00～15:00の場合
            double startSolarTime = 8;
            double endSolarTime = 16;
            // 測定範囲が9:00～15:00の場合
            if(h9Toh15)
            {
                startSolarTime = 9;
                endSolarTime = 15;
            }

            int intervalType = 60;// 計算間隔(単位：秒)
            if(calculationSpanLongRadioButtonB.Checked)
            {
                intervalType = 60;
            }
            else if(calculationSpanShortRadioButtonB.Checked)
            {
                intervalType = 10;
            }
            else
            {
                Debug.Assert(false);
            }

            // 日影データリスト作成
            List<ShadowData> shadowDataList = GetShadowDataList(latitudeRad, solarDeclinationRad, intervalType, startSolarTime, endSolarTime);

            // 影の四角リスト作成
            List<List<Quadrilateral>> shadowQuadsList = new List<List<Quadrilateral>>(shadowDataList.Count);
            for(int i = 0; i < shadowDataList.Count; i++)
            {
                ShadowData shadowData = shadowDataList[i];
                double solarTime = shadowData.solarTime;
                double xMagnification = shadowData.xMagnification;
                double yMagnification = shadowData.yMagnification;

                // ベース線からなる多角形ごとに影の四角形領域のリスト作成
                int capacity = baseLinesPolygonList.Count;
                List<Quadrilateral> shadowQuads = new List<Quadrilateral>(capacity);// ベース線からできる影の四角リスト
                foreach(BaseLinePolygon baseLinePolygon in baseLinesPolygonList)
                {
                    // ベース線と高さ文字情報からできる影の四角形領域のリスト作成取得
                    var quads = GetQuadrilateralList(doc, baseLinePolygon, heightTextShapeList, solarTime, xMagnification, yMagnification, locationHeight, dueNorthDeg);
                    shadowQuads.AddRange(quads);
                }

                shadowQuadsList.Add(shadowQuads);
            }

            // 点指定した点の文字のリスト取得
            var markDotTextShapeList = new List<TextShape>();
            int markDotTextShapeListCount = GetWithAttributesTextShapeList(doc, attributesValueMarkDot, markDotTextShapeList);
            if(markDotTextShapeListCount == 0)
            {
                MessageBox.Show("点指定した文字がありません。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                return false;
            }

            int intervalTime = intervalType;

            bool sameNumber = false;
            pinpointShadowAreaInfoList.Clear();
            for(int i = 0; i < markDotTextShapeListCount; i++)
            {
                TextShape textShape = markDotTextShapeList[i];
                Point2d textLocatePoint = textShape.Point;

                // 番号の文字を数値に
                string numberText = textShape.Text;
                string numberText2 = numberText.Remove(0, customSettings.markDotNumberHeadText.Length);
                if(!int.TryParse(numberText2, out int number))
                {
                    MessageBox.Show("指定点の番号が正しくないものがあります。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return false;
                }

                // 既に追加されていないか
                if(pinpointShadowAreaInfoList.ContainsKey(number))
                {
                    if(!sameNumber)
                    {
                        string msg = customSettings.markDotNumberHeadText + number.ToString() + " が複数見つかりました。同じ番号のものは最後に見つかった場所のものを用います。よろしいですか?";
                        if(MessageBox.Show(msg, AddInProgramName, MessageBoxButtons.OKCancel, MessageBoxIcon.Warning) != DialogResult.OK)
                            return false;
                    }

                    sameNumber = true;
                    pinpointShadowAreaInfoList.Remove(number);
                }

                // 指定点の日影時間情報
                PinpointShadowAreaInfo pinpointShadowAreaInfo = CreatePinpointShadowAreaInfo(shadowQuadsList, numberText, textLocatePoint);

                // リストに追加
                pinpointShadowAreaInfoList.Add(number, pinpointShadowAreaInfo);
            }

            return true;
        }

        /// <summary>
        /// 計算して結果を配置コマンドで描き出す図形作成
        /// </summary>
        /// <param name="drawing"></param>
        /// <param name="locatePoint"></param>
        /// <param name="withGraph"></param>
        /// <returns></returns>
        private Shape[] CreateShapesForCommandLocateComputation(Point2d locatePoint, bool withGraph)
        {
            if(pinpointShadowAreaInfoList.Count <= 0)
               return null;

            //ドキュメントの取得
            Document doc = app.ActiveDocument;

            // 一覧にして表示、グラフ対応
            ArrayList shapes = new ArrayList();// 戻り値の図形リスト
            double textHeight = GetTextFontHeight() * customSettings.locateComputationTextHightRatio;// 文字高さ
            int roopCounter = 0;
            foreach(KeyValuePair<int, PinpointShadowAreaInfo> pair in pinpointShadowAreaInfoList)
            {
                int number = pair.Key;
                PinpointShadowAreaInfo pinpointShadowAreaInfo = pair.Value;

                // 時分秒文字列変換
                // 総日影時間
                int totalShadowTimeS = pinpointShadowAreaInfo.GetTotalShadowTimesSeconds();
                string totalShadowTimeText = ConvertTimeValueToHMSText(totalShadowTimeS);

                string outputText = pinpointShadowAreaInfo.numberText;

                string space0 = "";
                if(pinpointShadowAreaInfo.numberText.Length < 5)
                    space0 = "   ";
                else if(pinpointShadowAreaInfo.numberText.Length < 6)
                    space0 = "  ";

                outputText += space0 + "[ " + totalShadowTimeText + " ]";// 影の合計時間文字列

                int shadowTimesListCount = pinpointShadowAreaInfo.shadowTimesList.Count;
                for(int i = 0; i < shadowTimesListCount; i++)
                {
                    // 影の始まりの時間～終りの時間文字列
                    ShadowTimes shadowTimes = pinpointShadowAreaInfo.shadowTimesList[i];
                    string shadowTimesValueToHMSText = GetStartEndShadowTimesValueToHMSText(shadowTimes);
                    if(shadowTimesValueToHMSText != "")
                        outputText += shadowTimesValueToHMSText;
                }

                double intervalRowHeight = textHeight * 1.5;
                double spaceTextAndGraph = 0.0; // 時間文字とグラフの間隔
                double graphRowHeight = 0.0;    // グラフの高さ(余白含む)
                if(withGraph)
                {
                    spaceTextAndGraph = textHeight * 0.1;
                    graphRowHeight = textHeight;
                }

                Point2d textLocatePoint = new Point2d(locatePoint.X, locatePoint.Y - (intervalRowHeight + spaceTextAndGraph + graphRowHeight) * roopCounter);
                TextShape text = app.ShapeFactory.CreateText(outputText, textLocatePoint, 0.0);
                doc.TextSettings.CopyToShape(text);

                text.DirectionVertical = false;// 横書き
                text.FontHeight = textHeight;// 文字高さセット
                text.Alignment = Alignment.TopLeft;// 配置点：左上
                shapes.Add(text);

                // グラフ
                if(withGraph)
                {
                    int startTimeH = 8;// グラフの開始時間
                    int endTimeH = 16;// グラフの終了時間
                    double graphColWidth = graphRowHeight * customSettings.locateComputationGraphColWidthRatio; // 1時間の幅
                    double graphTotalWidth = graphColWidth * (double)(endTimeH - startTimeH); // 時間の幅
                    double offset = graphRowHeight * 0.1; // グラフの余白

                    // グラフの各点作成
                    Point2d graphLocatePointTopLeft = new Point2d(textLocatePoint.X, textLocatePoint.Y - (textHeight + spaceTextAndGraph + offset));
                    Point2d graphLocatePointTopRight = new Point2d(graphLocatePointTopLeft.X + graphTotalWidth, graphLocatePointTopLeft.Y);
                    Point2d graphLocatePointBottomLeft = new Point2d(graphLocatePointTopLeft.X, graphLocatePointTopLeft.Y - (graphRowHeight - offset));
                    Point2d graphLocatePointBottomRight = new Point2d(graphLocatePointBottomLeft.X + graphTotalWidth, graphLocatePointBottomLeft.Y);

                    // 影の時間帯をハッチングで塗る(グラフは8時～16時の間)
                    int graphAllTimeS = (endTimeH - startTimeH) * 3600;

                    // 影の時間部分をハッチングで塗る
                    for(int i = 0; i < shadowTimesListCount; i++)
                    {
                        // 影の始まりの時間～終りの時間
                        ShadowTimes shadowTimes = pinpointShadowAreaInfo.shadowTimesList[i];
                        int startShadowTimeS = shadowTimes.GetStartShadowTimeSeconds();
                        int endShadowTimeS = shadowTimes.GetEndShadowTimeSeconds();

                        double firstHitShadowTimePointX = graphLocatePointTopLeft.X + (graphTotalWidth / (double)graphAllTimeS) * (double)(startShadowTimeS - startTimeH * 3600);
                        double lastHitShadowTimePointX = graphLocatePointTopLeft.X + (graphTotalWidth / (double)graphAllTimeS) * (double)(endShadowTimeS - startTimeH * 3600);

                        if(lastHitShadowTimePointX - firstHitShadowTimePointX > RcPrecisionConfusion)
                        {
                            Point2d[] polylinePoints = new Point2d[5];
                            polylinePoints[0] = new Point2d(firstHitShadowTimePointX, graphLocatePointTopLeft.Y);
                            polylinePoints[1] = new Point2d(lastHitShadowTimePointX, graphLocatePointTopLeft.Y);
                            polylinePoints[2] = new Point2d(lastHitShadowTimePointX, graphLocatePointBottomRight.Y);
                            polylinePoints[3] = new Point2d(firstHitShadowTimePointX, graphLocatePointBottomRight.Y);
                            polylinePoints[4] = polylinePoints[0];
                            PolylineShape polylineShape = app.ShapeFactory.CreatePolyline(polylinePoints);
                            Shape[] regionShapes = new Shape[1];
                            regionShapes[0] = polylineShape;
                            HatchShape hatchShape = app.ShapeFactory.CreateHatch(regionShapes);
                            shapes.Add(hatchShape);
                            doc.HatchSettings.CopyToShape(hatchShape);
                            hatchShape.HatchType = HatchType.Solid;
                            hatchShape.ShowRegion = false;
                            hatchShape.ReverseColor = true;// 背景色と同じ場合は白黒反転
                        }
                    }

                    // 横線
                    LineShape graphLineTop = app.ShapeFactory.CreateLine(graphLocatePointTopLeft, graphLocatePointTopRight);
                    LineShape graphLineBottom = app.ShapeFactory.CreateLine(graphLocatePointBottomLeft, graphLocatePointBottomRight);
                    shapes.Add(graphLineTop);
                    doc.BasicShapeSettings.CopyToShape(graphLineTop);
                    shapes.Add(graphLineBottom);
                    doc.BasicShapeSettings.CopyToShape(graphLineBottom);

                    // 縦線
                    for(int i = 0; i < (endTimeH - startTimeH + 1); i++)
                    {
                        Point2d graphGageLinePointTop = new Point2d(graphLocatePointTopLeft.X + graphColWidth * i, graphLocatePointTopLeft.Y + offset);
                        Point2d graphGageLinePointBottom = new Point2d(graphGageLinePointTop.X, graphGageLinePointTop.Y - graphRowHeight - offset);

                        LineShape graphGageLine = app.ShapeFactory.CreateLine(graphGageLinePointTop, graphGageLinePointBottom);
                        shapes.Add(graphGageLine);
                        doc.BasicShapeSettings.CopyToShape(graphGageLine);
                    }
                }
           
                roopCounter++;
            }

            return (Shape[])shapes.ToArray(typeof(Shape));
        }

        /// <summary>
        /// 秒値を時分秒の時間文字列に変換 ex. 19855 -> 5:30'55"
        /// </summary>
        /// <param name="timeValueS"></param>
        /// <returns></returns>
        private string ConvertTimeValueToHMSText(int timeValueS)
        {
            int shadowTimeH = Math.DivRem(timeValueS, 3600, out int shadowTimeS);
            int shadowTimeM = Math.DivRem(shadowTimeS, 60, out shadowTimeS);
            string hmsText = shadowTimeH.ToString() + ":" + String.Format("{0:00}", shadowTimeM) + "'" + String.Format("{0:00}", shadowTimeS) + "\"";
            return hmsText;
        }

        /// <summary>
        /// 影の始まりの時間～終りの時間文字列 取得
        /// </summary>
        /// <param name="shadowTimes"></param>
        /// <returns></returns>
        public string GetStartEndShadowTimesValueToHMSText(ShadowTimes shadowTimes)
        {
            string shadowTimesValueToHMSText = "";

            int startShadowTimeS = shadowTimes.GetStartShadowTimeSeconds();
            string startShadowTimeText = ConvertTimeValueToHMSText(startShadowTimeS);

            int endShadowTimeS = shadowTimes.GetEndShadowTimeSeconds();
            string endShadowTimeText = ConvertTimeValueToHMSText(endShadowTimeS);

            // 同じ時間なら空文字を返す
            if(startShadowTimeText == endShadowTimeText)
                return shadowTimesValueToHMSText;

            string space1 = "";
            if(startShadowTimeText.Length < 9)
                space1 = " ";

            string space2 = "";
            if(endShadowTimeText.Length < 9)
                space2 = " ";

            // 影の始まりの時間～終りの時間文字列
            shadowTimesValueToHMSText = " ( " + space1 + startShadowTimeText + " )-( " + space2 + endShadowTimeText + " )";

            return shadowTimesValueToHMSText;
        }
    }
}