﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Windows.Forms;
using RootPro.RootProCAD;
using RootPro.RootProCAD.Command;
using RootPro.RootProCAD.Geometry;
using RootPro.RootProCAD.UI;
using System.Diagnostics;
using System.Text.RegularExpressions;
using System.Linq;

/// <summary>
/// 日影図作成プログラム
/// 概要はDockingBarUserControl.cs参照
/// ここは、主に等時間日影図作成のための処理コードを書いている
/// 
/// 例：8時から16時までの間で3時間以上影になる部分を求める場合（1分間隔計算）
/// 
/// １．日影全体を覆う範囲の格子状に並んだ点グリッドを用意
/// ２．ベース線と指定した間隔の時刻のそれからなる影の四角形を求める。
/// ３．一定の時間分の影の四角形データを考慮して、その四角形内にあるグリッド点を求める。
/// ４．３.で求めた点の集まりの外周を線で結ぶことで等時間日影図とする(最終的には誤差考慮のため３．の点を中心としたセルの四隅の点の集まりを線で結んで囲む。最終的にはその点と点を結ぶ線の中点同士を結んでなめらかに表している)
/// 
/// ※使用許諾
/// ・このアドインを実行した結果についての一切の責任はお使いになられる方にあります。
/// ・このアドインは、プログラムソース自体を改変・流用することが可能です。
/// 　また、このプログラムソースを元に独自のアドインやソフトウェアを作成し、無償・有償を問わず第三者に配布することも認めます。
/// 　ただし、改変・流用する場合は、株式会社ルートプロ(RootPro Corporation)作成の
/// 　当プログラムソースを参考にしていることを第三者にわかるように明記してください。
/// 　
/// 　Copyright @ 2019 RootPro Co.,Ltd. (Japan)
/// 
/// </summary>


namespace RCAddInShadowDrawing
{
    partial class DockingBarUserControl
    {
        double dueNorthDegForDrawRangeCommand = 0.0;    // 「範囲を指定」コマンドのための真北方向角度メンバ変数

        /// <summary>
        /// グリッド点の状態
        /// </summary>
        [Flags]
        public enum GridPointStatus
        {
            NonCheck = 0x000, // 未チェック
            InStructure = 0x001, // 建物内
            MoreThanATimeShadow = 0x002, // 一定時間以上影(範囲外周線確定前)
            OnMoreThanATimeShadowAreaOutline = 0x004, // 一定時間以上影の範囲の外周線上
            InMoreThanATimeShadowAreaOutline = 0x008  // 一定時間以上影の範囲の外周線内
        }

        /// <summary>
        /// 接続されてきた前の点からの方向
        /// </summary>
        public enum JointToThisPointVector : byte
        {
            ToR = 1,   // 右へ
            ToRB = 2,  // 右下へ
            ToB = 3,   // 下へ
            ToLB = 4,  // 左下へ
            ToL = 5,   // 左へ
            ToLT = 6,  // 左上へ
            ToT = 7,   // 上へ
            ToRT = 8   // 右上へ
        }

        public class GridPoint
        {
            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="p"></param>
            /// <param name="stat"></param>
            public GridPoint(Point2d p, GridPointStatus stat = GridPointStatus.NonCheck)
            {
                point = p;
                shadowStatus = stat;
            }

            public Point2d point;
            public GridPointStatus shadowStatus;
        }

        /// <summary>
        /// 等時間日影図作成のときの測定範囲のグリッドクラス
        /// </summary>
        public class Grid
        {

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="app"></param>
            /// <param name="gridOriginPoint">グリッド原点</param>
            /// <param name="vCount">グリッドの縦方向の数</param>
            /// <param name="hCount">グリッドの横方向の数</param>
            /// <param name="gridPitch">グリッドの間隔</param>
            /// <param name="rotateAngleDeg">角度</param>
            public Grid(AppAddIn app, Point2d gridOriginPoint, int vCount, int hCount, double gridPitch, double rotateAngleDeg)
            {
                pitch = gridPitch;
                gridPoints = new GridPoint[vCount, hCount];
                originPoint = gridOriginPoint;
                rotateAngle = rotateAngleDeg;

                double rotateAngleRad = ToRadian(rotateAngle);

                for(int i = 0; i < vCount; i++)
                {
                    double y = gridOriginPoint.Y + gridPitch * i;

                    for(int j = 0; j < hCount; j++)
                    {
                        double x = gridOriginPoint.X + gridPitch * j;

                        Point2d point = new Point2d(x, y);

                        // グリッド基点を中心に回転
                        point = GetRotatePoint(point, originPoint, rotateAngleRad);

                        // 座標セット
                        gridPoints[i, j] = new GridPoint(point);
                    }
                }
            }


            public GridPoint[,] GetGridPoints()
            {
                return gridPoints;
            }

            /// <summary>
            /// グリッド点の縦方向の数
            /// </summary>
            /// <returns></returns>
            public int GetVCount()
            {
                return gridPoints.GetLength(0);
            }

            /// <summary>
            /// グリッド点の横方向の数
            /// </summary>
            /// <returns></returns>
            public int GetHCount()
            {
                return gridPoints.GetLength(1);
            }

            public double GetPitch()
            {
                return pitch;
            }

            public Point2d GetOriginPoint()
            {
                return originPoint;
            }

            public double GetRotateAngle()
            {
                return rotateAngle;
            }

            private GridPoint[,] gridPoints;// グリッド点配列
            private double pitch;           // 間隔
            private Point2d originPoint;    // 原点
            private double rotateAngle;     // 回転角度
        }

        /// <summary>
        /// 点のグリッド中での位置インデックスクラス
        /// </summary>
        public class PointIndexVH
        {
            public PointIndexVH(int v, int h)
            {
                this.v = v;
                this.h = h;
            }

            public int v;
            public int h;
        }

        // 線の始点終点クラス
        private class LinePoints
        {
            public LinePoints(Point2d p1, Point2d p2)
            {
                point1 = new Point2d(p1.X, p1.Y);
                point2 = new Point2d(p2.X, p2.Y);
            }

            public Point2d point1;
            public Point2d point2;
        }

        /// <summary>
        /// 影の時間クラス
        /// </summary>
        public class ShadowTimes
        {
            /// <summary>
            /// コンストラクター
            /// </summary>
            /// <param name="startTime">影になり始める時間</param>
            /// <param name="endTime">影が終わる時間</param>
            public ShadowTimes(double startTime, double endTime)
            {
                Debug.Assert(startTime <= endTime);

                this.startShadowTime = startTime;
                this.endShadowTime = endTime;
            }

            private double startShadowTime;// 影になり始める時間(単位：時、12.5 -> 12時30分 -> 12時1800秒)
            public double GetStartShadowTime() { return startShadowTime; }
                        private double endShadowTime;  // 影が終わる時間(単位：時)
            public double GetEndShadowTime() { return endShadowTime; }

            /// <summary>
            /// 影である時間(単位：秒)
            /// </summary>
            /// <returns></returns>
            public int GetShadowTimeSeconds()
            {
                int shadowTimeSeconds = RoundShadowTimeToSeconds(endShadowTime) - RoundShadowTimeToSeconds(startShadowTime);
                return shadowTimeSeconds;
            }

            /// <summary>
            /// 影になり始める時間を取得(単位：秒)
            /// </summary>
            /// <returns></returns>
            public int GetStartShadowTimeSeconds()
            {
                int startShadowTimeS = RoundShadowTimeToSeconds(this.startShadowTime);
                return startShadowTimeS;
            }

            /// <summary>
            /// 影が終わる時間を取得(単位：秒)
            /// </summary>
            /// <returns></returns>
            public int GetEndShadowTimeSeconds()
            {
                int startShadowTimeS = RoundShadowTimeToSeconds(this.endShadowTime);
                return startShadowTimeS;
            }

            /// <summary>
            /// 時間を秒にした値を取得
            /// </summary>
            /// <param name="shadowTime"></param>
            /// <returns></returns>
            public static int RoundShadowTimeToSeconds(double shadowTime)
            {
                int shadowTimeSeconds = (int)Math.Round(shadowTime * 3600.0, MidpointRounding.AwayFromZero);
                return shadowTimeSeconds;
            }

        }

        /// <summary>
        /// 点の影情報クラス
        /// </summary>
        public class PinpointShadowAreaInfo
        {
            public PinpointShadowAreaInfo(string numberText)
            {
                this.numberText = numberText;
                shadowTimesList = new List<ShadowTimes>();
            }

            public string numberText;
            public List<ShadowTimes> shadowTimesList;
            
            /// <summary>
            /// この点が日影になる時間帯の追加
            /// </summary>
            /// <param name="shadowTimes"></param>
            public void AddShadowTimes(ShadowTimes shadowTimes)
            {
                shadowTimesList.Add(shadowTimes);
            }

            /// <summary>
            /// この点の日影になる総時間取得(単位：秒)
            /// </summary>
            /// <returns></returns>
            public int GetTotalShadowTimesSeconds()
            {
                int totalShadowTimesSeconds = 0;

                foreach(ShadowTimes shadowTimes in shadowTimesList)
                    totalShadowTimesSeconds += shadowTimes.GetShadowTimeSeconds();

                return totalShadowTimesSeconds;
            }
        }

        /// <summary>
        /// 範囲内を計算のチェックボタンクリックイベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void RangeCalculationCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            UpdateCalculationSpanButton();

            // 長い間隔の方をオンにする
            calculationSpanLongRadioButtonA.Checked = true;
        }

        private void UpdateCalculationSpanButton()
        {
            // 計算の間隔も切り替える
            if(rangeCalculationCheckBox.Checked)
            {
                calculationSpanLongRadioButtonA.Text = "10秒間隔";
                calculationSpanShortRadioButtonA.Text = "4秒間隔";
            }
            else
            {
                calculationSpanLongRadioButtonA.Text = "1分間隔";
                calculationSpanShortRadioButtonA.Text = "10秒間隔";
            }
        }

        /// <summary>
        /// 範囲を指定ボタンクリックイベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void DrawRangeButton_Click(object sender, EventArgs e)
        {
            app.CommandManager.CancelCurrentCommand();

            //ドキュメントの取得
            Document doc = app.ActiveDocument;

            // 用紙自身がカレントのレイアウトの場合は実行させない（影の線をm->mm換算して追加するためサイズが適当ではない）
            Layout　currentlayout = doc.CurrentLayout;
            if(currentlayout.Drawing is Paper)
            {
                String strErr = "カレントのレイアウトが用紙になっています。";
                MessageBox.Show(strErr, AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }

            // 真北の角度を探す
            double dueNorthDeg = 0.0;       // 真北の角度(deg)
            int result = DueNorthSearch(doc, out dueNorthDeg);
            if(result != 1)
            {
                if(result == 0)
                    MessageBox.Show("真北が設定されていません。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                else if(result == -1)
                    MessageBox.Show("真北が図面中に複数あります。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);

                return;
            }

            // 範囲を指定コマンドのための真北方向角度メンバ変数更新
            dueNorthDegForDrawRangeCommand = dueNorthDeg;

            // コマンドの実行
            app.CommandManager.ExecuteCommand(ShadowDrawingDrawRangeCommandName);
        }

        /// <summary>
        /// 範囲の削除ボタンクリックイベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void DeleteRangeButton_Click(object sender, EventArgs e)
        {
            //ドキュメントの取得
            Document doc = app.ActiveDocument;

            // 既に範囲があるか探して、あれば削除するよういする。
            PolylineShape alreadyCalcRange = FindCalcRange();
            Drawing alreadyCalcRangeDrawing = null;
            if(alreadyCalcRange != null)
            {
                // カレント以外で表示のみならエラー
                // または編集可能なレイヤでなければエラー
                alreadyCalcRangeDrawing = alreadyCalcRange.Drawing;
                if((!alreadyCalcRangeDrawing.IsCurrent && !alreadyCalcRangeDrawing.Editable) || (!alreadyCalcRange.Layer.IsCurrent && !alreadyCalcRange.Layer.Editable))
                {
                    MessageBox.Show("編集できない部分図かレイヤに既に「範囲」があります。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return;
                } 
            }
            else
            {
                return;
            }

            // Undoの開始
            doc.UndoManager.BeginUndoUnit();

            // 既にある範囲を削除
            if(alreadyCalcRange != null)
            {
                Debug.Assert(alreadyCalcRangeDrawing != null);
                alreadyCalcRangeDrawing.Shapes.Remove(alreadyCalcRange);
            }

            // UNDO 可能な操作を終了
            doc.UndoManager.EndUndoUnit();

        }


        /// <summary>
        /// 等時間日影図作成ボタンクリックイベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void ShadowDrawing20Button_Click(object sender, EventArgs e)
        {
            app.CommandManager.CancelCurrentCommand();
            DrawShadowDrawing(120.0f);
        }

        private void ShadowDrawing25Button_Click(object sender, EventArgs e)
        {
            app.CommandManager.CancelCurrentCommand();
            DrawShadowDrawing(150.0f);
        }

        private void ShadowDrawing30Button_Click(object sender, EventArgs e)
        {
            app.CommandManager.CancelCurrentCommand();
            DrawShadowDrawing(180.0f);
        }

        private void ShadowDrawing40Button_Click(object sender, EventArgs e)
        {
            app.CommandManager.CancelCurrentCommand();
            DrawShadowDrawing(240.0f);
        }

        private void ShadowDrawing50Button_Click(object sender, EventArgs e)
        {
            app.CommandManager.CancelCurrentCommand();
            DrawShadowDrawing(300.0f);
        }

        /// <summary>
        /// 時間指定(等時間日影図) ボタンクリック
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void shadowDrawingSpecifiedTimeButton_Click(object sender, EventArgs e)
        {
            app.CommandManager.CancelCurrentCommand();

            // コマンドの実行
            app.CommandManager.ExecuteCommand(ShadowDrawingSpecifyTimeBCommandName);
        }

        /// <summary>
        /// ピッチ情報を配置ボタンクリック
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>        
        private void pitchButton_Click(object sender, EventArgs e)
        {
            app.CommandManager.CancelCurrentCommand();

            string strValue = pitchTextBox.Text;
            if(strValue == "")
            {
                MessageBox.Show("等時間日影図を作成した後に実行できます。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
            }

            // コマンドの実行
            app.CommandManager.ExecuteCommand(ShadowDrawingPitchCommandName);
        }

        /// <summary>
        /// 等時間日影図作成
        /// </summary>
        /// <param name="minutes"></param>
        private void DrawShadowDrawing(float minutes)
        {
            // ピッチ情報クリア
            pitchTextBox.Text = "";
            pitchTextBox.Update();

            // ウェイトカーソル
            Cursor.Current = Cursors.WaitCursor;

            //ドキュメントの取得
            Document doc = app.ActiveDocument;
            Drawing drawing = doc.CurrentDrawing;

            // 実行できる状態の図面か
            if(!CheckDocument(doc, true))
                return;

            // 観測地点の緯度
            double latitudeDeg = 0; // (deg)

            // 日赤緯(冬至:-23.45°、春秋分:0°、夏至:23.45°、任意時期)
            double solarDeclinationDeg = 0.0;// 初期値:春秋分(deg)

            // 測定面の高さ
            double locationHeight = 0.0;

            // 真北の角度を探す
            double dueNorthDeg = 0.0;       // 真北の角度(deg)
            int result = DueNorthSearch(doc, out dueNorthDeg);
            if(result != 1)
            {
                if(result == 0)
                    MessageBox.Show("真北が設定されていません。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                else if(result == -1)
                    MessageBox.Show("真北が図面中に複数あります。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);

                return;
            }

            // 観測地点の緯度、日赤緯、測定面の高さ
            if(!GetShadowBaseData(out latitudeDeg, out solarDeclinationDeg, out locationHeight))
                return;

            // 高さの文字のリスト取得
            var heightTextShapeList = new List<TextShape>();
            int count = GetWithAttributesTextShapeList(doc, attributesValueZ, heightTextShapeList);
            if(count == 0)
            {
                MessageBox.Show("高さの設定された線がありません。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }

            double maxHeightValue = 0.0;
            if(!ChkHeightTextShapeList(heightTextShapeList, out maxHeightValue))
            {
                MessageBox.Show("高さのデータまたは測定面の高さが正しくありません。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }
 
            // 実行できるレイアウトの図面かチェック
            if(!CheckLayout(doc, heightTextShapeList))
            {
                return;
            }

            double calcRangeWidth = 1000.0;// 範囲を指定している場合の幅の初期値
            double calcRangeHeight = 1000.0;// 範囲を指定している場合の高さの初期値
            Point2d calcRangeOriginPoint = new Point2d();// 範囲を指定している場合の範囲の左下原点の初期値
            int intervalType = 60;// 計算間隔(単位：秒)
            // 範囲内を計算か
            if(rangeCalculationCheckBox.Checked)
            {
                if(calculationSpanLongRadioButtonA.Checked)
                    intervalType = 10;
                else if(calculationSpanShortRadioButtonA.Checked)
                    intervalType = 4;
                else
                {
                    Debug.Assert(false);
                }

                // 範囲の情報取得
                if(!GetCalcRangeData(dueNorthDeg, out calcRangeWidth, out calcRangeHeight, out calcRangeOriginPoint))
                    return;
            }
            else
            {
                if(calculationSpanLongRadioButtonA.Checked)
                    intervalType = 60;
                else if(calculationSpanShortRadioButtonA.Checked)
                    intervalType = 10;
                else
                {
                    Debug.Assert(false);
                }
            }

            // 高さの「ベース線」(高さが書かれている建物の縁を表す線)リストを取得
            List<List<BaseLine>> baseLinesList = GetHeightBaseLineShapeList(doc, heightTextShapeList);
            if(baseLinesList.Count == 0)
                return;

            // ベース線からなる多角形のリスト作成
            List<BaseLinePolygon> baseLinesPolygonList = new List<BaseLinePolygon>();
            List<BaseLinePolygon> subBaseLinesPolygonList = null;
            foreach(List<BaseLine> baseLines in baseLinesList)
            {
                subBaseLinesPolygonList = GetBaseLinesPolygonList(doc, locationHeight, baseLines);
                if(subBaseLinesPolygonList.Count > 0)
                    baseLinesPolygonList.AddRange(subBaseLinesPolygonList);
            }

            // ベース線の全図形範囲の中心点
            Point2d baseLinesBndBoxCenterPoint = GetLinesBoxCenterPoint(baseLinesList);

            // ベース線を真北方向の逆に回転させたBndBox(測定範囲の大きさの基準に使う)
            BndBox rotateBaseLinesBndBox = GetRotateLinesBndBox(baseLinesList, baseLinesBndBoxCenterPoint, -dueNorthDeg);

            // 8時からスタート
            // 点グリッドを用意して、求めたい一定の時間以上影になる部分の点を求める(最終的にはその点を中心とした四角形の点が影になりうる点として処理する)。
            // 最後に点の集まりの範囲を線で結んで等時間日影図とする。

            // 日影データを取得
            double latitudeRad = latitudeDeg * (Math.PI / 180d);
            double solarDeclinationRad = solarDeclinationDeg * (Math.PI / 180d); // 日赤緯(冬至: -23.45°、春秋分: 0°、夏至: 23.45°、任意時期)
            double shadowTime = minutes / 60.0;// 日影である時間(一定時間以上日影)(この時間以上日影である部分を求める)(単位：時)
            bool h9Toh15 = timeRangeComboBox.SelectedIndex == 1 ? true : false;// true:測定時間が9:00～15:00の場合
            double startSolarTime = 8;
            double endSolarTime = 16;
            // 測定範囲が9:00～15:00の場合
            if(h9Toh15)
            {
                startSolarTime = 9;
                endSolarTime = 15;
            }

            ////////////////////////////////////////////////////////
            // 配置スケールが1/200の部分図レイアウト
            // 用紙サイズもA2(420×594mm)
            // ピッチが660mm(0.66m)を基本に考える
            // スケールの分母が大きくなればピッチも大きくする
            // 用紙サイズも多くくなれば同じくピッチも大きくする
            // 範囲指定で範囲を小さくしたときはピッチも小さくできる
            ////////////////////////////////////////////////////////
            double basicScale = 1.0 / 200.0;
            double basicRangeWidth = 420.0  / basicScale;
            double basicRangeHeight = 594.0 / basicScale;
            double basicGridPitch = 660; // 測定ピッチ初期値(mm)

            // 2つのグリッドを用意。
            // 1つ目のグリッド点が影に含まれたらその点を中心にした周りの4点を影とする2つ目のグリッドを用意。
            // 最後にその2つ目のグリッドの影の点の集まりの周りを線で囲む(2つ目のグリッドは中心点が影と判断された四角いセルで影を考えるために用意)
            double heightMagnification = maxHeightValue / 100.0 + 1.0;// 建物の最大高による倍率調整値
            double rangeWidth1 = rotateBaseLinesBndBox.GetWidth() * 3.0 * heightMagnification;   // 測定範囲の幅(mm) n倍
            double rangeHeight1 = rotateBaseLinesBndBox.GetHeight() * 4.0 * heightMagnification; // 測定範囲の高さ(mm) m倍
            double gridRangeWidth = basicRangeWidth;
            double gridRangeHeight = basicRangeHeight;
            Point2d gridOriginPoint = new Point2d();// 測定範囲の左下基準点

            // 範囲を指定している場合
            if(rangeCalculationCheckBox.Checked)
            {
                gridRangeWidth = calcRangeWidth;
                gridRangeHeight = calcRangeHeight;
                gridOriginPoint = calcRangeOriginPoint;
            }
            else
            {
                gridRangeWidth = Math.Min(rangeWidth1, basicRangeWidth);// 小さいほうに合わせる
                gridRangeHeight = Math.Min(rangeHeight1, basicRangeHeight);
                gridOriginPoint = new Point2d(baseLinesBndBoxCenterPoint.X - gridRangeWidth * 0.5, baseLinesBndBoxCenterPoint.Y - rotateBaseLinesBndBox.GetHeight() * 0.5);// 測定範囲の左下基準点
                gridOriginPoint = app.Geometry.RotatePoint(gridOriginPoint, baseLinesBndBoxCenterPoint, dueNorthDeg);// 北方向に回転
            }

            // ピッチ
            double gridPitch = basicGridPitch * Math.Max(gridRangeWidth, gridRangeHeight) / Math.Max(basicRangeWidth, basicRangeHeight);
            // ビューが部分図の場合はスケールが関係ないのでそのまま
            // 用紙の場合はカレントの部分図レイアウトのスケールによる調整も行う
            TabView tabView = app.ActiveTabView;
            if(tabView is DrawingView drawingView)
            {
                // 用紙
                if(drawingView.Drawing is Paper)
                {
                    Layout currentlayout = doc.CurrentLayout;
                    double layoutScaleX = currentlayout.ScaleX;
                    gridPitch = gridPitch * basicScale / layoutScaleX;
                }
            }

            // 丸める
            gridPitch /= 1000.0;//mm->m
            gridPitch = Math.Round(gridPitch, 2, MidpointRounding.AwayFromZero);// 小数第3位を四捨五入 ex. 0.664->0.660
            gridPitch *= 1000.0;//m->mm 

            // 最大・最小値チェック
            gridPitch = Math.Min(gridPitch, customSettings.maxGridPitch);
            gridPitch = Math.Max(gridPitch, customSettings.minGridPitch);

            // 点グリッド用意
            int vCount = (int)(gridRangeHeight / gridPitch) + 1;
            int hCount = (int)(gridRangeWidth / gridPitch) + 1;
            Grid grid = new Grid(app, gridOriginPoint, vCount, hCount, gridPitch, dueNorthDeg);

            // 日影データリスト作成
            List<ShadowData> shadowDataList = GetShadowDataList(latitudeRad, solarDeclinationRad, intervalType, startSolarTime, endSolarTime);

            // 影の四角リスト作成
            List<List<Quadrilateral>> shadowQuadsList = new List<List<Quadrilateral>>(shadowDataList.Count);
            for(int i = 0; i < shadowDataList.Count; i++)
            {
                ShadowData shadowData = shadowDataList[i];
                double solarTime = shadowData.solarTime;
                double xMagnification = shadowData.xMagnification;
                double yMagnification = shadowData.yMagnification;

                // ベース線からなる多角形ごとに影の四角形領域のリスト作成
                int capacity = baseLinesPolygonList.Count;
                List<Quadrilateral> shadowQuads = new List<Quadrilateral>(capacity);// ベース線からできる影の四角リスト
                foreach(BaseLinePolygon baseLinePolygon in baseLinesPolygonList)
                {
                    // ベース線と高さ文字情報からできる影の四角形領域のリスト作成取得
                    var quads = GetQuadrilateralList(doc, baseLinePolygon, heightTextShapeList, solarTime, xMagnification, yMagnification, locationHeight, dueNorthDeg);
                    shadowQuads.AddRange(quads);
                }

                shadowQuadsList.Add(shadowQuads);
            }

            // ベース線（建物）の部分の点フラグセット
            SetInBaseLinesPolygonGridPointStat(grid, baseLinesPolygonList);

            // 指定した時間以上に影にあるところのグリッドの点にフラグをセット
            SetShadowFlgToGridPoints(shadowQuadsList, grid, shadowTime);

            // ここまででグリッドの各点で影となった点がその点を中心にして影となる4点のセルを表すグリッドを作成
            Grid shadowGrid = MakeShadowAreaGrid(grid);

            // 影の範囲を表す図形作成 ※測定範囲の境界部分はその先も影の可能性があるので線で結ばないようにする
            bool showCheckGridPoints = false;// true 確認用なので描画までに時間がかかる
            string shadowTimeText = shadowTime.ToString("F1");
            List<Shape> shadowShapes = CreateShadowAreaShapes(shadowGrid, showCheckGridPoints, shadowTimeText, baseLinesBndBoxCenterPoint);
                

            ///////////////////////////////////////////////////////////
            // UNDO 可能な操作を開始
            doc.UndoManager.BeginUndoUnit();

            // 図形追加
            Drawing currentDrawing = doc.CurrentDrawing;	    // カレント部分図の取得
            ArrayList addGroupShapeArray = new ArrayList();     // グループ化するための図形リスト

            for(int i = 0; i < shadowShapes.Count; i++)
            {
                Shape shape = shadowShapes[i];

                // レイヤのセット
                Layer currentLayer = doc.LayerTable.CurrentLayer;
                shape.Layer = currentLayer;

                // 追加
                Shape addShape = currentDrawing.Shapes.Add(shape);

                // グループ
                addGroupShapeArray.Add(addShape);
            }

            // グループ化(グリッド点を描画しないとき)
            if(addGroupShapeArray.Count > 0 && !showCheckGridPoints)
            {
                Shape shapeGroup = currentDrawing.Shapes.AddDrawGroup((Shape[])addGroupShapeArray.ToArray(typeof(Shape)), false);
            }

            // UNDO 可能な操作を終了
            doc.UndoManager.EndUndoUnit();
            /////////////////////////////////////////////////////////////

            // ピッチの表示
            double mGridPitch = gridPitch / 1000.0;
            string textPitch = mGridPitch.ToString();
            pitchTextBox.Text = textPitch;
        }


        /// <summary>
        /// 範囲の情報取得
        /// </summary>
        /// <param name="dueNorthDeg"></param>
        /// <param name="calcRangeWidth"></param>
        /// <param name="calcRangeHeight"></param>
        /// <param name="calcRangeOriginPoint"></param>
        /// <returns></returns>
        bool GetCalcRangeData(double dueNorthDeg, out double calcRangeWidth, out double calcRangeHeight, out Point2d calcRangeOriginPoint)
        {
            calcRangeWidth = 1000.0;// 範囲を指定している場合の幅の初期値
            calcRangeHeight = 1000.0;// 範囲を指定している場合の高さの初期値
            calcRangeOriginPoint = new Point2d();// 範囲を指定している場合の範囲の左下原点の初期値

            PolylineShape calcRangeShape = FindCalcRange();
            if(calcRangeShape == null)
            {
                MessageBox.Show("範囲が見つかりません。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                return false;
            }

            // 点列
            Point2d[] points = calcRangeShape.Points;

            // 傾きチェック
            // 真北の線の角度
            double dueNorthLineAngle = app.Geometry.WrapAngleTo2PI(app.Geometry.DegreeToRadian(dueNorthDeg + 90.0));
            LineShape chkLine = app.ShapeFactory.CreateLine(points[0], points[1]);
            double chkLineAngle1 = app.Geometry.WrapAngleTo2PI(app.Geometry.DegreeToRadian(chkLine.Angle));
            double chkLineAngle2 = app.Geometry.WrapAngleTo2PI(app.Geometry.DegreeToRadian(chkLine.Angle + 90.0));
            double chkLineAngle3 = app.Geometry.WrapAngleTo2PI(app.Geometry.DegreeToRadian(chkLine.Angle + 180.0));
            double chkLineAngle4 = app.Geometry.WrapAngleTo2PI(app.Geometry.DegreeToRadian(chkLine.Angle + 270.0));

            // 平行または垂直ならOK
            bool isParallel1 = app.Geometry.IsRadianEquals(dueNorthLineAngle, chkLineAngle1);
            bool isParallel2 = app.Geometry.IsRadianEquals(dueNorthLineAngle, chkLineAngle2);
            bool isParallel3 = app.Geometry.IsRadianEquals(dueNorthLineAngle, chkLineAngle3);
            bool isParallel4 = app.Geometry.IsRadianEquals(dueNorthLineAngle, chkLineAngle4);

            if(!isParallel1 && !isParallel2 && !isParallel3 && !isParallel4)
            {
                MessageBox.Show("範囲の傾きが真北の方向と一致していません。作成しなおしてください。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                return false;
            }

            // 左下の点の座標を取得、長方形を保っているかもチェック
            //Nullable<Point2d> bottomLeftPoint = calcRangeShape.GetRectanglePoint(Alignment.BottomLeft);
            //Nullable<double> rectangleWidth = calcRangeShape.GetRectangleWidth();
            //Nullable<double> rectangleHeight = calcRangeShape.GetRectangleHeight();
            // 真北の角度分逆に回転して水平にしたときの左下の点を求める
            Point2d centerPoint = CenterPoint(points[0], points[2]);
            int pointsCount = points.Length;
            Point2d[] points2 = new Point2d[pointsCount];
            for(int i = 0; i < pointsCount; i++)
            {
                points2[i] = app.Geometry.RotatePoint(points[i], centerPoint, -dueNorthDeg);
            }

            PolylineShape polylineShape2 = app.ShapeFactory.CreatePolyline(points2);
            Nullable<Point2d> bottomLeftPoint = polylineShape2.GetRectanglePoint(Alignment.BottomLeft);
            Nullable<double> rectangleWidth = polylineShape2.GetRectangleWidth();
            Nullable<double> rectangleHeight = polylineShape2.GetRectangleHeight();

            if(bottomLeftPoint == null || rectangleWidth == null || rectangleHeight == null
                || customSettings.miniCalcRangeSize - rectangleWidth > RcPrecisionConfusion || rectangleWidth - customSettings.maxCalcRangeSize > RcPrecisionConfusion
                || customSettings.miniCalcRangeSize - rectangleHeight > RcPrecisionConfusion || rectangleHeight - customSettings.maxCalcRangeSize > RcPrecisionConfusion)
            {
                MessageBox.Show("範囲の四角形が不正です。作成しなおしてください。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                return false;
            }
 
            calcRangeOriginPoint = app.Geometry.RotatePoint((Point2d)bottomLeftPoint, centerPoint, dueNorthDeg);// 真北方向に合わせて回転した点
            calcRangeWidth = (double)rectangleWidth;
            calcRangeHeight = (double)rectangleHeight;

            return true;
        }


        /// <summary>
        /// ベース線で囲まれる多角形（建物の外周）に含まれる点は建物内とする
        /// </summary>
        /// <param name="grid"></param>
        /// <param name="baseLinesPolygonList"></param>
        private void SetInBaseLinesPolygonGridPointStat(Grid grid, List<BaseLinePolygon> baseLinesPolygonList)
        {
            GridPoint[,] gridPoints = grid.GetGridPoints();
            int vCount = grid.GetVCount();
            int hCount = grid.GetHCount();

            // ベース線で囲まれる多角形（建物の外周）に含まれる点に建物内というフラフセット
            foreach(BaseLinePolygon baseLinePolygon in baseLinesPolygonList)
            {
                List<Point2d> points = baseLinePolygon.polygonPoints;
                int pointsCount = points.Count;
                if(pointsCount < 3)
                    break;

                Point2d firstPoint = points.First();
                Point2d lastPoint = points.Last();
                int pointsCount2 = pointsCount;
                bool equalFirstLast = true;
                if(!app.Geometry.IsPointEquals(firstPoint, lastPoint))
                {
                    pointsCount2++;
                    equalFirstLast = false;
                }

                Point2d[] points2 = new Point2d[pointsCount2];
                for(int k = 0; k < pointsCount; k++)
                {
                    points2[k] = points[k];

                    // 始点と終点を結んで閉じる 
                    if(k == pointsCount - 1 && !equalFirstLast)
                        points2[k + 1] = points.First();
                }

                for(int i = 0; i < vCount; i++)
                {
                    for(int j = 0; j < hCount; j++)
                    {
                        // 多角形(建物)に含まれるか
                        // ※ app.Geometry.ContainsPointInPolygonでは時間がかかる
                        if(IsPointInPolygon(gridPoints[i, j].point, points2))
                            gridPoints[i, j].shadowStatus |= GridPointStatus.InStructure;// 建物内
                    }
                }
            }
        }

        /// <summary>
        /// 指定した点が多角形の内部点か
        /// </summary>
        /// <param name="p"></param>
        /// <param name="polyPoints"></param>
        /// <returns></returns>
        private bool IsPointInPolygon(Point2d p, Point2d[] polyPoints)
        {
            int pointLength = polyPoints.Length;

            Point2d p1, p2;
            bool inside = false;
            Point2d oldPoint = polyPoints[pointLength - 1];

            for(int i = 0; i < pointLength; i++)
            {
                Point2d newPoint = polyPoints[i];
                if(newPoint.X > oldPoint.X)
                {
                    p1 = oldPoint;
                    p2 = newPoint;
                }
                else
                {
                    p1 = newPoint;
                    p2 = oldPoint;
                }

                if((p1.X < p.X) == (p.X <= p2.X) && (p.Y - p1.Y) * (p2.X - p1.X) < (p2.Y - p1.Y) * (p.X - p1.X))
                {
                    inside = !inside;
                }

                oldPoint = newPoint;
            }

            return inside;
        }

        /// <summary>
        /// 指定した時間以上に影になるところのグリッドの点にフラグをセット
        /// </summary>
        /// <param name="shadowQuadsList"></param>
        /// <param name="orgGrid"></param>
        /// <param name="shadowTime">影である総時間(単位:時)</param>
        /// <returns></returns>
        private void SetShadowFlgToGridPoints(List<List<Quadrilateral>> shadowQuadsList, Grid grid, double shadowTime)
        {
            int vCount = grid.GetVCount();
            int hCount = grid.GetHCount();
            GridPoint[,] gridPoints = grid.GetGridPoints();
            int shadowTimeSeconds = ShadowTimes.RoundShadowTimeToSeconds(shadowTime);

            // あるグリッドの点が一定の時間以上影であるかみていく
            for(int i = 0; i < vCount; i++)
            {
                for(int j = 0; j < hCount; j++)
                {
                    // 調べるグリッドの点
                    Point2d point = gridPoints[i, j].point;

                    // 同じ場所でも日影と日なたが繰り返される場合もあることを考慮
                    // 指定点の日影時間情報
                    PinpointShadowAreaInfo pinpointShadowAreaInfo = CreatePinpointShadowAreaInfo(shadowQuadsList, "0", point);
                    int totalShadowTimeSeconds = pinpointShadowAreaInfo.GetTotalShadowTimesSeconds();
                    // 一定時間以上影ならフラグをセット
                    if(totalShadowTimeSeconds >= shadowTimeSeconds)
                        gridPoints[i, j].shadowStatus |= GridPointStatus.MoreThanATimeShadow;// 一定時間以上影
                }
            }
        }

        /// <summary>
        /// 指定した点の影情報作成
        /// </summary>
        /// <param name="shadowQuadsList"></param>
        /// <param name="numberText"></param>
        /// <param name="point"></param>
        /// <returns></returns>
        private PinpointShadowAreaInfo CreatePinpointShadowAreaInfo(List<List<Quadrilateral>> shadowQuadsList, string numberText, Point2d point)
        {
             // 指定点の日影時間情報
            PinpointShadowAreaInfo pinpointShadowAreaInfo = new PinpointShadowAreaInfo(numberText);

            double startShadowTime = 0.0;
            double endShadowTime = 0.0;
            bool shadowState = false;// 日影状態 false:日なた、true:日影

            // 時間毎の四角形群に指定点が含まれている合計時間を求める
            // 同じ場所でも日影と日なたが繰り返される場合もある
            int listCount = shadowQuadsList.Count;
            for(int i = 0; i < listCount; i++)
            {
                List<Quadrilateral> shadowQuads = shadowQuadsList[i];
                if(IsPointInShadowQuads(shadowQuads, point, out double solarTime))
                {
                    if(!shadowState)
                    {
                        shadowState = true;
                        startShadowTime = solarTime;
                    }

                    endShadowTime = solarTime;
                }
                else
                {
                    if(shadowState)
                    {
                        // 日影になる時間帯を追加
                        if(endShadowTime > startShadowTime)
                        {
                            ShadowTimes shadowTimes = new ShadowTimes(startShadowTime, endShadowTime);
                            pinpointShadowAreaInfo.AddShadowTimes(shadowTimes);
                        }
                    }

                    shadowState = false;
                }
            }

            // 最後まで影のままだった場合
            if(shadowState)
            {
                // 日影になる時間帯を追加
                if(endShadowTime > startShadowTime)
                {
                    ShadowTimes shadowTimes = new ShadowTimes(startShadowTime, endShadowTime);
                    pinpointShadowAreaInfo.AddShadowTimes(shadowTimes);
                }
            }

            return pinpointShadowAreaInfo;
        }

        /// <summary>
        /// 四角のリスト内に点が含まれているものがあるか
        /// </summary>
        /// <param name="shadowQuads"></param>
        /// <param name="point"></param>
        /// <param name="hitSolarTime">out</param>
        /// <returns></returns>
        private bool IsPointInShadowQuads(List<Quadrilateral> shadowQuads, Point2d point, out double hitSolarTime)
        {
            hitSolarTime = 0.0;

            // 四角のリスト
            int listCount = shadowQuads.Count;
            for(int i = 0; i < listCount; i++)
            {
                Quadrilateral quad = shadowQuads[i];

                // 四角内に含まれるか
                if(IsPointInPolygon(point, quad.GetPoints()))
                {
                    hitSolarTime = quad.solarTime;
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// 措定したグリッドの各点で影となった点がその点を中心にして影となる4点のセルを表すグリッドを作成
        /// （ピッチの半分縦横に移動した原点から1行1列増やしたグリッドを用意し、元のグリッドを重ねたときに
        /// 元のグリッドの点を中心とした四角いセルの四隅の角を新しい点の集まりとするグリッドを用意する。
        /// すなわち、元のグリッドの影の点を中心とした新しグリッドの4点を囲んだ部分は影だと考える）
        /// </summary>
        /// <param name="shadowCellCenterPointGrid"></param>
        /// <returns></returns>
        private Grid MakeShadowAreaGrid(Grid shadowCellCenterPointGrid)
        {
            int scpg_vCount = shadowCellCenterPointGrid.GetVCount();
            int scpg_hCount = shadowCellCenterPointGrid.GetHCount();
            double scpg_gridPitch = shadowCellCenterPointGrid.GetPitch();
            Point2d scpg_gridOriginPoint = shadowCellCenterPointGrid.GetOriginPoint();
            GridPoint[,] scpg_gridPoints = shadowCellCenterPointGrid.GetGridPoints();

            int vCount = scpg_vCount + 1;
            int hCount = scpg_hCount + 1;
            double gridPitch = scpg_gridPitch;
            Point2d gridOriginPoint = new Point2d(scpg_gridOriginPoint.X - gridPitch * 0.5, scpg_gridOriginPoint.Y - gridPitch * 0.5);
            double rotateAngle = shadowCellCenterPointGrid.GetRotateAngle();

            Grid shadowGrid = new Grid(app, gridOriginPoint, vCount, hCount, gridPitch, rotateAngle);

            // 建物内フラグと一定時間以上影(範囲外周線確定前)フラグをshadowGridにセット(shadowCellCenterPointGridの点をセルの中心と考えたときのセルの四隅の点をshadowGridの点にしていく)
            for(int v = 0; v < scpg_vCount; v++)
            {
                for(int h = 0; h < scpg_hCount; h++)
                {
                    // 一定時間以上影(範囲外周線確定前)フラグ
                    if((scpg_gridPoints[v, h].shadowStatus & GridPointStatus.MoreThanATimeShadow) == GridPointStatus.MoreThanATimeShadow)
                    {
                        AddShadowStatusTo4Points(shadowGrid, v, h, GridPointStatus.MoreThanATimeShadow);
                    }
                   
                    // 建物内フラグ
                    if((scpg_gridPoints[v, h].shadowStatus & GridPointStatus.InStructure) == GridPointStatus.InStructure)
                    {
                        AddShadowStatusTo4Points(shadowGrid, v, h, GridPointStatus.InStructure);
                    }
                }
            }

            return shadowGrid;            
        }

        /// <summary>
        /// 指定した点を左下として他の4隅の点に同じフラグをセット
        /// </summary>
        /// <param name="shadowGrid"></param>
        /// <param name="v"></param>
        /// <param name="h"></param>
        /// <param name="gridPointStatus"></param>
        private void AddShadowStatusTo4Points(Grid shadowGrid, int v, int h, GridPointStatus gridPointStatus)
        {
            int vCount = shadowGrid.GetVCount();
            int hCount = shadowGrid.GetHCount();
            GridPoint[,] gridPoints = shadowGrid.GetGridPoints();

            Debug.Assert(v >= 0 && v <= vCount && h >= 0 && h <= hCount);

            gridPoints[v, h].shadowStatus |= gridPointStatus;
            gridPoints[v+1, h].shadowStatus |= gridPointStatus;
            gridPoints[v+1, h+1].shadowStatus |= gridPointStatus;
            gridPoints[v, h+1].shadowStatus |= gridPointStatus;
        }


        /// <summary>
        /// 一定時間以上影になる範囲を表す図形リスト作成
        /// 影フラグのある点の集まりを囲むように線で結ぶ
        /// （最外周の影になるグリッドの点同士を結ぶ線の中点を線で繋いで囲む）
        /// </summary>
        /// <param name="shadowGrid"></param>
        /// <param name="addShadowAreaPointMarker"></param>
        /// <param name="shadowTimeText"></param>
        /// <param name="basePoint"></param>
        /// <returns></returns>
        private List<Shape> CreateShadowAreaShapes(Grid shadowGrid, bool addShadowAreaPointMarker, string shadowTimeText, Point2d basePoint)
        {
            int vCount = shadowGrid.GetVCount();
            int hCount = shadowGrid.GetHCount();
            GridPoint[,] gridPoints = shadowGrid.GetGridPoints();
 
            // 一定の時間以上影になる範囲を表す図形の作成
            List<Shape> shadowShapes = new List<Shape>();
            List<LinePoints> linePointsList = new List<LinePoints>();

            // 確認用のグリッド中の影の点を追加する場合
            if(addShadowAreaPointMarker)
                AddShadowAreaPointMarkerShape(shadowShapes, shadowGrid);

            // 影の塊が複数ある場合に対応
            bool loopEnd1 = false;
            while(!loopEnd1)
            {
                // 最初の影の点を探す
                int startVIndex = 0;
                int startHIndex = 0;
                if(!GetFirstShadowPointOnOutlineIndex(shadowGrid, ref startVIndex, ref startHIndex))
                {
                    loopEnd1 = true;
                    break;
                }

                // 最初の点から次の影の縁(影の範囲のアウトライン上)の点を探す
                //List<Point2d> pointList = new List<Point2d>();
                List<PointIndexVH> pointIndexVHList = new List<PointIndexVH>();
                PointIndexVH pointIndexVH;

                int v = startVIndex;
                int h = startHIndex;
                JointToThisPointVector jointToThisPointVector = JointToThisPointVector.ToR;

                // 最初の1点目を追加
                //Point2d firstShadowPointOnOutline = gridPoints[startVIndex, startHIndex].point;
                gridPoints[startVIndex, startHIndex].shadowStatus |= GridPointStatus.OnMoreThanATimeShadowAreaOutline; // 影範囲の外周上の点フラグOn
                pointIndexVH = new PointIndexVH(startVIndex, startHIndex);
                pointIndexVHList.Add(pointIndexVH);

                // 2点目以降の点列
                int loopCounter = 0;
                int maxLoopCount = vCount * hCount;// 無限ループ防止の念のため
                bool loopEnd2 = false;
                while(!loopEnd2)
                {
                    // 次の点のインデックスを求める
                    bool finded = GetNextShadowPosintIndex(shadowGrid, ref jointToThisPointVector, ref v, ref h);
                    // 次の点が見つかった場合
                    if(finded)
                    {
                        // 既に影範囲の外周上の点だったらここで終了
                        if((gridPoints[v, h].shadowStatus & GridPointStatus.OnMoreThanATimeShadowAreaOutline) == GridPointStatus.OnMoreThanATimeShadowAreaOutline)
                            loopEnd2 = true;

                        Point2d shadowPointOnOutline = gridPoints[v, h].point;
                        gridPoints[v, h].shadowStatus |= GridPointStatus.OnMoreThanATimeShadowAreaOutline; // 影範囲の外周上の点フラグOn

                        pointIndexVH = new PointIndexVH(v, h);
                        pointIndexVHList.Add(pointIndexVH);
                    }
                    // 次の点が見つからなかった場合は終了
                    else
                    {
                        // 一回り大きく探してもいいかも。。。
                        loopEnd2 = true;
                    }

                    loopCounter++;
                    if(loopCounter > maxLoopCount)
                    {
                        loopEnd2 = true;
                        loopEnd1 = true;
                    }
                }

                // インデックスリストから線分で範囲を結ぶ。グリッド範囲の際(きわ)の点は線にしない。
                // 線分の座標リストに追加する
                AddLinePointsList(linePointsList, shadowGrid, pointIndexVHList);

                // 確定した影の範囲（多角形）の内側になる点にフラグセット
                SetInMoreThanATimeShadowAreaOutlineFlg(shadowGrid, pointIndexVHList);

                //loopEnd1 = true; // TEST：一階のループで止める
            }

            // 線分の座標リストから図形を作成して図形リストに追加する
            // 線分の中点同士を結ぶ線にする
            AddLinePointsListToShadowShapes(linePointsList, shadowShapes, shadowTimeText, basePoint);

            return shadowShapes;
        }


        /// <summary>
        /// 次の一定時間影になっている点のインデックスと向きを取得
        /// </summary>
        /// <param name="grid"></param>
        /// <param name="jointToThisPointVector">前の点から指定したインデックスの点に向かった向き(in)と次の点への向き(our)</param>
        /// <param name="v">前の点から指定したインデックスv(in)と次の点のインデックスv(out)</param>
        /// <param name="h">前の点から指定したインデックスh(in)と次の点のインデックスh(out)</param>
        /// <returns></returns>
        private bool GetNextShadowPosintIndex(Grid grid, ref JointToThisPointVector jointToThisPointVector, ref int v, ref int h)
        {
            int startSearchPosition = 0;// 次に繋げる影の点の検索開始ポジション
            // 1---2---3
            // |   |   |
            // 8---*---4
            // |   |   |
            // 7---6---5
            int positinV = 0;
            int positinH = 0;

            // 今のインデックスの点がどの向きに向かって接続された点かで処理を分ける
            // ※ルール：今の点(*)を中心に向かって来た方向から反時計回りに90度の点から時計回りで見ていって最初の影の点を次の影の点とする(グリッドの左下(v, h = 0, 0)から最初の影の点を探してその点からスタートしているから)
            switch(jointToThisPointVector)
            {
                // 右へ向かって来た場合
                case JointToThisPointVector.ToR:
                    startSearchPosition = 1;// 左上
                    break;
                // 右下へ向かって来た場合
                case JointToThisPointVector.ToRB:
                    startSearchPosition = 2;// 上
                    break;
                // 下へ向かって来た場合
                case JointToThisPointVector.ToB:
                    startSearchPosition = 3;// 右上
                    break;
                // 左下へ向かって来た場合
                case JointToThisPointVector.ToLB:
                    startSearchPosition = 4;// 右
                    break;
                // 左へ向かって来た場合
                case JointToThisPointVector.ToL:
                    startSearchPosition = 5;// 右下
                    break;
                // 左上へ向かって来た場合
                case JointToThisPointVector.ToLT:
                    startSearchPosition = 6;// 下
                    break;
                // 上へ向かって来た場合
                case JointToThisPointVector.ToT:
                    startSearchPosition = 7;// 左下
                    break;
                // 右上へ向かって来た場合
                case JointToThisPointVector.ToRT:
                    startSearchPosition = 8;// 左
                    break;
            }

            // 時計回りに探して最初の影の点を見つける
            for(int i = 0; i < 7; i++)
            {
                int nextPosition = i + startSearchPosition;

                if(nextPosition > 8)
                    nextPosition -= 8;

                GetShadowPointPositin(nextPosition, v, h,  ref positinV, ref positinH);

                if(IsMoreThanATimeShadowPoint(grid, positinV, positinH))
                {
                    v = positinV;
                    h = positinH;

                    Debug.Assert(nextPosition >= 1 && nextPosition <= 8);

                    switch(nextPosition)
                    {
                        // 左上へ向かった場合
                        case 1:
                            jointToThisPointVector = JointToThisPointVector.ToLT;
                            break;
                        // 上へ向かった場合
                        case 2:
                            jointToThisPointVector = JointToThisPointVector.ToT;
                            break;
                        // 右上へ向かった場合
                        case 3:
                            jointToThisPointVector = JointToThisPointVector.ToRT;
                            break;
                        // 右へ向かった場合
                        case 4:
                            jointToThisPointVector = JointToThisPointVector.ToR;
                            break;
                        // 右下へ向かった場合
                        case 5:
                            jointToThisPointVector = JointToThisPointVector.ToRB;
                            break;
                        // 下へ向かった場合
                        case 6:
                            jointToThisPointVector = JointToThisPointVector.ToB;
                            break;
                        // 左下へ向かった場合
                        case 7:
                            jointToThisPointVector = JointToThisPointVector.ToLB;
                            break;
                        // 左へ向かった場合
                        case 8:
                            jointToThisPointVector = JointToThisPointVector.ToL;
                            break;
                    }

                    return true;
                }

            }

            return false;
        }

        /// <summary>
        /// 指定した場所のグリッドの点のインデックスを取得
        /// </summary>
        /// <param name="nextPosition"></param>
        /// <param name="v"></param>
        /// <param name="h"></param>
        /// <param name="positinV"></param>
        /// <param name="positinH"></param>
        private void GetShadowPointPositin(int nextPosition, int v, int h, ref int positinV, ref int positinH)
        {
            Debug.Assert(nextPosition >= 1 && nextPosition <= 8);

            // nextPosition
            // 1---2---3
            // |   |   |
            // 8---*---4
            // |   |   |
            // 7---6---5
            switch(nextPosition)
            {
                case 1:
                    positinV = v + 1;
                    positinH = h - 1;
                    break;
                case 2:
                    positinV = v + 1;
                    positinH = h;
                    break;
                case 3:
                    positinV = v + 1;
                    positinH = h + 1;
                    break;
                case 4:
                    positinV = v;
                    positinH = h + 1;
                    break;
                case 5:
                    positinV = v - 1;
                    positinH = h + 1;
                    break;
                case 6:
                    positinV = v - 1;
                    positinH = h;
                    break;
                case 7:
                    positinV = v - 1;
                    positinH = h - 1;
                    break;
                case 8:
                    positinV = v;
                    positinH = h - 1;
                    break;
            }

        }


        /// <summary>
        /// 指定したインデックスの点が一定時間以上影の点か
        /// </summary>
        /// <param name="grid"></param>
        /// <param name="v"></param>
        /// <param name="h"></param>
        /// <returns></returns>
        private bool IsMoreThanATimeShadowPoint(Grid grid, int v, int h)
        {
            bool isShadow = false;

            GridPoint[,] gridPoints = grid.GetGridPoints();
            int vCount = grid.GetVCount();
            int hCount = grid.GetHCount();

            if(v < 0 || v >= vCount)
                return isShadow;

            if(h < 0 || h >= hCount)
                return isShadow;

            GridPointStatus gridPointStatus = gridPoints[v, h].shadowStatus;

            // 一定時間以上影の点なら
            if((gridPointStatus & GridPointStatus.MoreThanATimeShadow) == GridPointStatus.MoreThanATimeShadow)
                isShadow = true;

            return isShadow;
        }

        /// <summary>
        /// 最初の影の縁(影の範囲のアウトライン上)の点を決めてそのグリッド点のインデックスを返す
        /// 下の行から左から右に探して、なければ1行上がってまた左から右に探すを繰り返す)
        /// </summary>
        /// <param name="grid"></param>
        /// <param name="v"></param>
        /// <param name="h"></param>
        /// <returns></returns>
        private bool GetFirstShadowPointOnOutlineIndex(Grid grid, ref int v, ref int h)
        {
            GridPoint[,] gridPoints = grid.GetGridPoints();
            int vCount = grid.GetVCount();
            int hCount = grid.GetHCount();

            for(int i = 0; i < vCount; i++)
            {
                for(int j = 0; j < hCount; j++)
                {
                    GridPointStatus gridPointStatus = gridPoints[i, j].shadowStatus;

                    // 既に一定時間以上影の範囲の外周線上や中として確定している点はスキップ
                    if((gridPointStatus & GridPointStatus.OnMoreThanATimeShadowAreaOutline) == GridPointStatus.OnMoreThanATimeShadowAreaOutline)
                        continue;

                    if((gridPointStatus & GridPointStatus.InMoreThanATimeShadowAreaOutline) == GridPointStatus.InMoreThanATimeShadowAreaOutline) 
                        continue;


                    // 一定時間以上影の点
                    if((gridPointStatus & GridPointStatus.MoreThanATimeShadow) == GridPointStatus.MoreThanATimeShadow)
                    {
                        v = i;
                        h = j;

                        return true;
                    }
                }
            }

            return false;
        }


        /// <summary>
        /// 一定時間影の点のインデックスリストからその範囲を表す線の始点と終点をリストに追加する
        /// </summary>
        /// <param name="lineShapeList"></param>
        /// <param name="grid"></param>
        /// <param name="pointIndexVHList"></param>
        private void AddLinePointsList(List<LinePoints> linePointsList, Grid grid, List<PointIndexVH> pointIndexVHList)
        {
            int vCount = grid.GetVCount();
            int hCount = grid.GetHCount();

            GridPoint[,] gridPoints = grid.GetGridPoints();
            Point2d point1;
            Point2d point2;
            Point2d oldEndPoint = new Point2d();
            GridPointStatus gridPointStatus1;
            GridPointStatus gridPointStatus2;
            GridPointStatus oldEndGridPointStatus = GridPointStatus.NonCheck;

            bool polyStart = true;// つながる線の最初の点
            int pointIndexVHListCount = pointIndexVHList.Count;

            for(int i = 0; i < pointIndexVHListCount; i++)
            {
                PointIndexVH pointIndexVH = pointIndexVHList[i];
                int v = pointIndexVH.v;
                int h = pointIndexVH.h;

                if(polyStart)
                {
                    oldEndPoint = gridPoints[v, h].point;
                    polyStart = false;

                    oldEndGridPointStatus = gridPoints[v, h].shadowStatus;

                    continue;
                }
                else
                {
                    point1 = oldEndPoint;
                    point2 = gridPoints[v, h].point;

                    gridPointStatus1 = oldEndGridPointStatus;
                    gridPointStatus2 = gridPoints[v, h].shadowStatus;
                }

                // 線を作成してリストに追加
                // グロッド範囲の際(きわ)の点の場合はその先がある可能性もあるので線は作成しない
                if(!(v == 0 || h == 0 || v == vCount - 1 || h == hCount - 1))
                {
                    // 建物内の場合も線は作成しない
                    if(!((gridPointStatus1 & GridPointStatus.InStructure) == GridPointStatus.InStructure || (gridPointStatus2 & GridPointStatus.InStructure) == GridPointStatus.InStructure))
                    {
                        LinePoints linePoints = new LinePoints(point1, point2);
                        linePointsList.Add(linePoints);
                    }
                }

                oldEndPoint = point2;
                oldEndGridPointStatus = gridPointStatus2;
            }

        }

        /// <summary>
        /// 線分の座標リストから図形を作成して図形リストに追加する
        /// 線分の中点同士を結ぶ線にして直角の角を減らす
        /// </summary>
        /// <param name="linePointsList"></param>
        /// <param name="shadowShapes"></param>
        /// <param name="shadowTimeText"></param>
        /// <param name="basePoint"></param>
        private void AddLinePointsListToShadowShapes(List<LinePoints> linePointsList, List<Shape> shadowShapes, string shadowTimeText, Point2d basePoint)
        {
            //ドキュメントの取得
            Document doc = app.ActiveDocument;

            List<LineShape> aLines = new List<LineShape>();// 繋がっている線のリスト

            // 中点同士で結ぶ
            int linePointsListCount = linePointsList.Count;
            for(int i = 0; i < linePointsListCount - 1; i++)
            {
                LinePoints linePoints1 = linePointsList[i];
                LinePoints linePoints2 = linePointsList[i + 1];

                // 1つ目と2つ目の線が繋がっていれば、中点同士を結んだ線を作成する
                if(IsPointEquals(linePoints1.point2, linePoints2.point1))
                {
                    Point2d point1 = CenterPoint(linePoints1.point1, linePoints1.point2);
                    Point2d point2 = CenterPoint(linePoints2.point1, linePoints2.point2);
                    LineShape lineShape = app.ShapeFactory.CreateLine(point1, point2);
                    shadowShapes.Add(lineShape);
                    doc.BasicShapeSettings.CopyToShape(lineShape);
                    aLines.Add(lineShape);
                }
                else
                {
                    // 途中で切れた場合、ここで一旦影の時間を表す文字を一緒に描き出す。
                    AddShadowTimeTextOnShadowAreaLines(shadowShapes, basePoint, aLines, shadowTimeText);
                    // リストクリア
                    aLines.Clear();
                }
            }

            // 時間を表す文字を描き出す
            if(aLines.Count > 0)
                AddShadowTimeTextOnShadowAreaLines(shadowShapes, basePoint, aLines, shadowTimeText);
 
        }

        /// <summary>
        /// 時間を表す文字を影の範囲を表す線上に追加
        /// </summary>
        /// <param name="shadowShapes"></param>
        /// <param name="basePoint"></param>
        /// <param name="aLines"></param>
        /// <param name="shadowTimeText"></param>
        private void AddShadowTimeTextOnShadowAreaLines(List<Shape> shadowShapes, Point2d basePoint, List<LineShape> aLines, string shadowTimeText)
        {
            //ドキュメントの取得
            Document doc = app.ActiveDocument;

            // 線の上に影の時間を描く
            if(aLines.Count > 3 && GetTimeTextPointAndAngle(aLines, basePoint, out Point2d locatePoint, out double locateAngle))
            {
                TextShape text = app.ShapeFactory.CreateText(shadowTimeText, locatePoint, locateAngle);
                doc.TextSettings.CopyToShape(text);

                text.DirectionVertical = false;// 横書き
                text.FontHeight = GetTextFontHeight();
                text.Alignment = Alignment.BottomMiddle;// 配置点：中下
                shadowShapes.Add(text);
            }
        }

        /// <summary>
        /// 確定した影の範囲（多角形）の内側になる点にフラグセット
        /// </summary>
        /// <param name="grid"></param>
        /// <param name="pointIndexVHList"></param>
        private void SetInMoreThanATimeShadowAreaOutlineFlg(Grid grid, List<PointIndexVH> pointIndexVHList)
        {
            int pointIndexVHListCount = pointIndexVHList.Count;
            if(pointIndexVHListCount < 3)
                return;

            GridPoint[,] gridPoints = grid.GetGridPoints();
 
            // 最初と最後の点座標を取得
            PointIndexVH pointIndexVH_First = pointIndexVHList.First();
            int v_First = pointIndexVH_First.v;
            int h_First = pointIndexVH_First.h;

            PointIndexVH pointIndexVH_Last = pointIndexVHList.Last();
            int v_Last = pointIndexVH_Last.v;
            int h_Last = pointIndexVH_Last.h;

            Point2d point_First = gridPoints[v_First, h_First].point;
            Point2d point_Last = gridPoints[v_Last, h_Last].point;
            GridPointStatus gridPointStatus_First = gridPoints[v_First, h_First].shadowStatus;

            // 閉じているか
            bool isClose = false;
            if(IsPointEquals(point_First, point_Last))
                isClose = true;

            // 点列作成
            int pointListSize = pointIndexVHListCount;
            if(!isClose)
                pointListSize++;

            Point2d[] shadowOutlinePoints = new Point2d[pointListSize];
            for(int i = 0; i < pointListSize; i++)
            {
                Point2d point;// = new Point2d();
                if(!isClose && i == pointListSize - 1)
                {
                    point = point_First;
                }
                else
                {
                    PointIndexVH pointIndexVH = pointIndexVHList[i];
                    int v = pointIndexVH.v;
                    int h = pointIndexVH.h;
                    point = gridPoints[v, h].point;
                }

                shadowOutlinePoints[i] = point;
            }

            Point2d gridPoint;
            int vCount = grid.GetVCount();
            int hCount = grid.GetHCount();
            for(int i = 0; i < vCount; i++)
            {
                for(int j = 0; j < hCount; j++)
                {
                    // 一定時間以上影の点でなければスキップ
                    GridPointStatus gridPointStatus = gridPoints[i, j].shadowStatus;
                    if(!((gridPointStatus & GridPointStatus.MoreThanATimeShadow) == GridPointStatus.MoreThanATimeShadow))
                        continue;

                    // 一定時間以上影の範囲の外周線上もスキップ
                    if((gridPointStatus & GridPointStatus.OnMoreThanATimeShadowAreaOutline) == GridPointStatus.OnMoreThanATimeShadowAreaOutline)
                        continue;

                    // 多角形の内部点か
                    gridPoint = gridPoints[i, j].point;
                    //int contains = app.Geometry.ContainsPointInPolygon(gridPoint, shadowOutlinePoints);
                    //if(Math.Abs(contains) == 2)
                    if(IsPointInPolygon(gridPoint, shadowOutlinePoints))
                    {
                        gridPoints[i, j].shadowStatus |= GridPointStatus.InMoreThanATimeShadowAreaOutline;// 範囲の内側フラグOn
                    }

                }
            }
        }

        /// <summary>
        /// 2点以上の点列からポリライン作成して図形リストに追加する
        /// </summary>
        /// <param name="shapeList"></param>
        /// <param name="pointList"></param>
        private bool AddPolylineShape(List<Shape> shapeList, List<Point2d> pointList)
        {
            if(pointList.Count > 1)
            {
                Point2d[] polylinePoints = new Point2d[pointList.Count];

                for(int i = 0; i < pointList.Count; i++)
                    polylinePoints[i] = pointList[i];

                PolylineShape polylineShape = app.ShapeFactory.CreatePolyline(polylinePoints);

                shapeList.Add(polylineShape);

                return true;
            }

            return false;
        }

        /// <summary>
        /// 影の範囲内のグリッドの点を点マーカー図形にしてリストに追加
        /// </summary>
        /// <param name="shadowShapes"></param>
        /// <param name="grid"></param>
        private void AddShadowAreaPointMarkerShape(List<Shape> shadowShapes, Grid grid)
        {
            int vCount = grid.GetVCount();
            int hCount = grid.GetHCount();
            GridPoint[,] gridPoints = grid.GetGridPoints();

            for(int i = 0; i < vCount; i++)
            {
                for(int j = 0; j < hCount; j++)
                {
                    // 建物内
                    if((gridPoints[i, j].shadowStatus & GridPointStatus.InStructure) == GridPointStatus.InStructure)
                    {
                        PointMarkerShape gridPointMarker = app.ShapeFactory.CreatePointMarker(gridPoints[i, j].point);
                        gridPointMarker.ColorNumber = 5;
                        gridPointMarker.MarkerType = PointMarkerType.FilledCircle;
                        gridPointMarker.Size = 50.0;
                        shadowShapes.Add(gridPointMarker);
                    }
                    // 一定時間以上影
                    else if((gridPoints[i, j].shadowStatus & GridPointStatus.MoreThanATimeShadow) == GridPointStatus.MoreThanATimeShadow)
                    {
                        PointMarkerShape gridPointMarker = app.ShapeFactory.CreatePointMarker(gridPoints[i, j].point);
                        gridPointMarker.ColorNumber = 7;
                        gridPointMarker.MarkerType = PointMarkerType.FilledCircle;
                        gridPointMarker.Size = 50.0;
                        shadowShapes.Add(gridPointMarker);
                    }
                    else
                    {
                        PointMarkerShape gridPointMarker = app.ShapeFactory.CreatePointMarker(gridPoints[i, j].point);
                        gridPointMarker.ColorNumber = 6;
                        gridPointMarker.MarkerType = PointMarkerType.FilledCircle;
                        gridPointMarker.Size = 50.0;
                        shadowShapes.Add(gridPointMarker);
                    }
                }
            }
        }

        /// <summary>
        /// 「範囲を指定」コマンドのマウス移動イベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void CommandDrawRange_MouseMove(Object sender, CommandMouseMoveEventArgs e)
        {
            Command command = (Command)sender;
            // Command.GetTemporaryParameterItemByID を使用してコマンドの一時的なパラメータアイテムを取得します。
            // 一時的なパラメータアイテムからは、すでにパラメータの入力が確定している場合は確定している入力値が、
            // 入力が確定していない場合はマウスの現在位置などの一時的な値が取得できます。

            PointParameterItem pointAParameterItem = (PointParameterItem)command.GetParameterItemByID(cornerPointAParameterId);
            PointParameterItem pointBParameterItem = (PointParameterItem)command.GetParameterItemByID(cornerPointBParameterId);
            if(pointAParameterItem.IsEmpty && pointBParameterItem.IsEmpty)
            {
                 return;
            }

            // 一時パラメータを取得する
            pointAParameterItem = (PointParameterItem)command.GetTemporaryParameterItemByID(cornerPointAParameterId);
            if(pointAParameterItem.IsEmpty)
                return;

            pointBParameterItem = (PointParameterItem)command.GetTemporaryParameterItemByID(cornerPointBParameterId);
            if(pointBParameterItem.IsEmpty)
                return;

            // 点A
            Point2d locatePointA = pointAParameterItem.Point;
            // 点B
            Point2d locatePointB = pointBParameterItem.Point;

            //ドキュメントの取得
            Document doc = app.ActiveDocument;
            // 現在のレイアウト
            Layout layout = doc.CurrentLayout;
            // 部分図の取得
            Drawing drawing = doc.CurrentDrawing;

//          Point2d localPickPoint = layout.TransformPointWorldToLocal(pickPoint);// アクティブなレイアウトに合わせた座標へ変換
 
            // 真北の角度
            double dueNorthDeg = dueNorthDegForDrawRangeCommand;

            // 図形作成
            PolylineShape polylineShape = CreateShapesForDrawRangeCommand(command, drawing, locatePointA, locatePointB, dueNorthDeg);
            if(polylineShape == null)
                return;

            // ラバーバンド用の図形リスト
            ArrayList rubberBandShapes = new ArrayList();
            rubberBandShapes.Add((Shape)polylineShape);

            try
            {
                // ラバーバンド描画
                command.DrawShapeRubberBand((Shape[])rubberBandShapes.ToArray(typeof(Shape)));
            }
            catch
            {
                throw;
            }

        }

        /// <summary>
        /// 「範囲を指定」コマンドのパラメータ変更イベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void CommandDrawRange_ParameterChanged(Object sender, CommandParameterChangedEventArgs e)
        {
      
            Command command = (Command)sender;

            PointParameterItem pointAParameterItem = (PointParameterItem)command.GetParameterItemByID(cornerPointAParameterId);
            PointParameterItem pointBParameterItem = (PointParameterItem)command.GetParameterItemByID(cornerPointBParameterId);
            if(pointAParameterItem.IsEmpty || pointBParameterItem.IsEmpty)
            {
                e.CommandState = CommandState.Continue;
                return;
            }

            // 点A
            Point2d locatePointA = pointAParameterItem.Point;
            // 点B
            Point2d locatePointB = pointBParameterItem.Point;
 
            //ドキュメントの取得
            Document doc = app.ActiveDocument;
            //部分図の取得
            Drawing drawing = doc.CurrentDrawing;
            // 現在のレイアウト
            Layout layout = doc.CurrentLayout;
               
            // 真北の角度
            double dueNorthDeg = dueNorthDegForDrawRangeCommand;

            // 図形の作成
            PolylineShape polylineShape = CreateShapesForDrawRangeCommand(command, drawing, locatePointA, locatePointB, dueNorthDeg);
            if(polylineShape == null)
            {
                e.CommandState = CommandState.Continue;
                return;
            }

            // 属性の追加
            ShapeAttributeCollection attributes = polylineShape.Attributes;// 範囲を表す属性の付加
            attributes.Add(attributesName, attributesValueCalcRange);

            // 既に範囲があるか探して、あれば削除するよういする。
            PolylineShape alreadyCalcRange = FindCalcRange();
            Drawing alreadyCalcRangeDrawing = null;
            if(alreadyCalcRange != null)
            {
                // カレント以外で表示のみならエラー
                // または編集可能なレイヤでなければエラー
                alreadyCalcRangeDrawing = alreadyCalcRange.Drawing;
                if((!alreadyCalcRangeDrawing.IsCurrent && !alreadyCalcRangeDrawing.Editable) || (!alreadyCalcRange.Layer.IsCurrent && !alreadyCalcRange.Layer.Editable))
                {
                    MessageBox.Show("編集できない部分図かレイヤに既に「範囲」があります。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    e.CommandState = CommandState.End;
                    return;
                } 
            }

            // Undoの開始
            doc.UndoManager.BeginUndoUnit();

            // 既にある範囲を削除
            if(alreadyCalcRange != null)
            {
                Debug.Assert(alreadyCalcRangeDrawing != null);
                alreadyCalcRangeDrawing.Shapes.Remove(alreadyCalcRange);
            }

            // 図形追加
            // レイヤのセット
            Layer currentLayer = doc.LayerTable.CurrentLayer;
            polylineShape.Layer = currentLayer;

            // 追加
            Shape addShape = drawing.Shapes.Add(polylineShape);

            //Undo の終了
            doc.UndoManager.EndUndoUnit();

            e.CommandState = CommandState.End;

        }

        /// <summary>
        /// 「範囲を指定」コマンドの図形作成
        /// </summary>
        /// <param name="command"></param>
        /// <param name="drawing"></param>
        /// <param name="locatePointA"></param>
        /// <param name="locatePointB"></param>
        /// <param name="dueNorthDeg"></param>
        /// <returns></returns>
        private PolylineShape CreateShapesForDrawRangeCommand(Command command, Drawing drawing, Point2d locatePointA, Point2d locatePointB, double dueNorthDeg)
        {
            Document doc = app.ActiveDocument;

            // locatePointAとlocatePointBを通る無限線を作成して交点を求めて真北の方向に傾いた四角を作る
            InfiniteLineShape infiniteLineShape1 = app.ShapeFactory.CreateInfiniteLine(locatePointA, dueNorthDeg);
            InfiniteLineShape infiniteLineShape2 = app.ShapeFactory.CreateInfiniteLine(locatePointA, dueNorthDeg + 90.0);
            InfiniteLineShape infiniteLineShape3 = app.ShapeFactory.CreateInfiniteLine(locatePointB, dueNorthDeg);
            InfiniteLineShape infiniteLineShape4 = app.ShapeFactory.CreateInfiniteLine(locatePointB, dueNorthDeg + 90.0);

            // 交点
            IntersectionPoint[] intersectionPoints14 = infiniteLineShape1.GetIntersectionPoint(infiniteLineShape4);
            if(intersectionPoints14.Length != 1)
                return null;

            IntersectionPoint[] intersectionPoints23 = infiniteLineShape2.GetIntersectionPoint(infiniteLineShape3);
            if(intersectionPoints23.Length != 1)
                return null;

            IntersectionPoint intersectionPoint1 = intersectionPoints14[0];
            IntersectionPoint intersectionPoint2 = intersectionPoints23[0];

            // 四角形の角の点座標
            Point2d[] points = new Point2d[5];
            points[0] = locatePointA;
            points[1] = intersectionPoint1.PointOnShape1.Point;
            points[2] = locatePointB;
            points[3] = intersectionPoint2.PointOnShape1.Point;
            points[4] = locatePointA;

            // 最小最大サイズチェック
            double distance1 = Get2PointsDistance(points[0], points[1]);
            double distance2 = Get2PointsDistance(points[1], points[2]);
            double minDistance = customSettings.miniCalcRangeSize;
            double maxDistance = customSettings.maxCalcRangeSize;
            if(distance1 - minDistance < RcPrecisionConfusion || distance2 - minDistance < RcPrecisionConfusion
                || distance1 - maxDistance >= RcPrecisionConfusion || distance2 - maxDistance >= RcPrecisionConfusion)
            {
                if(distance1 - minDistance < RcPrecisionConfusion && distance2 - minDistance < RcPrecisionConfusion)
                {
                    // 一辺がminDistanceの正方形にする
                    points[1] = new Point2d(locatePointA.X + minDistance, locatePointA.Y);
                    points[1] = GetRotatePoint(points[1], points[0], ToRadian(dueNorthDeg));

                    points[2] = new Point2d(locatePointA.X + minDistance, locatePointA.Y - minDistance);
                    points[2] = GetRotatePoint(points[2], points[0], ToRadian(dueNorthDeg));

                    points[3] = new Point2d(locatePointA.X, locatePointA.Y - minDistance);
                    points[3] = GetRotatePoint(points[3], points[0], ToRadian(dueNorthDeg));
                }
                else if(distance1 - minDistance < RcPrecisionConfusion)
                {
                    points[1] = new Point2d(points[0].X + minDistance, points[0].Y);
                    points[1] = GetRotatePoint(points[1], points[0], ToRadian(dueNorthDeg));

                    double distancce2B = distance2;
                    if(distance2 - maxDistance > RcPrecisionConfusion)
                        distancce2B = maxDistance;

                    points[3] = new Point2d(points[0].X, points[0].Y - distancce2B);
                    points[3] = GetRotatePoint(points[3], points[0], ToRadian(dueNorthDeg));

                    points[2] = new Point2d(points[3].X + minDistance, points[3].Y);
                    points[2] = GetRotatePoint(points[2], points[3], ToRadian(dueNorthDeg));
                }
                else if(distance2 - minDistance < RcPrecisionConfusion)
                {
                    double distancce1B = distance1;
                    if(distance1 - maxDistance > RcPrecisionConfusion)
                        distancce1B = maxDistance;

                    points[1] = new Point2d(points[0].X + distancce1B, points[0].Y);
                    points[1] = GetRotatePoint(points[1], points[0], ToRadian(dueNorthDeg));

                    points[2] = new Point2d(points[1].X, points[1].Y - minDistance);
                    points[2] = GetRotatePoint(points[2], points[1], ToRadian(dueNorthDeg));

                    points[3] = new Point2d(points[0].X, points[0].Y - minDistance);
                    points[3] = GetRotatePoint(points[3], points[0], ToRadian(dueNorthDeg));
                }
                else
                {
                    // 一辺がmaxDistanceの正方形にする
                    points[1] = new Point2d(locatePointA.X + maxDistance, locatePointA.Y);
                    points[1] = GetRotatePoint(points[1], points[0], ToRadian(dueNorthDeg));

                    points[2] = new Point2d(locatePointA.X + maxDistance, locatePointA.Y - maxDistance);
                    points[2] = GetRotatePoint(points[2], points[0], ToRadian(dueNorthDeg));

                    points[3] = new Point2d(locatePointA.X, locatePointA.Y - maxDistance);
                    points[3] = GetRotatePoint(points[3], points[0], ToRadian(dueNorthDeg));
                }

                // locatePointBの位置に合わせて正方形の向きを変える
                Point2d dirPoint = GetRotatePoint(locatePointB, locatePointA, ToRadian(-dueNorthDeg));

                double dirAngleDeg = Get2PointsAngle(locatePointA, dirPoint);
                dirAngleDeg = app.Geometry.WrapAngleTo360(dirAngleDeg);// 0度以上360度未満に補正

                if(dirAngleDeg <= 90.0)
                {
                    points[2] = GetRotatePoint(points[2], points[1], ToRadian(180.0));
                    points[3] = GetRotatePoint(points[3], points[0], ToRadian(180.0));
                }
                else if(dirAngleDeg <= 180.0)
                {
                    points[1] = GetRotatePoint(points[1], points[0], ToRadian(180.0));
                    points[2] = GetRotatePoint(points[2], points[0], ToRadian(180.0));
                    points[3] = GetRotatePoint(points[3], points[0], ToRadian(180.0));
                }
                else if(dirAngleDeg <= 270.0)
                {
                    points[1] = GetRotatePoint(points[1], points[0], ToRadian(180.0));
                    points[2] = GetRotatePoint(points[2], points[3], ToRadian(180.0));
                }
                else// if(dirAngleDeg < 360.0)
                {
                    // そのままでよい
                }
            }

            // 連続線作成
            PolylineShape polylineShape = app.ShapeFactory.CreatePolyline(points);

            doc.BasicShapeSettings.CopyToShape(polylineShape);
            polylineShape.ColorNumber = customSettings.calcRangeColorNumber;// 色

            return polylineShape;
        }

        /// <summary>
        /// 既にある範囲を表す図形を探す
        /// </summary>
        /// <returns></returns>
        PolylineShape FindCalcRange()
        {
            Document doc = app.ActiveDocument;

            // 対象となる部分図IDリスト取得
            List<int> searchDrawingIDList = GetSearchDrawingIDList(doc);

            DrawingCollection drawings = doc.Drawings;
            for(int i = 0; i < drawings.Count; i++)
            {
                Drawing drawing = drawings[i];

                // 対象の部分図か
                if(!searchDrawingIDList.Contains(drawing.ID))
                    continue;

                Shape shape = null;
                while((shape = drawing.Shapes.GetNextShape(shape)) != null)
                {
                    if(shape.Layer.IsEmpty)
                        continue;

                    // 連続線だけ
                    if(!(shape is PolylineShape))
                        continue;

                    // 図形から範囲を表す属性の検索
                    bool result = AttributeSearch(shape, attributesName, attributesValueCalcRange);
                    if(result)
                    {
                        return (PolylineShape)shape;
                    }
                }

            }

            return null;
        }

        /// <summary>
        /// ピッチ情報を配置コマンドマウス移動イベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void CommandPitch_MouseMove(Object sender, CommandMouseMoveEventArgs e)
        {
            Command command = (Command)sender;
            // Command.GetTemporaryParameterItemByID を使用してコマンドの一時的なパラメータアイテムを取得します。
            // 一時的なパラメータアイテムからは、すでにパラメータの入力が確定している場合は確定している入力値が、
            // 入力が確定していない場合はマウスの現在位置などの一時的な値が取得できます。

            // 一時パラメータを取得する
            PointParameterItem pointParameterItem = (PointParameterItem)command.GetTemporaryParameterItemByID(locatePointParameterId);
            if(pointParameterItem.IsEmpty)
                return;

            // 配置点
            Point2d locatePoint = pointParameterItem.Point;

            //ドキュメントの取得
            Document doc = app.ActiveDocument;
            Drawing drawing = doc.CurrentDrawing;
            Point2d orgPoint = new Point2d(0.0, 0.0);

            // 図形の作成
            if(rubberBandShapes == null)
            {
                // 配置点はとりあえず原点で作成する
                // ラバーバンドを作ってしまうので後で座標だけ変える
                Shape[] shapes = CreateShapesForCommandPitch(command, drawing, orgPoint);
                if(shapes == null)
                    return;

                rubberBandShapes = new List<Shape>();
                rubberBandShapes.AddRange(shapes);
            }

            List<Shape> rubberBandShapes2 = new List<Shape>();
            foreach(Shape shape in rubberBandShapes)
            {
                // マウスの位置に移動
                Shape shape2 = shape.Clone();
                shape2.Transform(orgPoint, 1.0, 1.0, 0.0, locatePoint);
                rubberBandShapes2.Add(shape2);
            }

            try
            {
                // ラバーバンド描画
                command.DrawShapeRubberBand(rubberBandShapes2.ToArray());
            }
            catch
            {
                throw;
            }

        }


        /// <summary>
        /// ピッチ情報を配置コマンドパラメータ変更イベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void CommandPitch_ParameterChanged(Object sender, CommandParameterChangedEventArgs e)
        {
            //ラバーバンド表示を削除する
            ClearRubberBand();

            Command command = (Command)sender;

            //ドキュメントの取得
            Document doc = app.ActiveDocument;
            Drawing drawing = doc.CurrentDrawing;

            Point2d locatePoint = new Point2d(0.0, 0.0);
            PointParameterItem pointParameterItem = (PointParameterItem)command.GetParameterItemByID(locatePointParameterId);
            if(pointParameterItem == null)
                return;

            if(pointParameterItem.IsEmpty)
                return;

            locatePoint = pointParameterItem.Point;

            // 図形の作成
            Shape[] shapes = CreateShapesForCommandPitch(command, drawing, locatePoint);
            if(shapes == null)
            {
                e.CommandState = CommandState.Continue;
                return;
            }

            // Undoの開始
            doc.UndoManager.BeginUndoUnit();
 
             for(int i = 0; i < shapes.Length; i++)
            {
                Shape shape = shapes[i];

                // レイヤのセット
                Layer currentLayer = doc.LayerTable.CurrentLayer;
                shape.Layer = currentLayer;

                // 追加
                Shape addShape = drawing.Shapes.Add(shape);
            }

            // Undo の終了
            doc.UndoManager.EndUndoUnit();


            pitchTextBox.Text = ""; // ピッチ情報クリア

            e.CommandState = CommandState.End;
        }

        /// <summary>
        /// ピッチ情報を配置コマンドで描き出す図形作成
        /// </summary>
        /// <param name="command"></param>
        /// <param name="drawing"></param>
        /// <param name="locatePoint"></param>
        /// <returns></returns>
        private Shape[] CreateShapesForCommandPitch(Command command, Drawing drawing, Point2d locatePoint)
        {
            string strValue = pitchTextBox.Text;
            if(strValue == "")
                return null;

            Document doc = drawing.Document;

            ArrayList shapes = new ArrayList();// 戻り値の図形リスト
            string str = "[ 測定ピッチ：" + strValue + "mm ]";

            TextShape text = app.ShapeFactory.CreateText(str, locatePoint, 0.0);
            doc.TextSettings.CopyToShape(text);

            text.DirectionVertical = false;// 横書き
            text.FontHeight = GetTextFontHeight();// 文字高さセット
            //text.Alignment = Alignment.MiddleRight;// 配置点：中右
            shapes.Add(text);
 
            return (Shape[])shapes.ToArray(typeof(Shape));
        }

    }

}
