﻿using RootPro.RootProCAD;
using RootPro.RootProCAD.Command;
using RootPro.RootProCAD.Geometry;
using RootPro.RootProCAD.UI;
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using System.Windows.Media;
using System.Windows.Media.Media3D; //Point3Dに必要
using static RCAddInShadowDrawing.DockingBarUserControl;
using static System.Windows.Forms.VisualStyles.VisualStyleElement.TextBox;
using Drawing = RootPro.RootProCAD.Drawing;
using DrawingCollection = RootPro.RootProCAD.DrawingCollection;

/// <summary>
/// 日影図作成プログラム
/// 概要
/// 　建物の高さに対してできる影の様子を日影図として表すことができる RootPro CAD のアドインプログラムです。
/// 　建物のある緯度と調べたい季節を選ぶことで、自動的に影の形を描きます。
/// 
/// ※使用許諾
/// ・このアドインを実行した結果についての一切の責任はお使いになられる方にあります。
/// ・このアドインは、プログラムソース自体を改変・流用することが可能です。
/// 　また、このプログラムソースを元に独自のアドインやソフトウェアを作成し、無償・有償を問わず第三者に配布することも認めます。
/// 　ただし、改変・流用する場合は、株式会社ルートプロ(RootPro Corporation)作成の
/// 　当プログラムソースを参考にしていることを第三者にわかるように明記してください。
/// 　
/// 　Copyright @ 2020 RootPro Co.,Ltd. (Japan)
/// </summary>


namespace RCAddInShadowDrawing
{
    public partial class DockingBarUserControl : UserControl
    {
        // アドインプログラム名
        string AddInProgramName = "RootPro CAD";
        // 説明図面のファイル名
        string helpFileNameA = "使い方(日影図).rpcd";// アドインのdllと同じ場所
        string helpFileNameB = "使い方(天空図).rpcd";// アドインのdllと同じ場所

        AppAddIn app;
        dynamic RCShadowDataCalculator; // 日影計算用外部ライブラリ

        // 設定値
        CustomSettings customSettings = null;
        AddInSettings addInSettings = null;

        public DockingBarUserControl(AppAddIn app)
        {
            this.app = app;
            InitializeComponent();

            this.Load += DockingBarUserControl_Load;
            this.Disposed += DockingBarUserControl_Disposed;

            // 天空図の半径 文字変更時イベントハンドラ
            hemisphereRadiusUpDown.TextChanged += new EventHandler(hemisphereRadiusUpDown_TextChanged);

            LoadRcLibrary();
        }

        /// <summary>
        /// 外部ライブラリのロード
        /// </summary>
        private void LoadRcLibrary()
        {
            // 日影データを取得するためのライブラリのロード
            string addInFullPath = app.AddInFullPath;// アドインのファイルパス
            string dir1 = Path.GetDirectoryName(addInFullPath);
            string dllFilePathName = dir1 + "\\" + "RCShadowDataCalculator.dll";
            System.Reflection.Assembly rcAddInToolsAsm = System.Reflection.Assembly.LoadFrom(dllFilePathName);
            var typeInfo = rcAddInToolsAsm.GetType("RCShadowDataCalculator.RcShadowData");
            RCShadowDataCalculator = Activator.CreateInstance(typeInfo);
        }


        /// <summary>
        /// ハンドルの破棄イベント処理
        /// RootPro CAD 終了時のメッセージ処理としてこれを使う
        /// </summary>
        /// <param name="e"></param>
        protected override void OnHandleDestroyed(EventArgs e)
        {
            // 設定値をファイルに書き込む
            Debug.Assert(customSettings != null);
            SaveCustomSettings(customSettings);

            Debug.Assert(addInSettings != null);
            GetAddInSettings(addInSettings);
            SaveAddInSettings(addInSettings);

            RemoveAppCommand();

            // 基底クラスのイベント
            base.OnHandleDestroyed(e);
        }

        // コマンド名
        string ShadowDrawingHeightSetCommandName = "RootPro.AddIn.ShadowDrawingHeightSetCommand";                                       // 高さ
        string ShadowDrawingDueNorthCommandName = "RootPro.AddIn.ShadowDrawingDueNorthCommand";                                         // 真北

        string ShadowDrawingSpecifyTimeCommandName = "RootPro.AddIn.ShadowDrawingSpecifyTimeCommand";                                   // 時間指定(時刻日影図)

        string ShadowDrawingShadowMagnificationTableCommandName = "RootPro.AddIn.ShadowDrawingShadowMagnificationTableCommand";         // 影倍率表
        string ShadowDrawingShadowLengthTableCommandName = "RootPro.AddIn.ShadowDrawingShadowLengthTableCommand";                       // 日影長さ表
        string ShadowDrawingAzimuthMagnificationDrawingCommandName = "RootPro.AddIn.ShadowDrawingAzimuthMagnificationDrawingCommand";   // 方位角倍率図

        string ShadowDrawingSettingsOutptCommandName = "RootPro.AddIn.ShadowDrawingSettingsOutptCommand";                               // 設定内容

        string ShadowDrawingDrawRangeCommandName = "RootPro.AddIn.ShadowDrawingDrawRangeCommand";                                       // 範囲を指定
        string ShadowDrawingSpecifyTimeBCommandName = "RootPro.AddIn.ShadowDrawingSpecifyTimeBCommand";                                 // 時間指定(等時間日影図)
        string ShadowDrawingPitchCommandName = "RootPro.AddIn.ShadowDrawingPitchCommand";                                               // ピッチ情報を配置

        string ShadowDrawingMarkDotCommandName = "RootPro.AddIn.ShadowDrawingMarkDotCommand";                                           // 点指定
        string ShadowDrawingLocateComputationCommandName = "RootPro.AddIn.ShadowDrawingLocateComputationCommand";                       // 計算して結果を配置

        string ShadowDrawingSkyMapCommandName = "RootPro.AddIn.ShadowDrawingSkyMapCommand";                                             // 天空図作成コマンド

        string ShadowDrawingSkyRateCompCalcCommandName = "RootPro.AddIn.ShadowDrawingSkyRateCompCalcCommand";                           // 天空率の比較コマンド

        List<Shape> rubberBandShapes = null;       // ラバーバンド図形リスト
        List<int> rubberBandShapesCountList = new List<int>();// ラバーバンド図形リストで複数のラバーバンド図形リストを扱う場合の図形リストごとの数の配列(天空図コマンドで測定点を連続で指定する場合に使用)
        List<Shape> rubberBandTable1Shapes = null; // ラバーバンド表1図形リスト
        List<int> rubberBandTable1ShapesCountList = new List<int>();
        List<Shape> rubberBandTable2Shapes = null; // ラバーバンド表2図形リスト
        List<int> rubberBandTable2ShapesCountList = new List<int>();
        List<Shape> rubberBandOptionShapes = null; // ラバーバンドオプション図形リスト

        // 数値を比較するときに使用する精度
        double RcPrecisionConfusion = Precision.Confusion;
        double RcPrecisionAngular = Precision.RadianAngular;
        double RcPrecisionScale = 1.0E-10;


        /// <summary>
        /// 日影図のタイプ
        /// </summary>
        enum ShadowDrawingType
        {
            OneHourInterval = 0,        // 1 時間毎
            ThirtyMinutesInterval = 1,  // 30 分毎
            SpecifyTime = 2,            // 時間指定
        };

        // 文字に付加する属性
        string attributesName = "ShadowDrawing";
        string attributesValueZ = "Z";// 高さ
        string attributesValueDueNorth = "DueNorth";// 真北
        string attributesValueCalcRange = "CalcRange";  // 計算範囲(連続線に付加する属性の値)
        string attributesValueMarkDot = "MarkDot";      // 点指定の点であることを表す属性

        /// <summary>
        /// 日影基本データクラス
        /// </summary>
        public class ShadowData
        {
            public double solarTime;            // 時間(単位：時、12.5 -> 12時30分 -> 12時1800秒)
            public double solarHeight;          // 太陽高度(rad)
            public double solarDirection;       // 太陽方位角(rad)
            public double shadowMagnification;  // 影の倍率
            public double xMagnification;       // X 倍率
            public double yMagnification;       // Y 倍率

            /// <summary>
            /// コンストラクタ
            /// </summary>
            public ShadowData()
            {
                solarTime = 12.0;
                solarHeight = 45.0;
                solarDirection = 0.0;
                shadowMagnification = 1.0;
                xMagnification = 1.0;
                yMagnification = 1.0;
            }
        }

        /// <summary>
        /// 四角形クラス
        /// </summary>
        public class Quadrilateral
        {
            /// <summary>
            /// コンストラクタ―
            /// </summary>
            /// <param name="app"></param>
            /// <param name="solarTime"></param>
            /// <param name="p0"></param>
            /// <param name="p1"></param>
            /// <param name="p2"></param>
            /// <param name="p3"></param>
            /// <param name="isOverhang"></param>
            public Quadrilateral(AppAddIn app, double solarTime, Point2d p0, Point2d p1, Point2d p2, Point2d p3, bool isOverhang = false)
            {
                // 交差していないという前提
                points[0] = p0;
                points[1] = p1;
                points[2] = p2;
                points[3] = p3;
                points[4] = p0; // 終点：閉じている四角形なので始点と同じ

                this.isOverhang = isOverhang;
                this.solarTime = solarTime;
            }

            /// <summary>
            /// 四角形の点列取得
            /// </summary>
            /// <returns></returns>
            public Point2d[] GetPoints()
            {
                return points;
            }

            public bool isOverhang;// オーバーハング状態(地盤面から浮いた状態)フラグ
            private Point2d[] points = new Point2d[5];
            public double solarTime;// 時間(単位：時、12.5 -> 12時30分 -> 12時1800秒)
        }

        /// <summary>
        /// 矩形領域
        /// </summary>
        public class BndBox
        {
            /// <summary>
            /// コンストラクタ―
            /// </summary>
            /// <param name="p0"></param>
            /// <param name="p1"></param>
            public BndBox(Point2d p0, Point2d p1)
            {
                if(p0.X > p1.X)
                {
                    maxX = p0.X;
                    minX = p1.X;
                }
                else
                {
                    maxX = p1.X;
                    minX = p0.X;
                }

                if(p0.Y > p1.Y)
                {
                    maxY = p0.Y;
                    minY = p1.Y;
                }
                else
                {
                    maxY = p1.Y;
                    minY = p0.Y;
                }
            }

            /// <summary>
            /// 座標の追加
            /// </summary>
            /// <param name="p"></param>
            public void Add(Point2d p)
            {
                if(p.X > maxX)
                {
                    maxX = p.X;
                }

                if(p.X < minX)
                {
                    minX = p.X;
                }

                if(p.Y > maxY)
                {
                    maxY = p.Y;
                }

                if(p.Y < minY)
                {
                    minY = p.Y;
                }
            }

            /// <summary>
            /// 矩形領域の左下座標、右上座標取得
            /// </summary>
            /// <returns></returns>
            public Point2d[] GetBoundingBox()
            {
                Point2d[] points = new Point2d[2];

                points[0] = new Point2d(minX, minY);
                points[1] = new Point2d(maxX, maxY);

                return points;
            }

            public double GetWidth()
            {
                double width = maxX - minX;
                return width;
            }

            public double GetHeight()
            {
                double height = maxY - minY;
                return height;
            }

            double minX;
            double maxX;
            double minY;
            double maxY;
        }


        /// <summary>
        /// 3D領域
        /// </summary>
        public class BndBox3D
        {
            public BndBox3D(Point3D p)
            {
                minX = p.X;
                maxX = p.X;
                minY = p.Y;
                maxY = p.Y;
                minZ = p.Z;
                maxZ = p.Z;
            }

            /// <summary>
            /// 座標の追加
            /// </summary>
            /// <param name="p"></param>
            public void Add(Point3D p)
            {
                if(p.X > maxX)
                {
                    maxX = p.X;
                }

                if(p.X < minX)
                {
                    minX = p.X;
                }

                if(p.Y > maxY)
                {
                    maxY = p.Y;
                }

                if(p.Y < minY)
                {
                    minY = p.Y;
                }

                if(p.Z > maxZ)
                {
                    maxZ = p.Z;
                }

                if(p.Z < minZ)
                {
                    minZ = p.Z;
                }
            }

            /// <summary>
            /// 領域の中心座標取得
            /// </summary>
            /// <returns></returns>
            public Point3D GetBoundingBoxCenterPoint()
            {
                Point3D point = new Point3D();
                point.X = (minX + maxX) * 0.5;
                point.Y = (minY + maxY) * 0.5;
                point.Z = (minZ + maxZ) * 0.5;

                return point;
            }

            /// <summary>
            /// 対角線の長さ
            /// </summary>
            /// <returns></returns>
            public double GetDiagonalLength()
            {
                double diagonalLength = Math.Sqrt(Math.Pow((maxX - minX), 2) + Math.Pow((maxY - minY), 2) + Math.Pow((maxZ - minZ), 2));
                return diagonalLength;
            }

            double minX;
            double maxX;
            double minY;
            double maxY;
            double minZ;
            double maxZ;
        }


        /// <summary>
        /// ベース線(高さ文字のある線)クラス
        /// </summary>
        public class BaseLine
        {
            public BaseLine()
            {
                line = null;
                startPointSideHeightText = null;
                endPointSideHeightText = null;
            }

            public LineShape line;
            public TextShape startPointSideHeightText;
            public TextShape endPointSideHeightText;
        }

        /// <summary>
        /// ベース線のポリゴンクラス
        /// </summary>
        public class BaseLinePolygon
        {
            public BaseLinePolygon(int drawingID, int layerID, bool isOverhang)
            {
                this.drawingID = drawingID;
                this.layerID = layerID;
                this.isOverhang = isOverhang;

                polygonPoints = new List<Point2d>();
            }

            public int drawingID;
            public int layerID;
            public bool isOverhang;// オーバーハング状態(地盤面から浮いた状態)フラグ

            public List<Point2d> polygonPoints;
        }

        /// <summary>
        /// 水平に対する2点のなす角度取得
        /// </summary>
        /// <param name="point1">基点</param>
        /// <param name="point2">方向点</param>
        /// <returns> 2点のなす角度(Degree) </returns>
        public double Get2PointsAngle(Point2d point1, Point2d point2)
        {
            //double angleRad = Get2PointsAngle2Rad(point1, point2);
            //double angleDeg = app.Geometry.RadianToDegree(angleRad);
            //return angleDeg;

            return app.Geometry.GetAngle(point1, point2);
        }

        /// <summary>
        /// 水平に対する2点のなす角度取得
        /// </summary>
        /// <param name="point1"></param>
        /// <param name="point2"></param>
        /// <returns>radian</returns>
        public static double Get2PointsAngle2Rad(Point2d point1, Point2d point2)
        {
            double radian = Math.Atan2(point2.Y - point1.Y, point2.X - point1.X);
            return radian;
        }

        /// <summary>
        /// 2点間の距離取得
        /// </summary>
        /// <param name="point1"></param>
        /// <param name="point2"></param>
        /// <returns></returns>
        public static double Get2PointsDistance(Point2d point1, Point2d point2)
        {
            double distance = Math.Sqrt((point2.X - point1.X) * (point2.X - point1.X) + (point2.Y - point1.Y) * (point2.Y - point1.Y));
            return distance;
        }

        /// <summary>
        /// 指定した点を基点を中心に回転した点の取得
        /// </summary>
        /// <param name="point"></param>
        /// <param name="origin"></param>
        /// <param name="angleRad"></param>
        /// <returns></returns>
        private static Point2d GetRotatePoint(Point2d point, Point2d origin, double angleRad)
        {
            //X=c+(a-c)cos(e)-(b-d)sin(e);
            //Y=d+(a-c)sin(e)+(b-d)cos(e);

            double x1 = point.X - origin.X;
            double y1 = point.Y - origin.Y;

            double x2 = origin.X + x1 * Math.Cos(angleRad) - y1 * Math.Sin(angleRad);
            double y2 = origin.Y + x1 * Math.Sin(angleRad) + y1 * Math.Cos(angleRad);

            Point2d rotatePoint = new Point2d(x2, y2);

            return rotatePoint;
        }

        /// <summary>
        /// Degree値の角度をRadian値に変換
        /// </summary>
        /// <param name="angle"></param>
        /// <returns></returns>
        public static double ToRadian(double angle)
        { 
            return (double)(angle * Math.PI / 180);
        }

        /// <summary>
        /// 2つの角度(Degree)が同じか
        /// </summary>
        /// <param name="angleDeg1"></param>
        /// <param name="angleDeg2"></param>
        /// <returns></returns>
        public bool IsEqualAngleDeg(double angleDeg1, double angleDeg2)
        {
            if(Math.Abs(angleDeg1 - angleDeg2) <= RcPrecisionAngular * (180d / Math.PI))
            {
                return true;
            }

            return false;
        }

        /// <summary>
        /// 2点が同じ座標か
        /// </summary>
        /// <param name="point1"></param>
        /// <param name="point2"></param>
        /// <returns></returns>
        public Boolean IsPointEquals(Point2d point1, Point2d point2)
        {
            return app.Geometry.IsPointEquals(point1, point2);
        }

        /// <summary>
        /// 2点間の中心座標取得
        /// </summary>
        /// <param name="point1"></param>
        /// <param name="point2"></param>
        /// <returns></returns>
        public Point2d CenterPoint(Point2d point1, Point2d point2)
        {
            Point2d centerPoint = app.Geometry.GetMiddlePoint(point1, point2);
            return centerPoint;
        }


        /// <summary>
        /// 四角形の指定した辺の線を取得
        /// </summary>
        /// <param name="quad"></param>
        /// <param name="index"></param>
        /// <returns></returns>
        public LineShape GetQuadrilateralLine(Quadrilateral quad, int index)
        {
            Debug.Assert(index >= 0 || index >= 3);
            if(index < 0 || index > 3)
                return null;

            Point2d[] points = quad.GetPoints();
            LineShape line = app.ShapeFactory.CreateLine(points[index], points[index + 1]);

            return line;
        }

        // このドッキングウィンドウの表示開始時処理
        private void DockingBarUserControl_Load(object sender, EventArgs e)
        {
            // ファイルから設定値を読み込む
            if(!LoadCustomSettings(out customSettings))
                customSettings = new CustomSettings();// 読み込めなかった場合は初期値

            if(!LoadAddInSettings(out addInSettings))
                addInSettings = new AddInSettings();// 読み込めなかった場合は初期値

            Debug.Assert(customSettings != null && addInSettings != null);

            // タブコントローラーのオーナードロー
            tabControl1.DrawMode = TabDrawMode.OwnerDrawFixed;

            // フォーム上のコントロール値セット
            SetAddInSettings(addInSettings);
            
            // 高さコマンドを追加する
            Command command = app.CommandManager.AddCommand(ShadowDrawingHeightSetCommandName, "高さ");
            AddParameters(command);
            // 開始イベントと終了イベントの追加
            command.Begin += new CommandBeginEventHandler(Command_Begin);
            command.End += new EventHandler(Command_End);

            // 真北 コマンド
            command = app.CommandManager.AddCommand(ShadowDrawingDueNorthCommandName, "真北");
            AddParameters(command);
            // 開始イベントと終了イベントの追加
            command.Begin += new CommandBeginEventHandler(Command_Begin);
            command.End += new EventHandler(Command_End);

            // 時間指定(時刻日影図) コマンド
            command = app.CommandManager.AddCommand(ShadowDrawingSpecifyTimeCommandName, "時間指定");
            AddParameters(command);
            // 開始イベントと終了イベントの追加
            command.Begin += new CommandBeginEventHandler(Command_Begin);
            command.End += new EventHandler(Command_End);

            // 影倍率表 コマンド
            command = app.CommandManager.AddCommand(ShadowDrawingShadowMagnificationTableCommandName, "影倍率表");
            AddParameters(command);
            // 開始イベントと終了イベントの追加
            command.Begin += new CommandBeginEventHandler(Command_Begin);
            command.End += new EventHandler(Command_End);

            // 日影長さ表 コマンド
            command = app.CommandManager.AddCommand(ShadowDrawingShadowLengthTableCommandName, "日影長さ表");
            AddParameters(command);
            // 開始イベントと終了イベントの追加
            command.Begin += new CommandBeginEventHandler(Command_Begin);
            command.End += new EventHandler(Command_End);

            // 方位角倍率図 コマンド
            command = app.CommandManager.AddCommand(ShadowDrawingAzimuthMagnificationDrawingCommandName, "方位角倍率図");
            AddParameters(command);
            // 開始イベントと終了イベントの追加
            command.Begin += new CommandBeginEventHandler(Command_Begin);
            command.End += new EventHandler(Command_End);

            // 設定内容 コマンド
            command = app.CommandManager.AddCommand(ShadowDrawingSettingsOutptCommandName, "設定内容");
            AddParameters(command);
            // 開始イベントと終了イベントの追加
            command.Begin += new CommandBeginEventHandler(Command_Begin);
            command.End += new EventHandler(Command_End);

            // 範囲を指定 コマンド
            command = app.CommandManager.AddCommand(ShadowDrawingDrawRangeCommandName, "範囲を指定");
            AddParameters(command);
            // 開始イベントと終了イベントの追加
            command.Begin += new CommandBeginEventHandler(Command_Begin);
            command.End += new EventHandler(Command_End);

            // 指定時間(等時間日影図) コマンド
            command = app.CommandManager.AddCommand(ShadowDrawingSpecifyTimeBCommandName, "指定時間(等時間日影図)");
            AddParameters(command);
            // 開始イベントと終了イベントの追加
            command.Begin += new CommandBeginEventHandler(Command_Begin);
            command.End += new EventHandler(Command_End);

           // ピッチ情報を配置 コマンド
            command = app.CommandManager.AddCommand(ShadowDrawingPitchCommandName, "ピッチ情報を配置");
            AddParameters(command);
            // 開始イベントと終了イベントの追加
            command.Begin += new CommandBeginEventHandler(Command_Begin);
            command.End += new EventHandler(Command_End);

           // 点指定 コマンド
            command = app.CommandManager.AddCommand(ShadowDrawingMarkDotCommandName, "点指定");
            AddParameters(command);
            // 開始イベントと終了イベントの追加
            command.Begin += new CommandBeginEventHandler(Command_Begin);
            command.End += new EventHandler(Command_End);

            // 計算して結果を配置 コマンド
            command = app.CommandManager.AddCommand(ShadowDrawingLocateComputationCommandName, "計算して結果を配置");
            AddParameters(command);
            // 開始イベントと終了イベントの追加
            command.Begin += new CommandBeginEventHandler(Command_Begin);
            command.End += new EventHandler(Command_End);

            // 天空図作成コマンド
            command = app.CommandManager.AddCommand(ShadowDrawingSkyMapCommandName, "天空図作成");
            AddParameters(command);
            // 開始イベントと終了イベントの追加
            command.Begin += new CommandBeginEventHandler(Command_Begin);
            command.End += new EventHandler(Command_End);

            // 天空率の比較コマンド
            command = app.CommandManager.AddCommand(ShadowDrawingSkyRateCompCalcCommandName, "天空率の比較");
            AddParameters(command);
            // 開始イベントと終了イベントの追加
            command.Begin += new CommandBeginEventHandler(Command_Begin);
            command.End += new EventHandler(Command_End);

            // 各コントロールの状態更新
            UpDateControlState();
        }

        // Disposed処理
        private void DockingBarUserControl_Disposed(object sender, EventArgs e)
        {

        }


        private void RemoveAppCommand()
        { 
            //コマンドの削除
            app.CommandManager.RemoveCommand(ShadowDrawingHeightSetCommandName);
            app.CommandManager.RemoveCommand(ShadowDrawingDueNorthCommandName);

            app.CommandManager.RemoveCommand(ShadowDrawingSpecifyTimeCommandName);

            app.CommandManager.RemoveCommand(ShadowDrawingShadowMagnificationTableCommandName);
            app.CommandManager.RemoveCommand(ShadowDrawingShadowLengthTableCommandName);
            app.CommandManager.RemoveCommand(ShadowDrawingAzimuthMagnificationDrawingCommandName);

            app.CommandManager.RemoveCommand(ShadowDrawingSettingsOutptCommandName);

            app.CommandManager.RemoveCommand(ShadowDrawingDrawRangeCommandName);
            app.CommandManager.RemoveCommand(ShadowDrawingSpecifyTimeBCommandName);
            app.CommandManager.RemoveCommand(ShadowDrawingPitchCommandName);

            app.CommandManager.RemoveCommand(ShadowDrawingMarkDotCommandName);
            app.CommandManager.RemoveCommand(ShadowDrawingLocateComputationCommandName);

            app.CommandManager.RemoveCommand(ShadowDrawingSkyMapCommandName);// 天空図作成コマンド
            app.CommandManager.RemoveCommand(ShadowDrawingSkyRateCompCalcCommandName); // 天空率の比較コマンド
        }

        /// <summary>
        /// 各コントロールの状態更新
        /// </summary>
        private void UpDateControlState()
        {
            // 測定点
            if(samplingPointNumberCheckBox.Checked)
            {
                samplingPointNumberUpDown.Enabled = true;
                drawSamplingPointNumberCheckBox.Enabled = true;
            }
            else
            {
                samplingPointNumberUpDown.Enabled = false;
                drawSamplingPointNumberCheckBox.Enabled = false;
            }

            // 天空率のみを表示する場合
            if(onlySkyRateCheckBox.Checked == false)
            {
                hatchComboBox.Enabled = true;// ハッチングの間隔
                scaleComboBox.Enabled = true;// 目盛

                triclinicTriangleCheckBox.Enabled = true;// 三斜求積図

                // 三斜求積図を作成する場合
                if(triclinicTriangleCheckBox.Checked)
                {
                    triclinicTriangleCheckBox.Enabled = true;// 三斜求積図
                    maxSeparateAngleComboBox.Enabled = true;// 最大分割角度
                    drawAllPointNumberCheckBox.Enabled = true;
                    quadratureTableCheckBox.Enabled = true;// 三斜求積表
                    positionTableCheckBox.Enabled = true;// 建物位置表

                    solarTrajectoryCheckBox.Enabled = false;// 太陽軌跡
                    solarTrajectoryInterval10MinutesCheckBox.Enabled = false;// 10分間隔
                    seasonsCheckBox.Enabled = false;    // 四季
                    directCorrectionCheckBox.Enabled = false;   // 方向補正
                }
                else
                {
                    maxSeparateAngleComboBox.Enabled = false;// 最大分割角度
                    drawAllPointNumberCheckBox.Enabled = false;
                    quadratureTableCheckBox.Enabled = false;// 三斜求積表
                    positionTableCheckBox.Enabled = false;// 建物位置表

                    solarTrajectoryCheckBox.Enabled = true;// 太陽軌跡
                    if(solarTrajectoryCheckBox.Checked)
                    {
                        solarTrajectoryInterval10MinutesCheckBox.Enabled = true;// 10分間隔
                        seasonsCheckBox.Enabled = true; // 四季
                        directCorrectionCheckBox.Enabled = true;// 方向補正
                    }
                    else
                    {
                        solarTrajectoryInterval10MinutesCheckBox.Enabled = false;// 10分間隔
                        seasonsCheckBox.Enabled = false; // 四季
                        directCorrectionCheckBox.Enabled = false;// 方向補正
                    }
                }

                circularProjectionAreaCheckBox.Enabled = true;   // 円と正射影の面積を表示
                skyRateCheckBox.Enabled = true;   // 天空率を表示
            }
            else
            {
                hatchComboBox.Enabled = false;// ハッチングの間隔
                scaleComboBox.Enabled = false;// 目盛

                triclinicTriangleCheckBox.Enabled = false;// 三斜求積図

                maxSeparateAngleComboBox.Enabled = false;// 最大分割角度
                drawAllPointNumberCheckBox.Enabled = false;
                quadratureTableCheckBox.Enabled = false;// 三斜求積表
                positionTableCheckBox.Enabled = false;// 建物位置表

                solarTrajectoryCheckBox.Enabled = false;// 太陽軌跡
                solarTrajectoryInterval10MinutesCheckBox.Enabled = false;// 10分間隔
                seasonsCheckBox.Enabled = false;    // 四季
                directCorrectionCheckBox.Enabled = false;   // 方向補正

                circularProjectionAreaCheckBox.Enabled = false;   // 円と正射影の面積を表示
                skyRateCheckBox.Enabled = false;   // 天空率を表示
            }
        }

        // 高さ ボタンクリック
        private void HeightSetButton_Click(object sender, EventArgs e)
        {
            // コマンドの実行
            app.CommandManager.ExecuteCommand(ShadowDrawingHeightSetCommandName);
        }

        // 真北 ボタンクリック
        private void DueNorthButton_Click(object sender, EventArgs e)
        {
            // コマンドの実行
            app.CommandManager.ExecuteCommand(ShadowDrawingDueNorthCommandName);
        }

        /// <summary>
        /// 使い方 ボタンクリック
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void HelpButton_Click(object sender, EventArgs e)
        {
            app.CommandManager.CancelCurrentCommand();

            // 説明図面ファイルを開く
            string addInFullPath = app.AddInFullPath;// アドインのファイルパス
            string folderPath = Path.GetDirectoryName(addInFullPath) + "\\";
            string filePathNameA = folderPath + helpFileNameA;// 日影図
            string filePathNameB = folderPath + helpFileNameB;// 天空図

            // 開く
            DocumentCollection documentCollection = app.Documents;
            Document docA = documentCollection.Open(filePathNameA, true);
            Document docB = documentCollection.Open(filePathNameB, true);

            // 選ばれているタブに合わせてアクティブのドキュメントを変える
            if(tabControl1.SelectedIndex == 0)
                docA.Activate();
            else
                docB.Activate();
        }

        /// <summary>
        /// 3D 確認 ボタンクリック
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void SpaceViewButton_Click(object sender, EventArgs e)
        {
            app.CommandManager.CancelCurrentCommand();

            Button btn = (Button)sender;
            string btName = btn.Name;
            bool withBaseShape = false;
            TextBox heightBox = new TextBox();
            if(btName == "spaceViewButton")
            {
                withBaseShape = withBaseShapeCheckBox.Checked;
                heightBox = locationHeightTextBox;
            }
            else if(btName == "spaceViewButton2")
            {
                withBaseShape = withBaseShapeCheckBox2.Checked;
                heightBox = locationHeightTextBox2;
            }

            // 3D線の作成
            List<Point3D[]> line3ds = new List<Point3D[]>();
            if(!GetBase3DLines(line3ds, heightBox))
                return;

            // 範囲を見る
            BndBox3D bndBox3D = null;
            foreach(Point3D[] linePoints in line3ds)
            {
                if(bndBox3D == null)
                    bndBox3D = new BndBox3D(linePoints[0]);

                bndBox3D.Add(linePoints[0]);
                bndBox3D.Add(linePoints[1]);
            }

            // 範囲の対角線の長さ
            double diagonalLength = bndBox3D.GetDiagonalLength();
            Debug.Assert(diagonalLength > RcPrecisionConfusion);
            if(diagonalLength <= RcPrecisionConfusion)
            {
                return;
            }

            // 収まるスケールを決める
            double spaceDiagonalLength = 1.5;
            if(withBaseShape == true)// 一般図形も表示させる場合はより広い範囲を表示させる
                spaceDiagonalLength = 0.8;

            double scale = spaceDiagonalLength / diagonalLength;

            // 範囲の中心が原点になるように移動してスケールもかける
            Point3D centerPoint3D = bndBox3D.GetBoundingBoxCenterPoint();
            foreach(Point3D[] linePoints in line3ds)
            {
                for(int i = 0; i < 2; i++)
                {
                    linePoints[i].X -= centerPoint3D.X;
                    linePoints[i].Y -= centerPoint3D.Y;
                    // linePoints[i].Z -= centerPoint3D.Z;// Z方向は変えない

                    linePoints[i].X *= scale;
                    linePoints[i].Y *= scale;
                    linePoints[i].Z *= scale;
                }
            }

            List<RC3DForm.RcLine> rcLine3ds = new List<RC3DForm.RcLine>();
            foreach(Point3D[] linePoints in line3ds)
            {
                //System.Windows.Media.Color lineColor = System.Windows.Media.Color.FromArgb(255, 0, 0, 0);
                RC3DForm.RcLine line = new RC3DForm.RcLine(linePoints[0], linePoints[1], Colors.Black, 1.0);
                rcLine3ds.Add(line);
            }

            // 一般図形も表示させる場合は高さゼロの3DLineとして追加する
            if(withBaseShape == true)
                AddZeroHeightLines(rcLine3ds, centerPoint3D, scale);

            // 3Dフォーム
            RC3DForm rc3DForm = new RC3DForm
            {
                AddInFullPath = app.AddInFullPath,// アドインのファイルパス

                // 線データのセット
                Lines = rcLine3ds
            };

            // フォームの表示
            rc3DForm.ShowDialog();
        }

        // 1 時間毎 ボタンクリック
        private void OneHourIntervalButton_Click(object sender, EventArgs e)
        {
            app.CommandManager.CancelCurrentCommand();

            // 日影図作成
            ShadowDrawing(ShadowDrawingType.OneHourInterval);
        }

        // 30 分毎 ボタンクリック
        private void ThirtyMinutesIntervalButton_Click(object sender, EventArgs e)
        {
            app.CommandManager.CancelCurrentCommand();

            // 日影図作成
            ShadowDrawing(ShadowDrawingType.ThirtyMinutesInterval);
        }

        // 時間指定 ボタンクリック
        private void SpecifyTimeButton_Click(object sender, EventArgs e)
        {
            app.CommandManager.CancelCurrentCommand();

            // コマンドの実行
            app.CommandManager.ExecuteCommand(ShadowDrawingSpecifyTimeCommandName);
        }

        // 影倍率表 ボタンクリック
        private void ShadowMagnificationTableButton_Click(object sender, EventArgs e)
        {
            if(!EditControlChk())
                return;

            // コマンドの実行
            app.CommandManager.ExecuteCommand(ShadowDrawingShadowMagnificationTableCommandName);
        }

        // 日影長さ表 ボタンクリック
        private void ShadowLengthTableButton_Click(object sender, EventArgs e)
        {
            if(!EditControlChk())
                return;

            // コマンドの実行
            app.CommandManager.ExecuteCommand(ShadowDrawingShadowLengthTableCommandName);
        }

        // 方位角倍率図 ボタンクリック
        private void AzimuthMagnificationDrawingButton_Click(object sender, EventArgs e)
        {
            if(!EditControlChk())
                return;

            //ドキュメントの取得
            Document doc = app.ActiveDocument;

            // 真北の角度を探す
            int result = DueNorthSearch(doc, out double dueNorth);
            if(result != 1)
            {
                if(result == 0)
                    MessageBox.Show("真北が設定されていません。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                else if(result == -1)
                    MessageBox.Show("真北が図面中に複数あります。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);

                return;
            }

            // コマンドの実行
            app.CommandManager.ExecuteCommand(ShadowDrawingAzimuthMagnificationDrawingCommandName);
        }

        // 設定内容 ボタンクリック
        private void SettingsOutputButton_Click(object sender, EventArgs e)
        {
            if(!EditControlChk())
                return;

            // コマンドの実行
            app.CommandManager.ExecuteCommand(ShadowDrawingSettingsOutptCommandName);
        }

        // コマンドパラメータID
        int selectShapeParameterId = 7;     // 図形パラメータID
        int angleTypeListParameterId = 8;   // 角度表記タイプリストパラメータID
        int locatePointParameterId = 10;    // 配置点パラメータID
        int thisHeightParameterId = 11;     // 日影長さ計算用の高さパラメータID
        int azimuthLengthParameterId = 12;  // 方位角線の長さパラメータID
        int solarTimeId = 13;               // 時間指定パラメータID
        int cornerPointAParameterId = 14;   // 角の座標A
        int cornerPointBParameterId = 15;   // 角の座標B
        int graphModeId = 16;               // グラフ付加モードID
        int samplingPointParameterId = 17;  // 測定点パラメータID
        int samplingPointsParameterId = 18; // 測定点(連続)パラメータID
        int spaceXParameterId = 19;         // 測定点(連続)のときのX方向間隔パラメータID
        int withNumberModeParameterId = 20;// 配置図に番号を作図モードパラメータID
        int tableLocateModeParameterId = 21;// 表の配置位置パラメータID
        int table1LocatePointParameterId = 22;// 表1の配置点パラメータID
        int table2LocatePointParameterId = 23;// 表2の配置点パラメータID

        int selectShapeAParameterId = 24;     // 図形AパラメータID
        int selectShapeBParameterId = 25;     // 図形BパラメータID

        int withFrameParameterId = 26;        // 枠モードパラメータID

        int samplingPointHeightsParameterId = 27;// 測定面の高さ(連続)パラメータID

        /// <summary>
        /// パラメータを追加する
        /// </summary>
        /// <param name="command"></param>
        /// <param name="InputParameterType"></param>
        private void AddParameters(Command command)
        {
            // 高さコマンド
            if(command.UniqueName == ShadowDrawingHeightSetCommandName)
            {
                // 図形パラメータを追加する
                CommandParameterGroup commandParameterGroup = command.ParameterGroups.Add(ParameterItemType.Shape, selectShapeParameterId, "図形");
                ShapeParameterItem parameterItem = (ShapeParameterItem)command.GetParameterItemByID(selectShapeParameterId); // パラメータアイテム
                parameterItem.ShapeTypeFilter = ShapeTypeFilters.Line | ShapeTypeFilters.Polyline;
                parameterItem.Description = "指定した線の端点に高さを表す文字を作成します。用紙での文字の大きさには部分図レイアウトの逆スケールがかかります。";
            }
            // 真北コマンド
            else if(command.UniqueName == ShadowDrawingDueNorthCommandName)
            {
                // 角度表記選択パラメータを追加
                CommandParameterGroup commandParameterGroup0 = command.ParameterGroups.Add(ParameterItemType.List, angleTypeListParameterId, "角度表記タイプ");
                ListParameterItem listParameterItem0 = (ListParameterItem)command.GetParameterItemByID(angleTypeListParameterId); // パラメータアイテム;
                string[] items = { "度分秒", "度" };
                listParameterItem0.Items = items;
                listParameterItem0.SelectedIndex = 0;
                listParameterItem0.ValueSaveMode = ParameterValueSaveMode.Always; // パラメータ値の保存方法
                listParameterItem0.Description = "角度の表記タイプを選んでください。";

                // 図形パラメータを追加する
                CommandParameterGroup commandParameterGroup1 = command.ParameterGroups.Add(ParameterItemType.Shape, selectShapeParameterId, "図形");
                ShapeParameterItem parameterItem1 = (ShapeParameterItem)command.GetParameterItemByID(selectShapeParameterId); // パラメータアイテム
                parameterItem1.ShapeTypeFilter = ShapeTypeFilters.Line;
                parameterItem1.Description = "指定した線の端に北方向を指す文字を作成します。用紙での文字の大きさには部分図レイアウトの逆スケールがかかります。";
            }
            // 時間指定(時刻日影図)
            else if(command.UniqueName == ShadowDrawingSpecifyTimeCommandName)
            {
                // Doubleパラメータを追加
                CommandParameterGroup commandParameterGroup0 = command.ParameterGroups.Add(ParameterItemType.Double, solarTimeId, "指定時間");
                DoubleParameterItem parameterItem0 = (DoubleParameterItem)command.GetParameterItemByID(solarTimeId); // パラメータアイテム
                parameterItem0.Description = "日影図を作成したい時間(8.0 ～ 16.0 の間)を指定してください。分は小数値で入力します。例：15 時 15 分の場合は、15.25 と入力します。";
            }
            // 影倍率表コマンド
            else if(command.UniqueName == ShadowDrawingShadowMagnificationTableCommandName)
            {
                // 座標パラメータを追加する
                CommandParameterGroup commandParameterGroup0 = command.ParameterGroups.Add(ParameterItemType.Point, locatePointParameterId, "配置点");
                PointParameterItem parameterItem0 = (PointParameterItem)command.GetParameterItemByID(locatePointParameterId); // パラメータアイテム
                parameterItem0.Description = "表の左上の座標を指定してください。大きさや色は、RootPro CAD の一般図形や文字の既定値に従います。用紙での文字の大きさには部分図レイアウトの逆スケールがかかります。";
            }
            // 日影長さ表コマンド
            else if(command.UniqueName == ShadowDrawingShadowLengthTableCommandName)
            {
                // 長さパラメータを追加
                CommandParameterGroup commandParameterGroup0 = command.ParameterGroups.Add(ParameterItemType.Length, thisHeightParameterId, "日影長さ計算用の高さ(m)");
                LengthParameterItem parameterItem0 = (LengthParameterItem)command.GetParameterItemByID(thisHeightParameterId); // パラメータアイテム
                parameterItem0.ValueSaveMode = ParameterValueSaveMode.Always; // パラメータ値の保存方法
                parameterItem0.Description = "日影長さ計算用の高さを入力してください。";

                // 座標パラメータを追加する
                CommandParameterGroup commandParameterGroup1 = command.ParameterGroups.Add(ParameterItemType.Point, locatePointParameterId, "配置点");
                PointParameterItem parameterItem1 = (PointParameterItem)command.GetParameterItemByID(locatePointParameterId); // パラメータアイテム
                parameterItem1.Description = "表の左上の座標を指定してください。大きさや色は、RootPro CAD の一般図形や文字の既定値に従います。用紙での文字の大きさには部分図レイアウトの逆スケールがかかります。";
            }
            // 方位角倍率図コマンド
            else if(command.UniqueName == ShadowDrawingAzimuthMagnificationDrawingCommandName)
            {
                // 長さパラメータを追加
                CommandParameterGroup commandParameterGroup0 = command.ParameterGroups.Add(ParameterItemType.Length, azimuthLengthParameterId, "方位角線の長さ");
                LengthParameterItem parameterItem0 = (LengthParameterItem)command.GetParameterItemByID(azimuthLengthParameterId); // パラメータアイテム
                parameterItem0.ValueSaveMode = ParameterValueSaveMode.Always; // パラメータ値の保存方法:コマンド終了後も保存する
                parameterItem0.Description = "方位角線の長さを指定してください。スケールを考慮してください。スケールが 1/100 なら 用紙上の長さの 100 倍の数値を入力します。";

                // 座標パラメータを追加する
                CommandParameterGroup commandParameterGroup1 = command.ParameterGroups.Add(ParameterItemType.Point, locatePointParameterId, "配置点");
                PointParameterItem parameterItem1 = (PointParameterItem)command.GetParameterItemByID(locatePointParameterId); // パラメータアイテム
                parameterItem1.Description = "配置する座標を指定してください。文字の大きさや色は、RootPro CAD の文字の既定値に従います。用紙での文字の大きさには部分図レイアウトの逆スケールがかかります。";
            }
            // 設定内容コマンド
            else if(command.UniqueName == ShadowDrawingSettingsOutptCommandName)
            {
                // 座標パラメータを追加する
                CommandParameterGroup commandParameterGroup0 = command.ParameterGroups.Add(ParameterItemType.Point, locatePointParameterId, "配置点");
                PointParameterItem parameterItem0 = (PointParameterItem)command.GetParameterItemByID(locatePointParameterId); // パラメータアイテム
                parameterItem0.Description = "左上の座標を指定してください。大きさや色は、RootPro CAD の文字の既定値に従います。用紙での文字の大きさには部分図レイアウトの逆スケールがかかります。";
            }
            // 範囲を指定コマンド
            else if(command.UniqueName == ShadowDrawingDrawRangeCommandName)
            {
                // 座標パラメータを追加する
                CommandParameterGroup commandParameterGroup0 = command.ParameterGroups.Add(ParameterItemType.Point, cornerPointAParameterId, "範囲の角の点");
                PointParameterItem parameterItem0 = (PointParameterItem)command.GetParameterItemByID(cornerPointAParameterId); // パラメータアイテム
                parameterItem0.Description = "範囲の角の 1 つの座標を指定してください。";

                // 座標パラメータを追加する
                CommandParameterGroup commandParameterGroup1 = command.ParameterGroups.Add(ParameterItemType.Point, cornerPointBParameterId, "対角の点");
                PointParameterItem parameterItem1 = (PointParameterItem)command.GetParameterItemByID(cornerPointBParameterId); // パラメータアイテム
                parameterItem1.Description = "範囲の角の対角の座標を指定してください。";
            }
            // 時間指定(等時間日影図)
            else if(command.UniqueName == ShadowDrawingSpecifyTimeBCommandName)
            {
                // Doubleパラメータを追加
                CommandParameterGroup commandParameterGroup0 = command.ParameterGroups.Add(ParameterItemType.Double, solarTimeId, "指定時間");
                DoubleParameterItem parameterItem0 = (DoubleParameterItem)command.GetParameterItemByID(solarTimeId); // パラメータアイテム
                parameterItem0.Description = "計算したい等時間日影図の時間(1.0 ～ 8.0 の間)を指定してください。分は小数値で入力します。例：3 時間 15 分の場合は、3.25 と入力します。";
            }
            // ピッチ情報を配置コマンド
            else if(command.UniqueName == ShadowDrawingPitchCommandName)
            {
                // 座標パラメータを追加する
                CommandParameterGroup commandParameterGroup0 = command.ParameterGroups.Add(ParameterItemType.Point, locatePointParameterId, "配置点");
                PointParameterItem parameterItem0 = (PointParameterItem)command.GetParameterItemByID(locatePointParameterId); // パラメータアイテム
                parameterItem0.Description = "ピッチ情報を配置する座標を指定してください。大きさや色は、RootPro CAD の一般図形や文字の既定値に従います。用紙での文字の大きさには部分図レイアウトの逆スケールがかかります。";
            }
            // 点指定コマンド
            else if(command.UniqueName == ShadowDrawingMarkDotCommandName)
            {
                // 座標パラメータを追加する
                CommandParameterGroup commandParameterGroup0 = command.ParameterGroups.Add(ParameterItemType.Point, locatePointParameterId, "配置点");
                PointParameterItem parameterItem0 = (PointParameterItem)command.GetParameterItemByID(locatePointParameterId); // パラメータアイテム
                parameterItem0.Description = "座標を指定してください。その場所に番号を付けた点を配置します。文字の大きさは、RootPro CAD の文字の既定値に従います。用紙での文字の大きさには部分図レイアウトの逆スケールがかかります。点の大きさは文字の大きさによって変わります。";
            }           
            // 計算して結果を配置コマンド
            else if(command.UniqueName == ShadowDrawingLocateComputationCommandName)
            {
                // リストパラメータを追加する
                CommandParameterGroup commandParameterGroup0 = command.ParameterGroups.Add(ParameterItemType.List, graphModeId, "グラフ");
                ListParameterItem listParameterItem0 = (ListParameterItem)command.GetParameterItemByID(graphModeId); // パラメータアイテム
                string[] items = { "なし", "あり" };
                listParameterItem0.Items = items;
                listParameterItem0.SelectedIndex = addInSettings.locateComputationWithGraph ? 1:0;
                listParameterItem0.ValueSaveMode = ParameterValueSaveMode.Always; // パラメータ値の保存方法
                listParameterItem0.Description = "計算結果にグラフを付加する場合は「あり」を選んでください。グラフの色はハッチングと一般図形の既定値に従います。ハッチングは塗りつぶしになります。";

                // 座標パラメータを追加する
                CommandParameterGroup commandParameterGroup1 = command.ParameterGroups.Add(ParameterItemType.Point, locatePointParameterId, "配置点");
                PointParameterItem parameterItem1 = (PointParameterItem)command.GetParameterItemByID(locatePointParameterId); // パラメータアイテム
                parameterItem1.Description = "計算結果を配置する左上の座標を指定してください。点指定コマンドで配置した番号の点の位置で影になる時間を計算して結果を配置します。文字の色は RootPro CAD の文字の既定値に従います。文字の大きさは RootPro CAD の文字の既定値により変わります。用紙での文字の大きさには部分図レイアウトの逆スケールがかかります。";
            }
            // 天空図作成コマンド
            else if(command.UniqueName == ShadowDrawingSkyMapCommandName)
            {
                // 座標パラメータと点列パラメータを切り替えられるように追加する
                // 切り替えた時はCommandSkyMap_ParameterCurrentIndexChangedイベントハンドラで他のパラメータを構築する
                // 点パラメータ
                CommandParameterGroup commandParameterGroup1 = command.ParameterGroups.Add(ParameterItemType.Point, samplingPointParameterId, "測定点");
                PointParameterItem parameterItem10 = (PointParameterItem)command.GetParameterItemByID(samplingPointParameterId); // パラメータアイテム
                parameterItem10.Description = "測定する場所の座標を指定してください。";
                // 点列パラメータ
                PointListParameterItem parameterItem11 = (PointListParameterItem)commandParameterGroup1.Items.Add(ParameterItemType.PointList, samplingPointsParameterId, "測定点(連続)");
                parameterItem11.Description = "測定する場所の座標を連続指定してください。";

                // 天空図の配置点座標パラメータを追加する
                CommandParameterGroup commandParameterGroup3 = command.ParameterGroups.Add(ParameterItemType.Point, locatePointParameterId, "天空図の配置点");
                PointParameterItem parameterItem30 = (PointParameterItem)command.GetParameterItemByID(locatePointParameterId); // パラメータアイテム
                parameterItem30.Description = "配置する天空図の円の中心座標を指定してください。";

                // 三斜求積表の配置点座標パラメータと建物位置表の配置点座標パラメータはコマンド実行時にオプション設定の状態に合わせて追加する
            }
            // 天空率の比較コマンド
            else if(command.UniqueName == ShadowDrawingSkyRateCompCalcCommandName)
            {
                // 図形パラメータを追加する
                CommandParameterGroup commandParameterGroup1 = command.ParameterGroups.Add(ParameterItemType.Shape, selectShapeAParameterId, "天空率a");
                ShapeParameterItem parameterItem1 = (ShapeParameterItem)command.GetParameterItemByID(selectShapeAParameterId); // パラメータアイテム
                parameterItem1.ShapeTypeFilter = ShapeTypeFilters.Text;
                parameterItem1.Description = "天空率aの文字を選択してください。作成される文字の色とサイズ、配置基準点以外の設定は RootPro CAD の文字の既定値に従います。";

                // 図形パラメータを追加する
                CommandParameterGroup commandParameterGroup2 = command.ParameterGroups.Add(ParameterItemType.Shape, selectShapeBParameterId, "天空率b");
                ShapeParameterItem parameterItem2 = (ShapeParameterItem)command.GetParameterItemByID(selectShapeBParameterId); // パラメータアイテム
                parameterItem2.Description = "天空率bの文字を選択してください";

                // 座標パラメータを追加する
                CommandParameterGroup commandParameterGroup3 = command.ParameterGroups.Add(ParameterItemType.Point, locatePointParameterId, "配置点");
                PointParameterItem parameterItem3 = (PointParameterItem)command.GetParameterItemByID(locatePointParameterId); // パラメータアイテム
                parameterItem3.Description = "比較結果を配置する座標を指定してください。";
            }
        }


        /// <summary>
        /// コマンドの開始時に呼ばれる
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Command_Begin(Object sender, CommandBeginEventArgs e)
        {
            Command command = (Command)sender;

            // コントロールの変更不可
            ControlsEnabled(false, command);

            // 高さ、真北コマンド
            if(command.UniqueName == ShadowDrawingHeightSetCommandName || command.UniqueName == ShadowDrawingDueNorthCommandName)
            {
                //パラメータ変更イベントの追加
                command.ParameterChanged += new CommandParameterChangedEventHandler(CommandA_ParameterChanged);

                //マウス移動イベントの追加
                command.MouseMove += new CommandMouseMoveEventHandler(CommandA_MouseMove);
            }
            // 指定時間コマンド(時刻日影図)&(等時間日影図)
            else if(command.UniqueName == ShadowDrawingSpecifyTimeCommandName || command.UniqueName == ShadowDrawingSpecifyTimeBCommandName)
            {
                //パラメータ変更イベントの追加
                command.ParameterChanged += new CommandParameterChangedEventHandler(CommandShadowDrawing_ParameterChanged);
            }
            // 影倍率表、日影長さ表、方位角倍率図、設定内容コマンド
            else if(command.UniqueName == ShadowDrawingShadowMagnificationTableCommandName || command.UniqueName == ShadowDrawingShadowLengthTableCommandName
                || command.UniqueName == ShadowDrawingAzimuthMagnificationDrawingCommandName || command.UniqueName == ShadowDrawingSettingsOutptCommandName)
            {
                //パラメータ変更イベントの追加
                command.ParameterChanged += new CommandParameterChangedEventHandler(CommandB_ParameterChanged);

                //マウス移動イベントの追加
                command.MouseMove += new CommandMouseMoveEventHandler(CommandB_MouseMove);
            }
            // 範囲を指定
            else if(command.UniqueName == ShadowDrawingDrawRangeCommandName)
            {
               　// パラメータ変更イベントの追加
               　command.ParameterChanged += new CommandParameterChangedEventHandler(CommandDrawRange_ParameterChanged);

                 // マウス移動イベントの追加
                 command.MouseMove += new CommandMouseMoveEventHandler(CommandDrawRange_MouseMove);
            }
            // ピッチ情報を配置
            else if(command.UniqueName == ShadowDrawingPitchCommandName)
            {
               　// パラメータ変更イベントの追加
               　command.ParameterChanged += new CommandParameterChangedEventHandler(CommandPitch_ParameterChanged);

                 // マウス移動イベントの追加
                 command.MouseMove += new CommandMouseMoveEventHandler(CommandPitch_MouseMove);
            }
            // 点指定
            else if(command.UniqueName == ShadowDrawingMarkDotCommandName)
            {
                 // パラメータ変更イベントの追加
                command.ParameterChanged += new CommandParameterChangedEventHandler(CommandMarkDot_ParameterChanged);

                // マウス移動イベントの追加
                command.MouseMove += new CommandMouseMoveEventHandler(CommandMarkDot_MouseMove);
            }
            // 計算して結果を配置
            else if(command.UniqueName == ShadowDrawingLocateComputationCommandName)
            {
                // パラメータ変更イベントの追加
                command.ParameterChanged += new CommandParameterChangedEventHandler(CommandLocateComputation_ParameterChanged);

                // マウス移動イベントの追加
                command.MouseMove += new CommandMouseMoveEventHandler(CommandLocateComputation_MouseMove);
            }
            // 天空図作成コマンド
            else if(command.UniqueName == ShadowDrawingSkyMapCommandName)
            {
                // パラメータ切り替えイベントの追加
                command.ParameterGroupCurrentIndexChanged += new CommandParameterGroupCurrentIndexChangedEventHandler(CommandSkyMap_ParameterCurrentIndexChanged);

                // パラメータ変更イベントの追加
                command.ParameterChanged += new CommandParameterChangedEventHandler(CommandSkyMap_ParameterChanged);

                // マウス移動イベントの追加
                command.MouseMove += new CommandMouseMoveEventHandler(CommandSkyMap_MouseMove);

                // 測定点を点列で指定する場合に表示するコマンドパラメータをオプション設定に合わせて変えたいのでここで処理する
                UpdateCommandParameterWithSamplingPointsParameter(command);

                // 表に関係したコマンドパラメータをオプション設定に合わせて変えたいのでここで処理する
                UpdateTableLocateCommandParameter(command);

                // 「配置図に番号を作図」パラメータの初期値
                ListParameterItem listParameterItem00 = (ListParameterItem)command.GetParameterItemByID(withNumberModeParameterId);
                if(listParameterItem00 != null)
                {
                    listParameterItem00.SelectedIndex = 0;// "しない"
                }
            }
            // 天空率の比較コマンド
            else if(command.UniqueName == ShadowDrawingSkyRateCompCalcCommandName)
            {
                // パラメータ変更イベントの追加
                command.ParameterChanged += new CommandParameterChangedEventHandler(CommandSkyRateCompCalc_ParameterChanged);

                // マウス移動イベントの追加
                command.MouseMove += new CommandMouseMoveEventHandler(CommandSkyRateCompCalc_MouseMove);
            }

            // 図形の既定値設定変更イベントハンドラ追加
            app.ShapeSettingsChanged += new ShapeSettingsEventHandler(App_ShapeSettingsChanged);
            // カレント部分図変更イベントハンドラ追加
            app.CurrentDrawingChanged += new CurrentDrawingChangedEventHandler(App_CurrentDrawingChanged);
            // カレントレイヤ変更イベントハンドラ追加
            app.CurrentLayerChanged += new CurrentLayerChangedEventHandler(App_CurrentLayerChanged);
            // レイヤ追加イベントハンドラ追加
            app.LayerAdded += new LayerEventHandler(App_LayerAdded);
            // レイヤ変更イベントハンドラ追加
            app.LayerChanged += new LayerChangedEventHandler(App_LayerChanged);
            // レイヤ削除イベントハンドラ追加
            app.LayerDeleting += new LayerDeletingEventHandler(App_LayerDeleting);

        }

        /// <summary>
        /// コマンドの終了時に呼ばれる
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Command_End(Object sender, EventArgs e)
        {
            Command command = (Command)sender;

            // コントロールの変更可能
            ControlsEnabled(true, command);

            // 高さ、真北コマンド
            if(command.UniqueName == ShadowDrawingHeightSetCommandName || command.UniqueName == ShadowDrawingDueNorthCommandName)
            {
                //パラメータ変更イベントの削除
                command.ParameterChanged -= new CommandParameterChangedEventHandler(CommandA_ParameterChanged);

                //マウス移動イベントの削除
                command.MouseMove -= new CommandMouseMoveEventHandler(CommandA_MouseMove);

            }
            // 指定時間コマンド(時刻日影図)&(等時間日影図)
            else if(command.UniqueName == ShadowDrawingSpecifyTimeCommandName || command.UniqueName == ShadowDrawingSpecifyTimeBCommandName)
            {
                //パラメータ変更イベントの削除
                command.ParameterChanged -= new CommandParameterChangedEventHandler(CommandShadowDrawing_ParameterChanged);
            }
            // 影倍率表、日影長さ表、方位角倍率図、設定内容コマンド
            else if(command.UniqueName == ShadowDrawingShadowMagnificationTableCommandName || command.UniqueName == ShadowDrawingShadowLengthTableCommandName
                || command.UniqueName == ShadowDrawingAzimuthMagnificationDrawingCommandName || command.UniqueName == ShadowDrawingSettingsOutptCommandName)
            {
                //パラメータ変更イベントの削除
                command.ParameterChanged -= new CommandParameterChangedEventHandler(CommandB_ParameterChanged);

                //マウス移動イベントの削除
                command.MouseMove -= new CommandMouseMoveEventHandler(CommandB_MouseMove);
            }
            // 範囲を指定
            else if(command.UniqueName == ShadowDrawingDrawRangeCommandName)
            {
                //パラメータ変更イベントの削除
                command.ParameterChanged -= new CommandParameterChangedEventHandler(CommandDrawRange_ParameterChanged);

                //マウス移動イベントの削除
                command.MouseMove -= new CommandMouseMoveEventHandler(CommandDrawRange_MouseMove);
            }
            // ピッチ情報を配置
            else if(command.UniqueName == ShadowDrawingPitchCommandName)
            {
                //パラメータ変更イベントの削除
                command.ParameterChanged -= new CommandParameterChangedEventHandler(CommandPitch_ParameterChanged);

                //マウス移動イベントの削除
                command.MouseMove -= new CommandMouseMoveEventHandler(CommandPitch_MouseMove);
            }
            // 点指定
            else if(command.UniqueName == ShadowDrawingMarkDotCommandName)
            {
                //パラメータ変更イベントの削除
                command.ParameterChanged -= new CommandParameterChangedEventHandler(CommandMarkDot_ParameterChanged);

                //マウス移動イベントの削除
                command.MouseMove -= new CommandMouseMoveEventHandler(CommandMarkDot_MouseMove);
            }
            // 計算して結果を配置
            else if(command.UniqueName == ShadowDrawingLocateComputationCommandName)
            {
                //パラメータ変更イベントの削除
                command.ParameterChanged -= new CommandParameterChangedEventHandler(CommandLocateComputation_ParameterChanged);

                //マウス移動イベントの削除
                command.MouseMove -= new CommandMouseMoveEventHandler(CommandLocateComputation_MouseMove);
            }
            // 天空図作成コマンド
            else if(command.UniqueName == ShadowDrawingSkyMapCommandName)
            {
                // パラメータ切り替えイベントの削除
                command.ParameterGroupCurrentIndexChanged -= new CommandParameterGroupCurrentIndexChangedEventHandler(CommandSkyMap_ParameterCurrentIndexChanged);

                //パラメータ変更イベントの削除
                command.ParameterChanged -= new CommandParameterChangedEventHandler(CommandSkyMap_ParameterChanged);

                //マウス移動イベントの削除
                command.MouseMove -= new CommandMouseMoveEventHandler(CommandSkyMap_MouseMove);
            }
            // 天空率の比較コマンド
            else if(command.UniqueName == ShadowDrawingSkyRateCompCalcCommandName)
            {
                //パラメータ変更イベントの削除
                command.ParameterChanged -= new CommandParameterChangedEventHandler(CommandSkyRateCompCalc_ParameterChanged);

                //マウス移動イベントの削除
                command.MouseMove -= new CommandMouseMoveEventHandler(CommandSkyRateCompCalc_MouseMove);
            }

            // 図形の既定値設定変更イベントの削除
            app.ShapeSettingsChanged -= new ShapeSettingsEventHandler(App_ShapeSettingsChanged);
            // カレント部分図変更イベントの削除
            app.CurrentDrawingChanged -= new CurrentDrawingChangedEventHandler(App_CurrentDrawingChanged);
            // カレントレイヤ変更イベントの削除
            app.CurrentLayerChanged -= new CurrentLayerChangedEventHandler(App_CurrentLayerChanged);
            // レイヤ追加イベントハンドラの削除
            app.LayerAdded -= new LayerEventHandler(App_LayerAdded);
            // レイヤ変更イベントハンドラの削除
            app.LayerChanged -= new LayerChangedEventHandler(App_LayerChanged);
            // レイヤ削除イベントハンドラの削除
            app.LayerDeleting -= new LayerDeletingEventHandler(App_LayerDeleting);

            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// 図形の既定値設定の変更イベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void App_ShapeSettingsChanged(object sender, ShapeSettingsEventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// カレントレイヤの変更イベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void App_CurrentLayerChanged(object sender, CurrentLayerChangedEventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// カレント部分図の変更イベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void App_CurrentDrawingChanged(object sender, CurrentDrawingChangedEventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        
        void App_LayerAdded(object sender, LayerEventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }


        void App_LayerChanged(object sender, LayerChangedEventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        void App_LayerDeleting(object sender, LayerDeletingEventArgs e)
        {
            //ラバーバンドクリア
            ClearRubberBand();
        }

        /// <summary>
        /// ラバーバンドクリア
        /// </summary>
        private void ClearRubberBand()
        {
            if(rubberBandShapes != null)
            {
                rubberBandShapes.Clear();
                rubberBandShapes = null;

                rubberBandShapesCountList.Clear();
            }

            if(rubberBandOptionShapes != null)
            {
                rubberBandOptionShapes.Clear();
                rubberBandOptionShapes = null;
            }

            if(rubberBandTable1Shapes != null)
            {
                rubberBandTable1Shapes.Clear();
                rubberBandTable1Shapes = null;

                rubberBandTable1ShapesCountList.Clear();
            }

            if(rubberBandTable2Shapes != null)
            {
                rubberBandTable2Shapes.Clear();
                rubberBandTable2Shapes = null;

                rubberBandTable2ShapesCountList.Clear();
            }
        }

        /// <summary>
        /// コントロールの変更可能・不可
        /// </summary>
        /// <param name="flg"></param>
        /// <param name="command"></param>
        void ControlsEnabled(bool flg, Command command)
        {
            ///// 日影図関係のコントロール /////

            // 測定面の高さ(日影図)
            locationHeightTextBox.Enabled = flg;

            //spaceViewButton.Enabled = flg;// 3D 確認
            //withBaseShapeCheckBox.Enabled = flg;

            latitudeTextBox.Enabled = flg;// 緯度
            seasonComboBox.Enabled = flg;// 季節
            // 任意時期のテキストボックス
            if(seasonComboBox.SelectedIndex == 3) // 季節が任意時期の場合だけ有効にできる
                anySeasonTextBox.Enabled = flg;
            else
                anySeasonTextBox.Enabled = false;

            timeRangeComboBox.Enabled = flg;

            ///// 天空図関係コントロール /////

            // 測定面の高さ(天空率)
            // 天空図作成のときに点列と同じ数の高さを入力するパラメーターが出ているときは
            // locationHeightTextBox2をfalseにしているため、コマンド終了時は必ずtrueに
            locationHeightTextBox2.Enabled = true;

            //spaceViewButton2.Enabled = flg;// 3D 確認
            //withBaseShapeCheckBox2.Enabled = flg;

            /// 他の天空率関連のコントロールはコマンド実行中も変更できるようにした
            bool skyMapCtrlsEnabled = false;
            if(skyMapCtrlsEnabled)
            {
                // 天空率のみを表示
                onlySkyRateCheckBox.Enabled = flg;

                // 天空図の半径
                hemisphereRadiusUpDown.Enabled = flg;

                // 天空率のみを表示でなければ
                if(!onlySkyRateCheckBox.Checked)
                {
                    // 三斜求積図
                    triclinicTriangleCheckBox.Enabled = flg;
                    if(!triclinicTriangleCheckBox.Checked)
                    {
                        // 最大分割角度
                        maxSeparateAngleComboBox.Enabled = false;

                        // 三斜求積表
                        quadratureTableCheckBox.Enabled = false;
                        // 建物位置表
                        positionTableCheckBox.Enabled = false;
                    }
                    else
                    {
                        // 最大分割角度
                        maxSeparateAngleComboBox.Enabled = flg;

                        // 三斜求積表
                        quadratureTableCheckBox.Enabled = flg;
                        // 建物位置表
                        positionTableCheckBox.Enabled = flg;
                    }
                }
            }
        }

        /// <summary>
        /// タブが切り替わったとき
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void tabControl1_SelectedIndexChanged(object sender, EventArgs e)
        {
            // 現在のコマンドを中止する
            app.CommandManager.CancelCurrentCommand();
        }


        private Boolean EditControlChk()
        {
            // 値が取得できればOK
            return GetShadowBaseData(out double latitude, out double solarDeclination, out double locationHeight);
        }

        /// <summary>
        /// 季節コンボボックスの変更時の処理
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void SeasonComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if(seasonComboBox.SelectedIndex == 3) // 季節が任意時期の場合だけ有効にできる
            {
                anySeasonTextBox.Enabled = true;
            }
            else
            {
                // 
                // 春秋分(deg)日赤緯(冬至: -23.45°、春秋分: 0°、夏至: 23.45°、任意時期)
                string str = "0";
                // 冬至
                if(seasonComboBox.SelectedIndex == 0)
                    str = "-23.45";
                // 夏至
                else if(seasonComboBox.SelectedIndex == 2)
                    str = "23.45";

                anySeasonTextBox.Text = str;
                anySeasonTextBox.Enabled = false;
            }
        }


        /// <summary>
        /// 見えている部分図のIDリスト取得
        /// </summary>
        /// <param name="doc"></param>
        /// <returns></returns>
        private List<int> GetSearchDrawingIDList(Document doc)
        {
            List<int> searchDrawingIDList = new List<int>();

            // 現在表示しているビューが部分図だけか用紙も見ているかで分ける
            // 対象となる部分図は表示している部分図ということになる
            TabView tabView = app.ActiveTabView;
            if(tabView is DrawingView drawingView)
            {
                Drawing drawing = drawingView.Drawing;
                // 部分図
                if(drawing is Part)
                {
                    searchDrawingIDList.Add(drawing.ID);
                }
                // 用紙
                else if(drawing is Paper)
                {
                    // 用紙上の部分図レイアウト
                    LayoutCollection papersLayouts = drawing.Layouts;
                    for(int i = 0; i < papersLayouts.Count; i++)
                    {
                        Layout layout = papersLayouts[i];
                        Drawing drawing2 = layout.Drawing;
                        int drawingID = drawing2.ID;

                        // カレント以外で表示されていなければ対象外
                        if(!drawing2.IsCurrent && !drawing2.Visible)
                            continue;

                        if(!searchDrawingIDList.Contains(drawingID))
                            searchDrawingIDList.Add(drawingID);
                    }
                }
            }

            return searchDrawingIDList;
        }


        /// <summary>
        /// 真北を探す
        /// </summary>
        /// <param name="doc">ドキュメント</param>
        /// <param name="dueNorthDeg">out 真北の角度(deg)</param>
        /// <returns>1:見つかった、0:見つからなかった、-1:複数あった</returns>
        private int DueNorthSearch(Document doc, out double dueNorthDeg)
        {
            dueNorthDeg = 0.0d;
            int couter = 0;

            // 対象となる部分図IDリスト取得
            List<int> searchDrawingIDList = GetSearchDrawingIDList(doc);

            DrawingCollection drawings = doc.Drawings;
            for(int i = 0; i < drawings.Count; i++)
            {
                Drawing drawing = drawings[i];

                // 対象の部分図か
                if(!searchDrawingIDList.Contains(drawing.ID))
                    continue;

                Shape shape = null;

                //図形を列挙する
                while((shape = drawing.Shapes.GetNextShape(shape)) != null)
                {
                    if(shape.Layer.IsEmpty)
                        continue;

                    // 文字だけ
                    if(!(shape is TextShape))
                        continue;

                    // 文字図形から属性の検索
                    bool result = AttributeSearch(shape, attributesName, attributesValueDueNorth);
                    if(result)
                    {
                        // 文字列から数値を取得
                        TextShape textShape = ((TextShape)shape);
                        string toText = textShape.Text;

                        try
                        {
                            // 真北の角度(度分秒表記の場合は十進数に変換される)
                            dueNorthDeg = app.NumericTextConverter.DmsTextToDegree(toText);
                        }
                        catch(Exception)
                        {
                            continue;
                        }

                        couter++;
                        if(couter > 1)
                            return -1;// 複数見つかったのでエラー
                    }
                }
            }

            if(couter == 1)
                return 1;

            return 0;
        }

        /// <summary>
        /// 属性文字列を検索する
        /// </summary>
        /// <param name="shape">調べる図形</param>
        /// <param name="searchName">検索属性名</param>
        /// <param name="searchValue">検索属値</param>
        /// <returns>true - 見つかった</returns>
        private bool AttributeSearch(Shape shape, string searchName, string searchValue)
        {
            for(int i = 0; i < shape.Attributes.Count; i++)
            {
                // 属性名が同じ 
                if(shape.Attributes[i].Name == searchName)
                {
                    // 属性値も同じなら
                    if(shape.Attributes[i].Value == searchValue)
                        return true;
                }

            }

            //見つかりませんでした。
            return false;
        }


        /// <summary>
        /// 高さ、真北コマンド実行中にマウスが移動イベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void CommandA_MouseMove(Object sender, CommandMouseMoveEventArgs e)
        {
            Command command = (Command)sender;
            // Command.GetTemporaryParameterItemByID を使用してコマンドの一時的なパラメータアイテムを取得します。
            // 一時的なパラメータアイテムからは、すでにパラメータの入力が確定している場合は確定している入力値が、
            // 入力が確定していない場合はマウスの現在位置などの一時的な値が取得できます。

            // 選択した図形の取得
            ShapeParameterItem shapeParameterItem = (ShapeParameterItem)command.GetTemporaryParameterItemByID(selectShapeParameterId);
            if(shapeParameterItem == null)
                return;

            if(shapeParameterItem.IsEmpty)
                return;

            Shape shape = shapeParameterItem.Shape.Shape;
            if(!(shape is LineShape))
                return;

            // カレントレイヤの図形だけ
            if(!shape.Layer.IsCurrent)
                return;

            //ドキュメントの取得
            Document doc = app.ActiveDocument;
            // 現在のレイアウト
            Layout layout = doc.CurrentLayout;

            Point2d startPoint = ((LineShape)shape).StartPoint;
            Point2d endPoint = ((LineShape)shape).EndPoint;
            Point2d pickPoint = shapeParameterItem.PickPoint;
            Point2d localPickPoint = layout.TransformPointWorldToLocal(pickPoint);// アクティブなレイアウトに合わせた座標へ変換

            double startSideDistance = app.Geometry.GetDistance(startPoint, localPickPoint);
            double endSideDistance = app.Geometry.GetDistance(endPoint, localPickPoint);

            // 文字配置位置
            Point2d locatePoint = startPoint;
            Point2d vectorPoint = endPoint;
            if(startSideDistance >= endSideDistance)
            {
                locatePoint = endPoint;
                vectorPoint = startPoint;
            }

            // 文字配置角度
            double textAngle = Get2PointsAngle(locatePoint, vectorPoint);// Degrees

            // 配置する文字
            string outputText = "";// 初期値
            string outputAttributesValue = "";

            // 高さコマンドの場合
            if(command.UniqueName == ShadowDrawingHeightSetCommandName)
            {
                textAngle = Get2PointsAngle(locatePoint, vectorPoint); // Degrees

                outputText = heightTextBox.Text.ToString();
                // 高さの文字チェック
                if(!ChekHeightText(outputText))
                    return;

                outputAttributesValue = attributesValueZ;
            }
            // 真北コマンドの場合
            else if(command.UniqueName == ShadowDrawingDueNorthCommandName)
            {
                double northAngle = Get2PointsAngle(vectorPoint, locatePoint);
                northAngle -= 90.0d;
                if(northAngle <= -180.0d)
                    northAngle += 360.0d;
                else if(northAngle > 180.0d)
                    northAngle -= 360.0d;

                // 角度表記タイプの取得
                ListParameterItem listParameterItem = (ListParameterItem)command.GetTemporaryParameterItemByID(angleTypeListParameterId);
                if(listParameterItem == null)
                    return;

                if(listParameterItem.IsEmpty)
                    return;

                // 度分秒表記の場合
                if(listParameterItem.SelectedIndex == 0)
                    outputText = DegreeToDmsString(northAngle, 4);
                else
                    outputText = DegreeToString(northAngle, 6);

                outputAttributesValue = attributesValueDueNorth;
            }


            //部分図の取得
            Drawing drawing = doc.CurrentDrawing;

            // ラバーバンド用の図形リスト
            ArrayList rubberBandShapes2 = new ArrayList();

            // 文字
            double textHeight = GetTextFontHeight();
            TextShape text = app.ShapeFactory.CreateText(outputText, locatePoint, textAngle);
            doc.TextSettings.CopyToShape(text);
            text.DirectionVertical = false;// 横書き
            text.FontHeight = textHeight;//　文字高さ
            text.Alignment = Alignment.BottomLeft;// 配置点：左下
            text.ColorNumber = customSettings.dueNorthTextColorNo;//色
            rubberBandShapes2.Add(text);

            // 真北コマンドの場合
            // 「真北」という文字を配置
            if(command.UniqueName == ShadowDrawingDueNorthCommandName)
            {
                Point2d locateTextPoint = locatePoint;
                locateTextPoint.X -= text.FontHeight * 0.5d;
                locateTextPoint = app.Geometry.RotatePoint(locateTextPoint, locatePoint, textAngle);
                TextShape text2 = app.ShapeFactory.CreateText("真北", locateTextPoint, textAngle + 90.0d);
                text2.DirectionVertical = false;// 横書き
                text2.FontHeight = textHeight;//　文字高さ
                text2.Alignment = Alignment.BottomMiddle;// 配置点：中下
                text2.ColorNumber = customSettings.dueNorthTextColorNo;//色
                rubberBandShapes2.Add(text2);
            }

            try
            {
                // ラバーバンド描画
                command.DrawShapeRubberBand((Shape[])rubberBandShapes2.ToArray(typeof(Shape)));
            }
            catch
            {
                throw;
            }

        }


        /// <summary>
        /// 高さ、真北コマンド実行時のパラメータの変化のイベントハンドラ
        /// 高さ、真北文字を追加します
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void CommandA_ParameterChanged(Object sender, CommandParameterChangedEventArgs e)
        {
            Command command = (Command)sender;

            // 選択した図形の取得
            ShapeParameterItem shapeParameterItem = (ShapeParameterItem)command.GetParameterItemByID(selectShapeParameterId);
            if(shapeParameterItem == null)
                return;

            if(shapeParameterItem.IsEmpty)
                return;

            Shape shape = shapeParameterItem.Shape.Shape;
            if(!(shape is LineShape))
            {
                MessageBox.Show("線図形を選んでください。連続線の場合は線に分解してください。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                e.CommandState = CommandState.ParameterError;
                return;
            }

            // カレントレイヤの図形だけ
            if(!shape.Layer.IsCurrent)
            {
                MessageBox.Show("カレントレイヤの線図形を選んでください。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                e.CommandState = CommandState.ParameterError;
                return;
            }

            //ドキュメントの取得
            Document doc = app.ActiveDocument;
            // 現在のレイアウト
            Layout layout = doc.CurrentLayout;

            Point2d startPoint = ((LineShape)shape).StartPoint;
            Point2d endPoint = ((LineShape)shape).EndPoint;
            Point2d pickPoint = shapeParameterItem.PickPoint;
            Point2d localPickPoint = layout.TransformPointWorldToLocal(pickPoint);// アクティブなレイアウトに合わせた座標へ変換

            double startSideDistance = app.Geometry.GetDistance(startPoint, localPickPoint);
            double endSideDistance = app.Geometry.GetDistance(endPoint, localPickPoint);

            // 文字配置位置
            Point2d locatePoint = startPoint;
            Point2d vectorPoint = endPoint;
            if(startSideDistance >= endSideDistance)
            {
                locatePoint = endPoint;
                vectorPoint = startPoint;
            }

            // 文字配置角度
            double textAngle = Get2PointsAngle(locatePoint, vectorPoint); // Degrees

            // 配置する文字
            string outputText = "";// 初期値
            string outputAttributesValue = "";

            // 高さコマンドの場合
            if(command.UniqueName == ShadowDrawingHeightSetCommandName)
            {
                textAngle = Get2PointsAngle(locatePoint, vectorPoint); // Degrees
                outputText = heightTextBox.Text.ToString();
                if(!ChekHeightText(outputText))
                {
                    MessageBox.Show("高さに適切な数値文字を指定してください。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    e.CommandState = CommandState.End;
                    return;
                }

                outputAttributesValue = attributesValueZ;
            }
            // 真北コマンドの場合
            else if(command.UniqueName == ShadowDrawingDueNorthCommandName)
            {
                double northAngle = Get2PointsAngle(vectorPoint, locatePoint);
                northAngle -= 90.0d;
                if(northAngle <= -180.0d)
                    northAngle += 360.0d;
                else if(northAngle > 180.0d)
                    northAngle -= 360.0d;

                // 角度表記タイプの取得
                ListParameterItem listParameterItem = (ListParameterItem)command.GetTemporaryParameterItemByID(angleTypeListParameterId);
                if(listParameterItem == null)
                    return;

                if(listParameterItem.IsEmpty)
                    return;

                // 度分秒表記の場合
                if(listParameterItem.SelectedIndex == 0)
                    outputText = DegreeToDmsString(northAngle, 4);
                else
                    outputText = DegreeToString(northAngle, 6);

                outputAttributesValue = attributesValueDueNorth;
            }

            // Undoの開始
            doc.UndoManager.BeginUndoUnit();

            // 既に同じ配置点にないか（値は関係ない）。あればそれは削除して、新しく配置する
            // 真北コマンドでは座標に関係なく属性が一致すれば削除する
            DrawingCollection drawings = doc.Drawings;
            for(int i = 0; i < drawings.Count; i++)
            {
                Drawing drawing = drawings[i];

                // カレント部分図以外で編集可能でない部分図の場合は対象外
                if(!drawing.IsCurrent && !drawing.Editable)
                    continue;

                Shape shape2 = null;
                ArrayList deleteShapeArray = new ArrayList();
                // 図形を列挙する
                while((shape2 = drawing.Shapes.GetNextShape(shape2)) != null)
                {
                    if(shape2.Layer.IsEmpty)
                        continue;

                    // 文字だけ
                    if(!(shape2 is TextShape))
                        continue;

                    // 真北コマンドの場合は編集可能なレイヤの文字なら削除対象
                    if(command.UniqueName == ShadowDrawingDueNorthCommandName)
                    {
                        if(!shape2.Layer.IsCurrent && !shape2.Layer.Editable)
                            continue;
                    }
                    // 高さコマンドではカレントレイヤのもの以外は消さない
                    else
                    {
                        // カレントレイヤ以外は削除しない
                        if(!shape2.Layer.IsCurrent)
                            continue;
                    }

                    // 文字図形から属性の検索
                    bool result = AttributeSearch(shape2, attributesName, outputAttributesValue);
                    // 見つかった場合
                    if(result)
                    {
                        if(command.UniqueName == ShadowDrawingDueNorthCommandName)
                        {
                            deleteShapeArray.Add(shape2);        //削除リストに追加
                            // 「真北」という文字と角度文字列があるはずなので続ける
                            continue;
                        }

                        // 文字の配置座標が同じなら削除
                        TextShape textShape = ((TextShape)shape2);
                        if(IsPointEquals(textShape.Point, locatePoint))
                        {
                            deleteShapeArray.Add(shape2);        //削除リストに追加
                            break;
                        }
                    }
                }

                // 図形の削除
                foreach(Shape deleteShape in deleteShapeArray)
                    drawing.Shapes.Remove(deleteShape);
            }

            //部分図の取得
            Drawing currentDrawing = doc.CurrentDrawing;

            // 文字の追加
            double textHeight = GetTextFontHeight();
            TextShape text = currentDrawing.Shapes.AddText(outputText, locatePoint, textAngle);
            text.Alignment = Alignment.BottomLeft;// 配置点：左下
            text.FontHeight = textHeight;//　文字高さ
            text.ColorNumber = customSettings.dueNorthTextColorNo;//色
            ShapeAttributeCollection attributes = text.Attributes;  // 属性
            attributes.Add(attributesName, outputAttributesValue);

            // 真北コマンドの場合
            // 「真北」という文字を配置
            if(command.UniqueName == ShadowDrawingDueNorthCommandName)
            {
                Point2d locateTextPoint = locatePoint;
                locateTextPoint.X -= text.FontHeight * 0.5d;
                locateTextPoint = app.Geometry.RotatePoint(locateTextPoint, locatePoint, textAngle);
                TextShape text2 = currentDrawing.Shapes.AddText("真北", locateTextPoint, textAngle + 90.0d); // 回転
                text2.FontHeight = textHeight;//　文字高さ    
                text2.Alignment = Alignment.BottomMiddle;// 配置点：中下
                text2.ColorNumber = customSettings.dueNorthTextColorNo;//色
                attributes = text2.Attributes;  // 属性
                attributes.Add(attributesName, outputAttributesValue);
            }

            // Undo の終了
            doc.UndoManager.EndUndoUnit();

            // 高さコマンドは続ける
            if(command.UniqueName == ShadowDrawingHeightSetCommandName)
                e.CommandState = CommandState.Execute;
            else
                e.CommandState = CommandState.End;
        }


        /// <summary>
        /// 高さの文字チェック(カンマ区切りの数字(2つまで)対応)
        /// </summary>
        /// <param name="heightText"></param>
        /// <returns></returns>
        private bool ChekHeightText(string heightText)
        {
            // チェック
            string[] arr = heightText.Split(',');
            Debug.Assert(arr.Length <= 2);
            if(arr.Length > 2)// 2つまで
                return false;

            //double lastHeight = 0.0;
            for(int i = 0; i < arr.Length; i++)
            {
                string toText = arr[i];
                toText = toText.TrimStart();

                // 文字列をdouble型に変換
                if(!double.TryParse(toText, out double height))
                    return false;

                // Debug.Assert(height >= 0.0);

                if(i == 0)
                {
                //    lastHeight = height;
                }
                else
                {
                    // 後ろの高さの方が小さければいい
                //    if(height >= lastHeight)
                //        return false;
                    // どちらが大きくてもようようにした。
                    // 大きい方から小さい方を引くという考えに変更
                }
            }

            return true;
        }

        /// <summary>
        /// 指定時間コマンド実行時のパラメータの変化のイベント
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void CommandShadowDrawing_ParameterChanged(Object sender, CommandParameterChangedEventArgs e)
        {
            Command command = (Command)sender;

            // 入力された時間の取得
            DoubleParameterItem doubleParameterItem = (DoubleParameterItem)command.GetParameterItemByID(solarTimeId);
            if(doubleParameterItem == null)
                return;

            if(doubleParameterItem.IsEmpty)
                return;

            double specifyTime = doubleParameterItem.Value;// 指定時間

            // 指定時間(時刻日影図)
            if(command.UniqueName == ShadowDrawingSpecifyTimeCommandName)
            {
                // 8～16時までの間
                if(specifyTime < 8.0 || specifyTime > 16.0)
                {
                    MessageBox.Show("8.0～16.0の間で指定してください。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return;
                }
            }
            // 指定時間(等時間日影図)
            else if(command.UniqueName == ShadowDrawingSpecifyTimeBCommandName)
            {
                if(specifyTime < 1.0 || specifyTime > 8.0)
                {
                    MessageBox.Show("1.0～8.0の間で指定してください。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return;
                }
            }
            else
            {
                return;
            }


            if(command.UniqueName == ShadowDrawingSpecifyTimeCommandName)
            {
                // 日影図作成
                ShadowDrawing(ShadowDrawingType.SpecifyTime, specifyTime);
            }
            else if(command.UniqueName == ShadowDrawingSpecifyTimeBCommandName)
            {
                // 等時間日影図作成
                float minute = (float)(specifyTime * 60.0);
                DrawShadowDrawing(minute);
            }

            // コマンド終了
            e.CommandState = CommandState.End;
        }

        /// <summary>
        /// 図面上の高さ文字とベース線から高さを反映させた3D線リストを取得
        /// 確認用の処理で使う
        /// </summary>
        /// <param name="line3ds"></param>
        /// <returns></returns>
        private bool GetBase3DLines(List<Point3D[]> line3ds, TextBox heightBox)
        {
            Document doc = app.ActiveDocument;		//ドキュメントの取得
            if(doc == null)
                return false;

            int err = 0;

            // 測定面の高さ
            string locationHeightTextBoxText = heightBox.Text;
            //文字列をdouble型に変換
            if(!double.TryParse(locationHeightTextBoxText, out double locationHeight))
            {
                err = 1;
                goto ERRLABEL;
            }

            // 高さの文字のリスト取得
            var heightTextShapeList = new List<TextShape>();
            int count = GetWithAttributesTextShapeList(doc, attributesValueZ, heightTextShapeList);
            if(count == 0)
            {
                err = 2;
                goto ERRLABEL;
            }

            double maxHeightValue = 0.0;
            if(!ChkHeightTextShapeList(heightTextShapeList, out maxHeightValue))
            {
                err = 3;
                goto ERRLABEL;
            }

            // 実行できるレイアウトの図面かチェック
            if(!CheckLayout(doc, heightTextShapeList))
            {
                err = 4;
                goto ERRLABEL;
            }

            // 高さの「ベース線」(高さが書かれている建物の縁を表す線)リストを取得
            List<List<BaseLine>> baseLinesList = GetHeightBaseLineShapeList(doc, heightTextShapeList);
            if(baseLinesList.Count == 0)
            {
                err = 5;
                goto ERRLABEL;
            }

            LayerTable layerTable = doc.LayerTable;

            foreach(List<BaseLine> baseLines in baseLinesList)
            {
                foreach(BaseLine baseLine in baseLines)
                {
                    LineShape line = baseLine.line;
                    TextShape startPointSideHeightText = baseLine.startPointSideHeightText;
                    TextShape endPointSideHeightText = baseLine.endPointSideHeightText;

                    double startPointHeightTop = 0.0;
                    double startPointHeightBottom = 0.0;
                    double endPointHeightTop = 0.0;
                    double endPointHeightBottom = 0.0;

                    if(baseLine.startPointSideHeightText != null || baseLine.endPointSideHeightText != null)
                    {
                        int layerID = baseLine.line.LayerID;
                        Layer layer = layerTable.GetLayerByID(layerID);
                        Layer parentlayer = layer.ParentLayer;
                        string layerName = layer.Name;
                        string layerGroupName = parentlayer.Name;

                        if(baseLine.startPointSideHeightText != null)
                        {
                            try
                            {
                                if(!GetHeightValue(baseLine.startPointSideHeightText.Text, layerGroupName, layerName, locationHeight, out startPointHeightTop, out startPointHeightBottom))
                                {
                                    err = 6;
                                    goto ERRLABEL;
                                }
                            }
                            catch
                            {
                                err = 6;
                                goto ERRLABEL;
                            }
                        }

                        if(baseLine.endPointSideHeightText != null)
                        {
                            try
                            {
                                if(!GetHeightValue(baseLine.endPointSideHeightText.Text, layerGroupName, layerName, locationHeight, out endPointHeightTop, out endPointHeightBottom))
                                {
                                    err = 6;
                                    goto ERRLABEL;
                                }
                            }
                            catch
                            {
                                err = 6;
                                goto ERRLABEL;
                            }
                        }
                    }


                    // Z 方向が上

                    // 始点側の線（下から上に伸びる線）
                    Point3D[] line1Points = new Point3D[2];
                    line1Points[0].X = line.StartPoint.X;
                    line1Points[0].Y = line.StartPoint.Y;
                    line1Points[0].Z = startPointHeightBottom;
                    line1Points[1].X = line.StartPoint.X;
                    line1Points[1].Y = line.StartPoint.Y;
                    line1Points[1].Z = startPointHeightTop;

                    // 終点側の線（下から上に伸びる線）
                    Point3D[] line2Points = new Point3D[2];
                    line2Points[0].X = line.EndPoint.X;
                    line2Points[0].Y = line.EndPoint.Y;
                    line2Points[0].Z = endPointHeightBottom;
                    line2Points[1].X = line.EndPoint.X;
                    line2Points[1].Y = line.EndPoint.Y;
                    line2Points[1].Z = endPointHeightTop;

                    // 始点側と終点側を下で結ぶ線
                    Point3D[] line3Points = new Point3D[2];
                    line3Points[0].X = line1Points[0].X;
                    line3Points[0].Y = line1Points[0].Y;
                    line3Points[0].Z = line1Points[0].Z;
                    line3Points[1].X = line2Points[0].X;
                    line3Points[1].Y = line2Points[0].Y;
                    line3Points[1].Z = line2Points[0].Z;

                    // 始点側と終点側を上で結ぶ線
                    Point3D[] line4Points = new Point3D[2];
                    line4Points[0].X = line1Points[1].X;
                    line4Points[0].Y = line1Points[1].Y;
                    line4Points[0].Z = line1Points[1].Z;
                    line4Points[1].X = line2Points[1].X;
                    line4Points[1].Y = line2Points[1].Y;
                    line4Points[1].Z = line2Points[1].Z;

                    line3ds.Add(line1Points);
                    line3ds.Add(line2Points);
                    line3ds.Add(line3Points);
                    line3ds.Add(line4Points);
                }
            }


            ERRLABEL:
            if(err != 0)
            {
                string strErr = "";
                if(err == 1)
                    strErr = "測定面の高さの数値が正しくありません。";
                else if(err == 2)
                    strErr = "高さの設定された線がありません。";
                else if(err == 3)
                    strErr = "高さのデータまたは測定面の高さが正しくありません・";
                else if(err == 4)
                    strErr = "実行できるレイアウトの図面ではありません。";
                else if(err == 5)
                    strErr = "高さの「ベース線」取得でエラーになりました。";
                else if(err == 6)
                    strErr = "測定面より低い高さの線があるか、レイヤ名、レイヤーグループ名でレベルの高さが適切でなないためエラーになりました。";

                if(strErr != "")
                    MessageBox.Show(strErr, AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);

                return false;
            }

            return true;
        }


        /// <summary>
        /// 一般図形（線）を高さのない3DLineとして追加する
        /// </summary>
        /// <param name="line3ds"></param>
        /// <param name="centerPoint3D"></param>
        /// <param name="scale"></param>
        private void AddZeroHeightLines(List<RC3DForm.RcLine> line3ds, Point3D centerPoint3D, double scale)
        {
             Document doc = app.ActiveDocument;		//ドキュメントの取得
            if(doc == null)
                return;

            List<Shape> shapeList = new List<Shape>();

            // 対象となる部分図IDリスト取得
            List<int> searchDrawingIDList = GetSearchDrawingIDList(doc);

            DrawingCollection drawings = doc.Drawings;
            for(int i = 0; i < searchDrawingIDList.Count; i++)
            {
                int drawingID = searchDrawingIDList[i];
                Drawing drawing = drawings.GetDrawingByID(drawingID);

                // 用紙上は対象外
                if(drawing is Paper)
                    continue;

                // 図形を列挙する
                Shape shape = null;
                while((shape = drawing.Shapes.GetNextShape(shape)) != null)
                {
                    if(shape.Layer.IsEmpty)
                        continue;

                    // カレントレイヤ以外で編集可能レイヤでない場合は何もしない(※表示のみは対象外)
                    if(!shape.Layer.IsCurrent && !shape.Layer.Editable)
                        continue;

                    // 線分・連続線・円・円弧・楕円・楕円弧・Bスプラインのみ
                    if(!(shape is BasicShape) || shape is HalfInfiniteLineShape || shape is InfiniteLineShape)
                        continue;

                    shapeList.Add(shape);
                }
            }

            foreach(Shape shape in shapeList)
            {
                Debug.Assert(shape is BasicShape);
                if(!(shape is BasicShape))
                    continue;

                // 色
                ColorItem colorItem = ((BasicShape)shape).Color;
                System.Windows.Media.Color lineColor = System.Windows.Media.Color.FromArgb(255, colorItem.R, colorItem.G, colorItem.B);

                double resolution = 10.0;// 分解したときの1線分長さ
                int maxPointCount = 512; // 最大線分数

                try
                {
                    // 一般図形を連続線に変換（Bスプラインや円弧なども）
                    PolylineShape polyline = ((BasicShape)shape).ConvertToPolyline(resolution, maxPointCount);
                    LineShape[] lineShapes = polyline.GetSplitLineShapes();

                    foreach(LineShape lineShape in lineShapes)
                    {
                        // 始点側と終点側を結ぶ線
                        Point3D[] linePoints = new Point3D[2];
                        linePoints[0].X = (lineShape.StartPoint.X - centerPoint3D.X) * scale;
                        linePoints[0].Y = (lineShape.StartPoint.Y - centerPoint3D.Y) * scale;
                        linePoints[0].Z = 0.0;
                        linePoints[1].X = (lineShape.EndPoint.X - centerPoint3D.X) * scale;
                        linePoints[1].Y = (lineShape.EndPoint.Y - centerPoint3D.Y) * scale;
                        linePoints[1].Z = 0.0;

                        RC3DForm.RcLine line = new RC3DForm.RcLine(linePoints[0], linePoints[1], lineColor, 1.0);
                        line3ds.Add(line);
                    }

                }
                catch(Exception)
                {
                    throw;
                }
            }
        }

        /// <summary>
        /// 日影図作成
        /// 1 時間毎、30 分毎、時間指定 コマンドで使用
        /// </summary>
        /// <param name="shadowDrawingType"></param>
        /// <param name="specifyTime">時間指定の場合の時間</param>
        private void ShadowDrawing(ShadowDrawingType shadowDrawingType, double specifyTime = 12.0)
        {
            // ウェイトカーソル
            Cursor.Current = Cursors.WaitCursor;

            Document doc = app.ActiveDocument;		//ドキュメントの取得
            if(doc == null)
                return;

            // 実行できる状態の図面か
            if(!CheckDocument(doc, true))
                return;

            // 観測地点の緯度
            double latitudeDeg = 0; // (deg)

            // 日赤緯(冬至:-23.45°、春秋分:0°、夏至:23.45°、任意時期)
            double solarDeclinationDeg = 0.0;// 初期値:春秋分(deg)

            // 真北の角度
            double dueNorthDeg = 0.0;// (deg)

            // 測定面の高さ
            double locationHeight = 0.0;

            // 真北の角度取得
            int result = DueNorthSearch(doc, out dueNorthDeg);
            if(result != 1)
            {
                if(result == 0)
                    MessageBox.Show("真北が設定されていません。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                else if(result == -1)
                    MessageBox.Show("真北が図面中に複数あります。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);

                return;
            }

            // 観測地点の緯度、日赤緯、測定面の高さ
            if(!GetShadowBaseData(out latitudeDeg, out solarDeclinationDeg, out locationHeight))
                return;

            // 高さの文字のリスト取得
            var heightTextShapeList = new List<TextShape>();
            int count = GetWithAttributesTextShapeList(doc, attributesValueZ, heightTextShapeList);
            if(count == 0)
            {
                MessageBox.Show("高さの設定された線がありません。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }

            double maxHeightValue = 0.0;
            if(!ChkHeightTextShapeList(heightTextShapeList, out maxHeightValue))
            {
                MessageBox.Show("高さのデータまたは測定面の高さが正しくありません。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }

            // 実行できるレイアウトの図面かチェック
            if(!CheckLayout(doc, heightTextShapeList))
            {
                return;
            }

            // 高さの「ベース線」(高さが書かれている建物の縁を表す線)リストを取得
            List<List<BaseLine>> baseLinesList = GetHeightBaseLineShapeList(doc, heightTextShapeList);
            if(baseLinesList.Count == 0)
                return;

            // ベース線からなる多角形のリスト作成
            List<BaseLinePolygon> baseLinesPolygonList = new List<BaseLinePolygon>();
            List<BaseLinePolygon> subBaseLinesPolygonList = null;
            foreach(List<BaseLine> baseLines in baseLinesList)
            {
                subBaseLinesPolygonList = GetBaseLinesPolygonList(doc, locationHeight, baseLines);
                if(subBaseLinesPolygonList.Count > 0)
                    baseLinesPolygonList.AddRange(subBaseLinesPolygonList);
            }

            // ベース線の全図形範囲の中心点
            Point2d baseLinesBoxCenterPoint = GetLinesBoxCenterPoint(baseLinesList);

            // 日影データを取得
            double latitudeRad = latitudeDeg * (Math.PI / 180d);
            double solarDeclinationRad = solarDeclinationDeg * (Math.PI / 180d); // 日赤緯(冬至: -23.45°、春秋分: 0°、夏至: 23.45°、任意時期)
            int intervalType = 3600;// データ取得時間間隔タイプ
            List<ShadowData> shadowDataList = null;
            // 指定時間で日影図作成の場合
            if(shadowDrawingType == ShadowDrawingType.SpecifyTime)
            {
                ShadowData shadowData = new ShadowData();
                GetShadowData(latitudeRad, solarDeclinationRad, specifyTime, shadowData);

                shadowDataList = new List<ShadowData>();
                shadowDataList.Add(shadowData);
            }
            // 1時間、30分毎の日影図作成の場合
            else
            {
                if(shadowDrawingType == ShadowDrawingType.ThirtyMinutesInterval)
                    intervalType = 1800;

                bool h9Toh15 = timeRangeComboBox.SelectedIndex == 1 ? true : false;// true:測定時間が9:00～15:00の場合
                double startSolarTime = 8;
                double endSolarTime = 16;
                // 測定範囲が9:00～15:00の場合
                if(h9Toh15)
                {
                    startSolarTime = 9;
                    endSolarTime = 15;
                }

                shadowDataList = GetShadowDataList(latitudeRad, solarDeclinationRad, intervalType, startSolarTime, endSolarTime);
            }

            // 時間毎の日影の範囲の線を作成
            List<ArrayList> shadowShapesList = new List<ArrayList>();
            for(int i = 0; i < shadowDataList.Count; i++)
            {
                double solarTime = shadowDataList[i].solarTime;
                double xMagnification = shadowDataList[i].xMagnification;
                double yMagnification = shadowDataList[i].yMagnification;

                // ベース線からなる多角形ごとに影の外周線リストを作る
                List<List<LineShape>> shadowOutlinesList = GetShadowOutlinesList(doc, baseLinesPolygonList, heightTextShapeList, solarTime, xMagnification, yMagnification, locationHeight, dueNorthDeg, null, true);

                // 図面に追加する図形リスト
                // 今処理中の時間の文字も添える
                List<LineShape> groupFrameLines = new List<LineShape>();
                ArrayList shapes = new ArrayList();
                for(int j = 0; j < shadowOutlinesList.Count; j++)
                {
                    List<LineShape> shadowOutlines = shadowOutlinesList[j];

                    // 影の線
                    for(int k = 0; k < shadowOutlines.Count; k++)
                    {
                        // 影の線
                        LineShape lineShape = shadowOutlines[k];

                        // 色と線種
                        // 時間指定コマンドの場合
                        if(shadowDrawingType == ShadowDrawingType.SpecifyTime)
                        {
                            lineShape.ColorNumber = customSettings.shadowLineColor2No;// 影の線色

                            // 1時間で割れる場合
                            if(solarTime % 1 == 0)
                                lineShape.LinetypeNumber = customSettings.shadowLineType1No;
                            // 30分で割れる場合
                            else if(solarTime % 0.5 == 0)
                                lineShape.LinetypeNumber = customSettings.shadowLineType2No;
                            // その他
                            else
                                lineShape.LinetypeNumber = customSettings.shadowLineType3No;
                        }
                        else
                        {
                            lineShape.ColorNumber = customSettings.shadowLineColor1No;// 影の線色

                            // 線種、30毎の出力の場合は交互に線種を変える
                            if(intervalType == 1800 && i % 2 == 1)
                                lineShape.LinetypeNumber = customSettings.shadowLineType2No;
                            else
                                lineShape.LinetypeNumber = customSettings.shadowLineType1No;
                        }

                        shapes.Add(lineShape);
                    }

                    // 時間表記文字
                    // 単独の線のリスト（他と繋がらない）のものは書き出す。また、単独でない場合でもこの後につ繋がるものがなければ書き出す。
                    if(IsLastSingleFrameLines(j, shadowOutlinesList))
                    {
                        groupFrameLines.AddRange(shadowOutlines);

                        if(GetTimeTextPointAndAngle(groupFrameLines, baseLinesBoxCenterPoint, out Point2d locatePoint, out double locateAngle))
                        {
                            TextShape textShape = CreateTimeText(solarTime);
                            textShape.Point = locatePoint;
                            textShape.Angle = locateAngle;
                            textShape.Alignment = Alignment.BottomMiddle;// 中下

                            shapes.Add(textShape);
                        }

                        groupFrameLines.Clear();
                    }
                    // 次以降のリスト中の線と一緒に配置位置を決めるので取っておく
                    else
                    {
                        groupFrameLines.AddRange(shadowOutlines);
                    }
                }

                if(shapes.Count > 0)
                    shadowShapesList.Add(shapes);
            }

            ///////////////////////////////////////////////////////////
            // UNDO 可能な操作を開始
            doc.UndoManager.BeginUndoUnit();

            // 図形追加
            Drawing currentDrawing = doc.CurrentDrawing;	    //カレント部分図の取得
            foreach(ArrayList shapes in shadowShapesList)
            {
                ArrayList addGroupShapeArray = new ArrayList(); // 時間毎にグループ化するための図形リスト
                int shapeCount = shapes.Count;
                for(int i = 0; i < shapeCount; i++)
                {
                    Shape shape = (Shape)shapes[i];

                    // レイヤのセット
                    Layer currentLayer = doc.LayerTable.CurrentLayer;
                    shape.Layer = currentLayer;

                    // 追加
                    Shape addShape = currentDrawing.Shapes.Add(shape);

                    // グループ
                    addGroupShapeArray.Add(addShape);
                }

                // グループ化
                Shape shapeGroup = currentDrawing.Shapes.AddDrawGroup((Shape[])addGroupShapeArray.ToArray(typeof(Shape)), false);
            }

            // UNDO 可能な操作を終了
            doc.UndoManager.EndUndoUnit();
            /////////////////////////////////////////////////////////////

        }

        /// <summary>
        /// 指定した属性の付加された文字リスト取得
        /// </summary>
        /// <param name="doc">ドキュメント</param>
        /// <param name="attributesValue">高さ文字リスト</param>
        /// <param name="textShapeList">高さ文字リスト</param>
        /// <returns>int 見つかった数 (-1:エラー)</returns>
        private int GetWithAttributesTextShapeList(Document doc, string attributesValue, List<TextShape> heightTextShapeList)
        {

            // 対象となる部分図IDリスト取得
            List<int> searchDrawingIDList = GetSearchDrawingIDList(doc);

            DrawingCollection drawings = doc.Drawings;
            for(int i = 0; i < searchDrawingIDList.Count; i++)
            {
                
                int drawingID = searchDrawingIDList[i];
                Drawing drawing = drawings.GetDrawingByID(drawingID);

                // 用紙上の文字は対象外
                if(drawing is Paper)
                    continue;

                // 図形を列挙する
                Shape shape = null;
                while((shape = drawing.Shapes.GetNextShape(shape)) != null)
                {
                    if(shape.Layer.IsEmpty)
                        continue;

                    // カレントレイヤ以外で編集可能レイヤでない場合は何もしない(※表示のみは対象外)
                    if(!shape.Layer.IsCurrent && !shape.Layer.Editable)
                        continue;

                    // 文字だけ
                    if(!(shape is TextShape))
                        continue;

                    // 文字図形から属性の検索
                    bool result = AttributeSearch(shape, attributesName, attributesValue);
                    if(result)
                    {
                        // 文字を取得
                        TextShape textShape = ((TextShape)shape);

                        // リストに追加
                        heightTextShapeList.Add(textShape);
                    }
                }
            }

            return heightTextShapeList.Count;
        }


        /// <summary>
        /// 高さ文字が適正かチェック
        /// </summary>
        /// <param name="heightTextShapeList"></param>
        /// <param name="maxHeightValue">out 高さ最大値</param>
        /// <returns></returns>
        private bool ChkHeightTextShapeList(List<TextShape> heightTextShapeList, out double maxHeightValue)
        {
            maxHeightValue = 0.0;// 高さ最大値初期化

            for(var i = 0; i < heightTextShapeList.Count; i++)
            {
                TextShape textShape = heightTextShapeList[i];

                string str = textShape.Text;

                // チェック
                string[] arr = str.Split(',');// カンマ区切りの数字対応
                Debug.Assert(arr.Length <= 2);
                if(arr.Length > 2)// 2つまで
                    return false;

                double value1 = 0.0;
                double value2 = 0.0;
                bool isValue2 = false;
                //double lastHeight = 0.0;
                for(int j = 0; j < arr.Length; j++)
                {
                    string toText = arr[j];
                    toText = toText.TrimStart();
                    // 文字列をdouble型に変換
                    if(!double.TryParse(toText, out double height))
                        continue;

                    //Debug.Assert(height >= 0.0);
                    //if(height > maxHeightValue)
                    //    maxHeightValue = height;

                    if(j == 0)
                    {
                        value1 = height;
                    }
                    else
                    {
                     //   Debug.Assert(height < lastHeight);
                     //   if(height > lastHeight)
                     //       return false;

                     // どちらが大きくてもようようにした。
                     // 大きい方から小さい方を引くという考えに変更

                        value2 = height;

                        isValue2 = true;
                    }
                }

                //↓↓↓↓↓↓ 2025/10/8 ↓↓↓↓↓↓
                // 2つの値があって、2つ目の方が大きい値の場合上下入れ替え
                if(isValue2 && value2 > value1)
                {
                    double tempValue = value1;
                    value1 = value2;
                    value2 = tempValue;
                }

                // 1つ目だけでマイナスの場合は絶対値にして value2 も同じ値にする
                if(!isValue2 && value1 < 0.0)
                {
                    value1 *= -1.0;
                    value2 = value1;
                    isValue2 = true;
                }
                
                if(value1 > maxHeightValue)
                    maxHeightValue = value1;
                //↑↑↑↑↑↑ 2025/10/8 ↑↑↑↑↑↑
            }

            return true;
        }

        /// <summary>
        /// 高さのベース線リストを取得
        /// </summary>
        /// <param name="doc">ドキュメント</param>
        /// <param name="heightTextShapeList"></param>
        /// <returns>ベース線リスト(異なる部分図ごとさらにレイヤごとにリストが作成される)</returns>
        private List<List<BaseLine>> GetHeightBaseLineShapeList(Document doc, List<TextShape> heightTextShapeList)
        {
            var baseLineShapesList = new List<List<BaseLine>>();

            // 部分図ごとでさらにレイヤごとにリストを作成する
            DrawingCollection drawings = doc.Drawings;
            for(int i = 0; i < drawings.Count; i++)
            {
                Drawing drawing = drawings[i];

                // カレント部分図以外で編集可能でない部分図の場合は対象外
                if(!drawing.IsCurrent && !drawing.Editable)
                    continue;

                // レイヤIDをキーにした線リストのマップ
                IDictionary<int, List<BaseLine>> lineShapeListMap = new Dictionary<int, List<BaseLine>>();

                //図形を列挙する
                Shape shape = null;
                while((shape = drawing.Shapes.GetNextShape(shape)) != null)
                {
                    if(shape.Layer.IsEmpty)
                        continue;

                    // カレントレイヤ以外で編集可能レイヤでない場合は対象外
                    if(!shape.Layer.IsCurrent && !shape.Layer.Editable)
                        continue;

                    // 線だけ
                    if(!(shape is LineShape))
                        continue;

                    LineShape lineShape = ((LineShape)shape);
                    int layerID = lineShape.LayerID;

                    // 部分図もレイヤも同じであることが条件
                    // 更に、レイヤが異なる場合は別のリストに追加する
                    // 高さ文字の配置点と一致する端点を持つ線をリストに追加する
                    int findSide = -1; // 0:始点側が見つかっている、1:終点側が見つかっている、2:両サイド見つかっている
                    BaseLine baseLine = null;
                    for(var j = 0; j < heightTextShapeList.Count; j++)
                    {
                        TextShape textShape = heightTextShapeList[j];

                        // 部分図とレイヤが同じものだけ
                        Drawing textShapeDrawing = textShape.Drawing;
                        if(textShapeDrawing.ID != drawing.ID || lineShape.LayerID != textShape.LayerID)
                            continue;

                        Point2d textPoint = textShape.Point;// 高さ文字の配置点座標

                        // ベース線の端点に一致する高さ文字を見つけてbaseLineを作る
                        if(IsPointEquals(textPoint, lineShape.StartPoint))
                        {
                            if(findSide < 0)
                            {
                                findSide = 0;
                                baseLine = new BaseLine
                                {
                                    line = lineShape,
                                    startPointSideHeightText = textShape
                                };
                            }
                            else if(findSide == 1)
                            {
                                findSide = 2;
                                baseLine.startPointSideHeightText = textShape;
                            }
                        }
                        else if(IsPointEquals(textPoint, lineShape.EndPoint))
                        {
                            if(findSide < 0)
                            {
                                findSide = 1;
                                baseLine = new BaseLine
                                {
                                    line = lineShape,
                                    endPointSideHeightText = textShape
                                };
                            }
                            else if(findSide == 0)
                            {
                                findSide = 2;
                                baseLine.endPointSideHeightText = textShape;
                            }
                        }


                        if(findSide == 2 || (findSide >= 0 && j == heightTextShapeList.Count))
                        {
                            List<BaseLine> lineList = null;

                            if(lineShapeListMap.ContainsKey(layerID))
                            {
                                lineList = lineShapeListMap[layerID];
                            }
                            else
                            {
                                lineList = new List<BaseLine>();
                                lineShapeListMap.Add(layerID, lineList);
                            }

                            // リストに追加
                            lineList.Add(baseLine);
                            findSide = -1;
                            break;
                        }
                    }
                }

                foreach(List<BaseLine> lineShapeList in lineShapeListMap.Values)
                {
                    baseLineShapesList.Add(lineShapeList);
                }
            }

            return baseLineShapesList;
        }

        /// <summary>
        /// ベース線と高さ文字情報からできる影の四角形のリスト作成取得
        /// </summary>
        /// <param name="doc"></param>
        /// <param name="baseLinePolygon"></param>
        /// <param name="heightTextShapeList"></param>
        /// <param name="solarTime"></param>
        /// <param name="xMagnification"></param>
        /// <param name="yMagnification"></param>
        /// <param name="locationHeight"></param>
        /// <param name="dueNorthDeg"></param>
        /// <returns></returns>
        private List<Quadrilateral> GetQuadrilateralList(Document doc, BaseLinePolygon baseLinePolygon, List<TextShape> heightTextShapeList, double solarTime, double xMagnification, double yMagnification, double locationHeight, double dueNorthDeg)
        {
            LayerTable layerTable = doc.LayerTable;

            int polygonPointCount = baseLinePolygon.polygonPoints.Count;

            int capacity = polygonPointCount;
            var quads = new List<Quadrilateral>(capacity); // 影の四角形リスト

            // ベース線の端点と一致する高さ文字を見つけて影の線を作成する
            for(var i = 0; i < polygonPointCount - 1; i++)
            {
                Point2d basePoint1 = baseLinePolygon.polygonPoints[i];
                Point2d basePoint2 = baseLinePolygon.polygonPoints[i + 1];

                string heightText1 = "";
                string heightText2 = "";

                int drawingID = baseLinePolygon.drawingID;
                int layerID = baseLinePolygon.layerID;

                // 高さを探す
                for(var j = 0; j < heightTextShapeList.Count; j++)
                {
                    TextShape textShape = heightTextShapeList[j];

                    // 部分図とレイヤも同じもの
                    if(drawingID != textShape.Drawing.ID || layerID != textShape.LayerID)
                        continue;

                    Point2d textPoint = textShape.Point;
                    if(heightText1 == "" && IsPointEquals(basePoint1, textPoint))
                        heightText1 = textShape.Text;
                    else if(heightText2 == "" && IsPointEquals(basePoint2, textPoint))
                        heightText2 = textShape.Text;

                    if(heightText1 != "" && heightText2 != "")
                        break;
                }

                Debug.Assert(heightText1 != "" && heightText2 != "", "両端点の高さが見つかるはず。見つからない場合は図面上で高さが正しくセットできていない。");

                // 高さ
                // レイヤ―グループとレイヤ名にあるレベルを取得してその値も反映させる
                Layer layer = layerTable.GetLayerByID(layerID);
                Layer parentlayer = layer.ParentLayer;
                //Layer parentlayer2 = parentlayer.ParentLayer;
                //if(parentlayer2 != null)
                //{
                //   Debug.Assert("レイヤグループは1階層までが前提、セットされた高さレベルはその図形のグループ名からのものだけ反映させます。");
                //}
                string layerName = layer.Name;
                string layerGroupName = parentlayer.Name;// ルートの場合は空 parentlayerID == -1

                double heightValue1 = 0.0;
                double bottomValue1 = 0.0;
                double heightValue2 = 0.0;
                double bottomValue2 = 0.0;

                // 高さを表す文字から高さの値を取得
                try
                {
                    if(!GetHeightValue(heightText1, layerGroupName, layerName, locationHeight, out heightValue1, out bottomValue1))
                        continue;
                }
                catch
                {
                    continue;
                }

                try
                {
                    if(!GetHeightValue(heightText2, layerGroupName, layerName, locationHeight, out heightValue2, out bottomValue2))
                        continue;
                }
                catch
                {
                    continue;
                }

                bool isOverhang = false;// オーバーハングフラグ

                // 四角形の1点目と2点目
                Point2d p0 = basePoint1;
                Point2d p1 = p0;
                if(bottomValue1 > RcPrecisionConfusion)// オーバーハング状態の場合
                {
                    isOverhang = true;

                    p0.X += bottomValue1 * xMagnification;
                    p0.Y += bottomValue1 * yMagnification;
                    // 真北に合わせて回転
                    p0 = app.Geometry.RotatePoint(p0, basePoint1, dueNorthDeg);
                }
                p1.X += heightValue1 * xMagnification;
                p1.Y += heightValue1 * yMagnification;
                // 真北に合わせて回転
                p1 = app.Geometry.RotatePoint(p1, basePoint1, dueNorthDeg);

                // 四角形の3点目と4点目
                Point2d p3 = basePoint2;
                Point2d p2 = p3;
                if(bottomValue2 > RcPrecisionConfusion)// オーバーハング状態の場合
                {
                    isOverhang = true;

                    p3.X += bottomValue2 * xMagnification;
                    p3.Y += bottomValue2 * yMagnification;
                    // 真北に合わせて回転
                    p3 = app.Geometry.RotatePoint(p3, basePoint2, dueNorthDeg);
                }
                p2.X += heightValue2 * xMagnification;
                p2.Y += heightValue2 * yMagnification;
                // 真北に合わせて回転
                p2 = app.Geometry.RotatePoint(p2, basePoint2, dueNorthDeg);

                // p0-p1, p2-p3が重なったり交差している場合は追加しない
                // p1とp2が同じ以外で（四角形または三角形になるものだけ)
                if(!IsPointEquals(p1, p2))
                {
                    if(IsPointEquals(p0, p1))
                        continue;

                    LineShape line1 = app.ShapeFactory.CreateLine(p0, p1);

                    if(IsPointEquals(p3, p2))
                        continue;

                    LineShape line2 = app.ShapeFactory.CreateLine(p3, p2);

                    IntersectionPoint[] intersectionPoints = line1.GetIntersectionPoint(line2);
                    int intersectionPointsCount = intersectionPoints.Length;
                    if(intersectionPointsCount > 0)
                        continue;
                }

                // 四角形作成
                Quadrilateral quad = new Quadrilateral(app, solarTime, p0, p1, p2, p3, isOverhang);
                quads.Add(quad);

            }

            return quads;
        }



        /// <summary>
        /// 高さを表す文字から高さの値を取得(レイヤグループ名、レイヤ名のレベル値も加算)
        /// カンマ区切りの2つまでの数値文字で指定できる
        /// 大きい方が上、小さい方が下の高さ(オーバーハング)(ex. 3.0, 6.0 -> heightTop = 6.0, heightBottom = 3.0)
        /// </summary>
        /// <param name="heightText"></param>
        /// <param name="layerGroupName"></param>
        /// <param name="layerName"></param>
        /// <param name="locationHeight">測定面の高さ</param>
        /// <param name="heightTop"></param>
        /// <param name="heightBottom"></param>
        /// <returns></returns>
        private bool GetHeightValue(string heightText, string layerGroupName, string layerName, double locationHeight, out double heightTop, out double heightBottom)
        {
            heightTop = 0.0;
            heightBottom = 0.0;

            // レイヤグループ名のレベル値
            bool isLayerGroupLevel = GetLayerLevelValue(layerGroupName, out double layerGroupLevelValue);

            // レイヤ名のレベル値
            bool isLayerLevel = GetLayerLevelValue(layerName, out double layerLevelValue);

            // 高さ文字からの値取得
            string[] arr = heightText.Split(',');
            if(arr.Length > 2)// 2つまで
                throw new ArgumentException("heightText arr.Length > 2");

            bool isHeightBottom = false;
            for(int i = 0; i < arr.Length; i++)
            {
                string toText = arr[i];
                toText = toText.TrimStart();
                // 文字列をdouble型に変換
                if(!double.TryParse(toText, out double height))
                    continue;

                if(i == 0)
                {
                    heightTop = height;
                }
                else if(i == 1)
                {
                    heightBottom = height;
                    isHeightBottom = true;
                }
            }

            //↓↓↓↓↓↓ 2025/10/8 ↓↓↓↓↓↓
            // 2つの値があって、2つ目の方が大きい値の場合上下入れ替え
            if(isHeightBottom && heightBottom > heightTop)
            {
                double tempHeightTop = heightTop;
                heightTop = heightBottom;
                heightBottom = tempHeightTop;
            }

            // 1つ目だけでマイナスの場合は絶対値にしてheightBottomも同じ値にする
            if(!isHeightBottom && heightTop < 0.0)
            {
                heightTop *= -1.0;
                heightBottom = heightTop;
                isHeightBottom = true;
            }
            //↑↑↑↑↑↑ 2025/10/8 ↑↑↑↑↑↑

            if(isLayerGroupLevel)
            {
                heightTop += layerGroupLevelValue;
                if(isHeightBottom)
                    heightBottom += layerGroupLevelValue;
            }

            if(isLayerLevel)
            {
                heightTop += layerLevelValue;
                if(isHeightBottom)
                    heightBottom += layerLevelValue;
            }

            // 測定面の高さを引く
            heightTop -= locationHeight;
//          if(heightTop < 0.0)
//          {
//              throw new ArgumentException("heightTop < 0.0");
//          }

            if(isHeightBottom)
            {
                heightBottom -= locationHeight;
//              Debug.Assert(heightBottom >= 0.0);
//              if(heightBottom < 0.0)
//                    throw new ArgumentException("heightBottom < 0.0");
            }

            heightTop *= 1000.0; // m -> mm
            heightBottom *= 1000.0; // m -> mm

        //    if(heightTop < 0.0 || heightBottom < 0.0)
        //        throw new ArgumentException("heightTop < 0.0 || heightBottom < 0.0");

            return true;
        }

        /// <summary>
        /// レイヤ名からレベル値取得
        /// </summary>
        /// <param name="layerName"></param>
        /// <param name="layerLevelValue"></param>
        /// <returns></returns>
        private bool GetLayerLevelValue(string layerName, out double layerLevelValue)
        {
            layerLevelValue = 0.0;

            int findIndex = layerName.IndexOf("#lv");
            if(findIndex >= 0 && layerName.Length > findIndex + 3 + 1)
            {
                string str = layerName.Substring(findIndex + 3 + 1);
                str = Regex.Replace(str, @"[^0-9.]", " ");
                str.Trim();

                if(!double.TryParse(str, out layerLevelValue))
                    return false;
            }

            return true;
        }

        /// <summary>
        /// 複数の線図形のある矩形範囲BndBox取得
        /// </summary>
        /// <param name="linesList"></param>
        /// <returns></returns>
        public BndBox GetLinesBndBox(List<List<BaseLine>> linesList)
        {
            Debug.Assert(linesList.Count > 0);
            BndBox bndBox = null;
            LineShape line = null;

            foreach(List<BaseLine> lines in linesList)
            {
                Debug.Assert(lines.Count > 0);

                for(int i = 0; i < lines.Count; i++)
                {
                    line = lines[i].line;

                    if(bndBox == null)
                    {
                        bndBox = new BndBox(line.StartPoint, line.EndPoint);
                    }
                    else
                    {
                        bndBox.Add(line.StartPoint);
                        bndBox.Add(line.EndPoint);
                    }
                }
            }

            return bndBox;
        }

        /// <summary>
        /// 指定した角度で回転させた複数の線図形の矩形範囲BndBox取得
        /// </summary>
        /// <param name="linesList"></param>
        /// <param name="orgPoint"></param>
        /// <param name="rotateDeg"></param>
        /// <returns></returns>
        public BndBox GetRotateLinesBndBox(List<List<BaseLine>> linesList, Point2d orgPoint, double rotateDeg)
        {
            Debug.Assert(linesList.Count > 0);
            BndBox bndBox = null;
            LineShape line = null;
            Point2d startPoint = new Point2d();
            Point2d endPoint = new Point2d();

            foreach(List<BaseLine> lines in linesList)
            {
                Debug.Assert(lines.Count > 0);

                for(int i = 0; i < lines.Count; i++)
                {
                    line = lines[i].line;

                    startPoint.X = line.StartPoint.X;
                    startPoint.Y = line.StartPoint.Y;
                    endPoint.X = line.EndPoint.X;
                    endPoint.Y = line.EndPoint.Y;
                    startPoint = app.Geometry.RotatePoint(startPoint, orgPoint, rotateDeg);// 回転
                    endPoint = app.Geometry.RotatePoint(endPoint, orgPoint, rotateDeg);// 回転

                    if(bndBox == null)
                    {
                        bndBox = new BndBox(startPoint, endPoint);
                    }
                    else
                    {
                        bndBox.Add(startPoint);
                        bndBox.Add(endPoint);
                    }
                }
            }

            return bndBox;
        }


        /// <summary>
        /// 複数の線図形のある矩形範囲の中心座標取得
        /// </summary>
        /// <param name="linesList"></param>
        /// <returns></returns>
        public Point2d GetLinesBoxCenterPoint(List<List<BaseLine>> linesList)
        {
            BndBox bndBox = GetLinesBndBox(linesList);

            // 矩形範囲の左下座標、右上座標
            Point2d[] points = bndBox.GetBoundingBox();
            Point2d boxCenterPoint = CenterPoint(points[0], points[1]);

            return boxCenterPoint;
        }

    
       /// <summary>
       /// ベース線からなる多角形ごとに影の外周線リストを作る
       /// </summary>
       /// <param name="doc"></param>
       /// <param name="baseLinesPolygonList"></param>
       /// <param name="heightTextShapeList"></param>
       /// <param name="solarTime"></param>
       /// <param name="xMagnification"></param>
       /// <param name="yMagnification"></param>
       /// <param name="locationHeight"></param>
       /// <param name="dueNorthDeg"></param>
       /// <param name="shadowQuads"></param>
       /// <param name="trimFlg"></param>
       /// <returns></returns>       
       private List<List<LineShape>> GetShadowOutlinesList(Document doc, List<BaseLinePolygon> baseLinesPolygonList, List<TextShape> heightTextShapeList, double solarTime, double xMagnification, double yMagnification, double locationHeight, double dueNorthDeg, List<Quadrilateral> shadowQuads, bool trimFlg)
       {
            // ベース線からできる影の四角リスト
            if(shadowQuads == null)
                shadowQuads = new List<Quadrilateral>();

            List<List<LineShape>> shadowWireFrameLinesList = new List<List<LineShape>>();
            foreach(BaseLinePolygon baseLinePolygon in baseLinesPolygonList)
            {
                // ベース線と高さ文字情報からできる影の四角形領域のリスト作成取得
                var quads = GetQuadrilateralList(doc, baseLinePolygon, heightTextShapeList, solarTime, xMagnification, yMagnification, locationHeight, dueNorthDeg);
                shadowQuads.AddRange(quads);

                // 四角形領域を重ねたワイヤーフレーム線を取得
                List<LineShape> shadowWireFrameLines = GetShadowWireFrameLines(quads);
                shadowWireFrameLinesList.Add(shadowWireFrameLines);
            }

            Debug.Assert(shadowWireFrameLinesList.Count == baseLinesPolygonList.Count);

            // ワイヤーフレーム線（それぞれのベース線でできたそれぞれの影の四角形領域の線を重ねた複数の線）から最も外側になる線だけを作る
            List<List<LineShape>> shadowOutlinesList = new List<List<LineShape>>();
            foreach(List<LineShape> shadowWireFrameLines in shadowWireFrameLinesList)
            {
                // ワイヤーフレーム線の分解
                // 線同士で交点がある場合はその線を交点の位置で分ける
                List<LineShape> cutLines = CutShadowWireFrameLines(shadowWireFrameLines, shadowQuads);
                if(cutLines.Count == 0)
                    continue;

                // 陰線を消す
                List<LineShape> shadowOutlines1 = EraseHiddenLines(cutLines, shadowQuads, baseLinesPolygonList);
                if(shadowOutlines1.Count == 0)
                    continue;

                // 重なっている線を消す
                List<LineShape> shadowOutlines2 = EraseOverlappingLines(shadowOutlines1);
                if(shadowOutlines2.Count == 0)
                    continue;

                // ベース線の端点と接続してしまうところはトリムして少し短くする(見た目と、端点が高さ文字の配置点と同じにならないようにするだけ)
                if(trimFlg)
                {
                    List<LineShape> shadowOutlines3 = TrimFrameLines(shadowOutlines2, baseLinesPolygonList);
                    if(shadowOutlines3.Count == 0)
                        continue;

                    // チェック
                    //shadowOutlines3 = shadowWireFrameLines;// ワイヤーフレームで確認
                    //shadowOutlines3 = cutLines;//　陰線しない状態で確認

                    // リストに追加
                    if(shadowOutlines3.Count > 0)
                        shadowOutlinesList.Add(shadowOutlines3);
                }
                else
                {
                    // リストに追加
                    if(shadowOutlines2.Count > 0)
                        shadowOutlinesList.Add(shadowOutlines2);
                }
            }

            return shadowOutlinesList;
       }
    
        /// <summary>
        /// 四角形を重ねたワイヤーフレーム線の取得
        /// </summary>
        /// <param name="quads"></param>
        /// <returns></returns>
        private List<LineShape> GetShadowWireFrameLines(List<Quadrilateral> quads)
        {
            Document doc = app.ActiveDocument;		//ドキュメントの取得

            var shadowWireFrameLines = new List<LineShape>();

            for(int i = 0; i < quads.Count; i++)
            {
                Quadrilateral quad = quads[i];

                int forSize = 3;
                // オーバーハング状態の場合はp0とp3を結ぶ
                if(quad.isOverhang)
                    forSize = 4;

                for(int j = 0; j < forSize; j++)
                {
                    LineShape line1 = GetQuadrilateralLine(quad, j);
                    Point2d startPoint1 = line1.StartPoint;
                    Point2d endPoint1 = line1.EndPoint;

                    bool already = false;

                    // 既にリストに同じ線があれば追加しない
                    for(int k = 0; k < shadowWireFrameLines.Count; k++)
                    {
                        LineShape line2 = shadowWireFrameLines[k];
                        Point2d startPoint2 = line2.StartPoint;
                        Point2d endPoint2 = line2.EndPoint;

                        // 端点が一致
                        if((IsPointEquals(startPoint1, startPoint2) && IsPointEquals(endPoint1, endPoint2))
                            || (IsPointEquals(startPoint1, endPoint2) && IsPointEquals(endPoint1, startPoint2)))
                        {
                            already = true;
                            break;
                        }
                    }

                    if(!already)
                    {
                        doc.BasicShapeSettings.CopyToShape(line1);
                        shadowWireFrameLines.Add(line1);
                    }
                }
            }

            return shadowWireFrameLines;
        }

        /// <summary>
        ///  線同士で交点がある場合はその線を交点の位置で分ける
        /// </summary>
        /// <param name="shadowWireFrameLines"></param>
        /// <param name="quads"></param>
        /// <returns></returns>
        private List<LineShape> CutShadowWireFrameLines(List<LineShape> shadowWireFrameLines, List<Quadrilateral> quads)
        {
            var cutLines = new List<LineShape>();

            for(int i = 0; i < shadowWireFrameLines.Count; i++)
            {
                LineShape line1 = shadowWireFrameLines[i];
                var crossPointsParameters = new List<double>();

                // 交点を求める（四角形の全辺を使う）
                for(int j = 0; j < quads.Count; j++)
                {
                    Quadrilateral quad = quads[j];
                    for(int k = 0; k < 4; k++)
                    {
                        LineShape line2 = GetQuadrilateralLine(quad, k);

                        IntersectionPoint[] intersectionPoints = line1.GetIntersectionPoint(line2);
                        int intersectionPointsCount = intersectionPoints.Length;
                        for(int l = 0; l < intersectionPointsCount; l++)
                        {
                            // 交点を求める
                            IntersectionPoint intersectionPoint = intersectionPoints[l];
                            double param = intersectionPoint.PointOnShape1.Parameter;
                            crossPointsParameters.Add(param);
                        }
                    }
                }


                // 交点がない場合はそのまま追加
                if(crossPointsParameters.Count == 0)
                {
                    cutLines.Add(line1);
                }
                else
                {
                    // 交点のパラメータから昇順に並び替えた点座標取得(線の始点終点も含む)
                    List<Point2d> points = GetPointListAtParameters(line1, crossPointsParameters);

                    for(int j = 0; j < points.Count - 1; j++)
                    {
                        Point2d point1 = points[j];
                        Point2d point2 = points[j + 1];

                        Debug.Assert(!IsPointEquals(point1, point2));
                        LineShape cutLine = (LineShape)line1.Clone();
                        cutLine.SetPoint(point1, point2);
                        cutLines.Add(cutLine);
                    }
                }
            }

            return cutLines;
        }

        /// <summary>
        /// 線上のパラメータから昇順に並び替えた点座標取得(線の始点終点も加える)
        /// </summary>
        /// <param name="line"></param>
        /// <param name="parameters"></param>
        /// <returns></returns>
        private List<Point2d> GetPointListAtParameters(LineShape line, List<double> crossPointsParameters)
        {
            // パラメータを昇順に並び替える
            crossPointsParameters.Sort();

            var points = new List<Point2d>();
            Point2d startPoint = line.GetPointAtParameter(line.StartParameter);
            Point2d endPoint = line.GetPointAtParameter(line.EndParameter);
            Point2d point = startPoint;
            points.Add(point);
            Point2d lastPoint = point;

            for(int i = 0; i < crossPointsParameters.Count; i++)
            {
                double param = crossPointsParameters[i];
                point = line.GetPointAtParameter(param);

                // 前の座標と同じ点は追加しない
                if(IsPointEquals(lastPoint, point))
                    continue;

                points.Add(point);
                lastPoint = point;
            }

            if(!IsPointEquals(lastPoint, endPoint))
                points.Add(endPoint);

            return points;
        }

        /// <summary>
        /// 陰線を消す
        /// </summary>
        /// <param name="cutLines"></param>
        /// <param name="quads"></param>
        /// <param name="baseLinesPolygonList</param>
        /// <returns></returns>
        private List<LineShape> EraseHiddenLines(List<LineShape> lines, List<Quadrilateral> quads, List<BaseLinePolygon> baseLinesPolygonList)
        {
            var shadowOutlines = new List<LineShape>();

            // 線の中点が、いずれかの四角形の内部にあるなら追加しない
            for(int i = 0; i < lines.Count; i++)
            {
                LineShape line = lines[i];

                // 中点
                Point2d centerPoint = CenterPoint(line.StartPoint, line.EndPoint);

                int firstPointOnLineQuadIndex = -1;// 最初に中点が四角形のポリゴン上となった四角形のインデックス初期値
                int onPolygonCount = 0;
                bool erase = false;
                for(int j = 0; j < quads.Count; j++)
                {
                    Quadrilateral quad = quads[j];
                    Point2d[] points = quad.GetPoints();

                    // 四角形に含まれるか
                    int result = app.Geometry.ContainsPointInPolygon(centerPoint, points);
                    // ポリゴンの中だった場合
                    if(Math.Abs(result) == 2)
                    {
                        erase = true;
                        break;
                    }
                    // ポリゴンの線上
                    else if(Math.Abs(result) == 1)
                    {
                        onPolygonCount++;

                        // ここに2回(2つの四角形の線上)きた場合、それらの四角が重ならない状態ならその中点の線は消してよい
                        if(firstPointOnLineQuadIndex < 0)
                        {
                            firstPointOnLineQuadIndex = j;
                        }
                        else
                        {
                            // 前回の四角形と今回の四角形は一辺が一致しているだけで面が重なることはないか
                            // 重ならないなら消してよい
                            if(!IsOverlap(quads[firstPointOnLineQuadIndex], quad))
                            {
                                erase = true;
                                break;
                            }
                        }
                    }

                }

                // ベース線でできた多角形で線が消えるか。
                foreach(BaseLinePolygon baseLinePolygon in baseLinesPolygonList)
                {
                    // オーバーハング状態の場合はそのまま
                    if(baseLinePolygon.isOverhang)
                        continue;

                    List<Point2d> polygonPoints = baseLinePolygon.polygonPoints;
                    Point2d[] points2 = polygonPoints.ToArray();

                    // 四角形に含まれるか。
                    int result = app.Geometry.ContainsPointInPolygon(centerPoint, points2);
                    // ポリゴンの中だった場合
                    if(Math.Abs(result) == 2)
                    {
                        erase = true;
                        break;
                    }

                    // 始点も終点も同じ線上の場合も消す
                    int points2Count = polygonPoints.Count;
                    for(int j = 0; j < points2Count - 1; j++)
                    {
                        Point2d p21 = polygonPoints[j];
                        Point2d p22 = polygonPoints[j + 1];// (polygonPointsは閉じていない場合もある)

                        LineShape line2 = app.ShapeFactory.CreateLine(p21, p22);

                        if(line2.IsPointOnShape(line.StartPoint) && line2.IsPointOnShape(line.EndPoint))
                        {
                            erase = true;
                            break;
                        }
                    }

                    if(erase)
                        break;
                }

                // 消えなかった線を追加
                if(!erase)
                    shadowOutlines.Add(line);
            }

            return shadowOutlines;
        }

        /// <summary>
        /// 重なっている線を除く
        /// </summary>
        /// <param name="lines"></param>
        /// <returns></returns>
        private List<LineShape> EraseOverlappingLines(List<LineShape> lines)
        {
            var lines2 = new List<LineShape>();

            // 線の始点と終点が他の線上にある場合は除く
            int linesCount = lines.Count;
            for(int i = 0; i < linesCount; i++)
            {
                LineShape line1 = lines[i];
                bool overlap = false;

                for(int k = i + 1; k < linesCount; k++)
                {
                    LineShape line2 = lines[k];
                    if(line2.IsPointOnShape(line1.StartPoint) && line2.IsPointOnShape(line1.EndPoint))
                    {
                        overlap = true;
                        break;
                    }
                }

                if(!overlap)
                    lines2.Add(line1);
            }

            return lines2;
        }


        /// <summary>
        /// 端点が他の線と繋がっていない線のインデックス取得
        /// </summary>
        /// <param name="lines"></param>
        /// <param name="singlePoint"></param>
        /// <param name="conectPoint"></param>
        /// <returns>始点か終点が他の線と繋がっていない最初の線のインデックスを返します。なかった場合は-1</returns>
        private int FindeFirstSingleConectLine(List<BaseLine> lines, out Point2d singlePoint, out Point2d conectPoint)
        {
            // 初期値
            int singleConectLineIndex = -1;// 端点が他と繋がっていない線のインデックス
            singlePoint = new Point2d(0.0, 0.0);   // 端点が他と繋がっていない座標
            conectPoint = new Point2d(0.0, 0.0);   // 端点が他と繋がっていない点を持つ線のその反対側の端点の座標(他と繋がっている座標)

            int linesCount = lines.Count;
            Debug.Assert(linesCount >= 1);

            // 始めの1本目の線分を決める（閉じていない場合があるので、線分のどちらかの端点が他の線とつながっていないものがあればそれからスタート）
            bool close = false;
            int checkLineIndex = 0;
            Point2d checkPoint = new Point2d(0.0, 0.0);
            for(int i = 0; i < linesCount; i++)
            {
                LineShape line1 = lines[i].line;
                checkLineIndex = 0;

                for(int j = 0; j < 2; j++)
                {
                    // まず、線1の始点が他の線とつながっていないか見る
                    checkPoint = line1.StartPoint;
                    if(j == 1)
                        checkPoint = line1.EndPoint;

                    for(int k = i + 1; k < linesCount; k++)
                    {
                        LineShape line2 = lines[k].line;
                        Point2d startPoint2 = line2.StartPoint;
                        Point2d endPoint2 = line2.EndPoint;

                        // 一致する端点があった場合
                        if(IsPointEquals(checkPoint, startPoint2) || IsPointEquals(checkPoint, endPoint2))
                        {
                            // 線1の両方が他の線の端点と一致していた場合は閉じていると判断
                            if(j == 1)
                                close = true;

                            break;
                        }
                    }

                    if(close)
                        break;
                }

                // 閉じていなかった場合
                if(!close)
                {
                    singleConectLineIndex = checkLineIndex;
                    singlePoint = checkPoint;
                    LineShape line = lines[singleConectLineIndex].line;
                    // 一致しない側の点が他と繋がっている点
                    if(IsPointEquals(singlePoint, line.StartPoint))
                        conectPoint = line.EndPoint;
                    else
                        conectPoint = line.StartPoint;

                    break;
                }
            }

            return singleConectLineIndex;
        }

        /// <summary>
        /// オーバーハング状態(地盤面から浮いた状態)か
        /// </summary>
        /// <param name="doc"></param>
        /// /// <param name="locationHeight">測定面の高さ</param>
        /// <param name="baseLineList"></param>
        /// <returns></returns>
        private bool IsOverhangBaseLines(Document doc, double locationHeight, List<BaseLine> baseLineList)
        {
            LayerTable layerTable = doc.LayerTable;

            foreach(BaseLine baseLine in baseLineList)
            {
                // リスト中の線に付随する高さ文字からオーバーハング状態の値があるか調べる
                if(baseLine.startPointSideHeightText != null || baseLine.endPointSideHeightText != null)
                {
                    int layerID = baseLine.line.LayerID;
                    Layer layer = layerTable.GetLayerByID(layerID);
                    Layer parentlayer = layer.ParentLayer;
                    string layerName = layer.Name;
                    string layerGroupName = parentlayer.Name;
                   
                    if(baseLine.startPointSideHeightText != null)
                    {
                        double heightTop = 0.0;
                        double heightBottom = 0.0;

                        try
                        {
                            if(GetHeightValue(baseLine.startPointSideHeightText.Text, layerGroupName, layerName, locationHeight, out heightTop, out heightBottom))
                            {
                                // オーバーハング状態(地面から浮いている)
                                if(heightBottom > RcPrecisionConfusion)
                                {
                                    return true;
                                }
                            }
                        }
                        catch
                        {
                            return false;
                        }
                    }

                    if(baseLine.endPointSideHeightText != null)
                    {
                        double heightTop = 0.0;
                        double heightBottom = 0.0;

                        try
                        {
                            if(GetHeightValue(baseLine.endPointSideHeightText.Text, layerGroupName, layerName, locationHeight, out heightTop, out heightBottom))
                            {
                                // オーバーハング状態(地面から浮いている)
                                if(heightBottom > RcPrecisionConfusion)
                                {
                                    return true;
                                }
                            }
                        }
                        catch
                        {
                            return false;
                        }
                    }
                }

            }

            return false;
        }

       /// <summary>
       /// ベース線からなる多角形のリスト作成
       /// </summary>
       /// <param name="doc"></param>
       /// <param name="locationHeight">測定面の高さ</param>
       /// <param name="baseLineList"></param>
       /// <returns></returns>
        private List<BaseLinePolygon> GetBaseLinesPolygonList(Document doc, double locationHeight, List<BaseLine> baseLineList)
        {
            List<BaseLinePolygon> baseLinesPolygonList = new List<BaseLinePolygon>();// ベース線からなる多角形のリスト

            // ベース線でできる多角形を求める(※多角形は一筆書きできるもので、途中の端点が複数の端点と一致することはないが閉じていない場合はある)
            List<BaseLine> remainingLineList = new List<BaseLine>(baseLineList);// リストのコピー

            bool isOverhang = IsOverhangBaseLines(doc, locationHeight, remainingLineList); // オーバーハング状態(地盤面から浮いた状態)か

            int linesCount = remainingLineList.Count;
            Debug.Assert(linesCount >= 1);

            BaseLinePolygon baseLinePolygon = null;
            Point2d firstPoint = new Point2d(0.0, 0.0);
            Point2d nextEndPoint = new Point2d(0.0, 0.0);
            bool firstFlg = true;

            while (linesCount > 0)
            {
                if(firstFlg)
                {
                    int startLineIndex = 0;// 初期値
                    LineShape line1 = remainingLineList[startLineIndex].line;
                    Point2d startPoint1 = line1.StartPoint;
                    Point2d endPoint1 = line1.EndPoint;
                    baseLinePolygon = new BaseLinePolygon(line1.Drawing.ID, line1.LayerID, isOverhang);

                    // 始点か終点が他の線と繋がっていない最初の線があればそれをスタートの線にする
                    int singleConectLineIndex = FindeFirstSingleConectLine(remainingLineList, out Point2d singlePoint, out Point2d conectPoint);
                    if(singleConectLineIndex >= 0)
                    {
                        line1 = remainingLineList[singleConectLineIndex].line;
                        startPoint1 = singlePoint;
                        endPoint1 = conectPoint;
                        startLineIndex = singleConectLineIndex;
                    }

                    firstPoint = startPoint1;
                    nextEndPoint = endPoint1;

                    baseLinePolygon.polygonPoints.Add(startPoint1);
                    baseLinePolygon.polygonPoints.Add(endPoint1);

                    remainingLineList.RemoveAt(startLineIndex);// 最初の線を削除

                    firstFlg = false;
                }

                Debug.Assert(baseLinePolygon != null);

                bool allSearched = false; // 最後まで探したフラグ
                linesCount = remainingLineList.Count;
                for(int i = linesCount - 1 ; i >= 0 ; i--)
                {
                    LineShape line2 = remainingLineList[i].line;
                    Point2d startPoint2 = line2.StartPoint;
                    Point2d endPoint2 = line2.EndPoint;

                    // 繋がるか
                    if(IsPointEquals(nextEndPoint, startPoint2) || IsPointEquals(nextEndPoint, endPoint2))
                    {
                        if(IsPointEquals(nextEndPoint, startPoint2))
                        {
                            baseLinePolygon.polygonPoints.Add(endPoint2);
                            nextEndPoint = endPoint2;
                        }
                        else
                        {
                            baseLinePolygon.polygonPoints.Add(startPoint2);
                            nextEndPoint = startPoint2;
                        }

                        remainingLineList.RemoveAt(i);
                        break;
                    }

                    if(i == 0)
                        allSearched = true;
                }

                linesCount = remainingLineList.Count;

                // 最初の点に繋がったかか、最後まで探したがそれ以上繋がらない場合
                if(IsPointEquals(firstPoint, nextEndPoint) || allSearched || linesCount == 0)
                {
                    if(baseLinePolygon.polygonPoints.Count > 0)
                        baseLinesPolygonList.Add(baseLinePolygon); // ここまでのポリゴンをリストに追加

                    firstFlg = true;// 次のポリゴンを探すためにフラグ変更
                }

            }

            return baseLinesPolygonList;
        }

        /// <summary>
        /// 二つの四角形が重なるっているか
        /// </summary>
        /// <param name="quad1">四角形1</param>
        /// <param name="quad2">四角形2</param>
        /// <returns></returns>
        private Boolean IsOverlap(Quadrilateral quad1, Quadrilateral quad2)
        {
            Point2d[] points1 = quad1.GetPoints();
            Point2d[] points2 = quad2.GetPoints();

            for(int i = 0 ; i < 4 ; i++)
            {
                Point2d p11;
                Point2d p12;

                p11 = points1[i];
                p12 = points1[i + 1];

                LineShape line1 = app.ShapeFactory.CreateLine(p11, p12);
                double line1StartParam = line1.StartParameter;
                double line1EndParam = line1.EndParameter;


                for(int j = 0 ; j < 4 ; j++)
                {
                    Point2d p21;
                    Point2d p22;

                    p21 = points2[j];
                    p22 = points2[j + 1];

                    LineShape line2 = app.ShapeFactory.CreateLine(p21, p22);
                    double line2StartParam = line2.StartParameter;
                    double line2EndParam = line2.EndParameter;

                    // 交点があった場合
                    IntersectionPoint[] intersectionPoints = line1.GetIntersectionPoint(line2);
                    if(intersectionPoints.Length == 1)
                    {
                        IntersectionPoint intersectionPoint = intersectionPoints[0];
                        double param = intersectionPoint.PointOnShape1.Parameter;
                        // 始点・終点は除く
                        if(app.Geometry.IsParameterEquals(line1StartParam, param) || app.Geometry.IsParameterEquals(line1EndParam, param))
                            continue;

                        return true;
                    }
                    else if(intersectionPoints.Length == 2)
                    {
                        IntersectionPoint intersectionPoint0 = intersectionPoints[0];
                        double param0 = intersectionPoint0.PointOnShape1.Parameter;

                        IntersectionPoint intersectionPoint1 = intersectionPoints[1];
                        double param1 = intersectionPoint1.PointOnShape2.Parameter;

                        // 始点・終点は除く
                        if(app.Geometry.IsParameterEquals(line1StartParam, param0) || app.Geometry.IsParameterEquals(line1EndParam, param0))
                        {
                            if(app.Geometry.IsParameterEquals(line2StartParam, param1) || app.Geometry.IsParameterEquals(line2EndParam, param1))
                                continue;
                        }

                        return true;
                    }

                  }

            }
            
            // 交点がなかった場合は重なっていないと判断する
            return false;
        }

        /// <summary>
        /// ベース線の端点と接続してしまうところはトリムして少し短くする
        /// </summary>
        /// <param name="shadowOutlines"></param>
        /// <param name="baseLinesPolygonList"></param>
        /// <returns></returns>
        private List<LineShape> TrimFrameLines(List<LineShape> shadowOutlines, List<BaseLinePolygon> baseLinesPolygonList)
        {
            List<LineShape> trimFrameLines = new List<LineShape>();

            int shadowOutlinesCount = shadowOutlines.Count;
            for(int i = 0 ; i < shadowOutlinesCount ; i++)
            {
                LineShape line = shadowOutlines[i];
                Point2d startPoint = line.StartPoint;
                Point2d endPoint = line.EndPoint;

                bool trimed = false;
                bool erase = false;
                foreach (BaseLinePolygon baseLinePolygon in baseLinesPolygonList)
                {
                    List<Point2d> polygonPoints = baseLinePolygon.polygonPoints;

                    for(int j = 0 ; j < polygonPoints.Count ; j++)
                    {
                        // 線の端点がベース線の端点と一致する場合のみトリムする(短くする)
                        Point2d polygonPoint = polygonPoints[j];

                        if(IsPointEquals(polygonPoint, startPoint) || IsPointEquals(polygonPoint, endPoint))
                        {
                            double lineLength = line.Length;
                            double cutLength = customSettings.shadowLineTrimLength;//短くする長さ
                            double newLineLength = lineLength - cutLength;
                            if(newLineLength > 0.0)
                            {
                                bool cutStartPointSide = false;
                                if(IsPointEquals(polygonPoint, startPoint))
                                    cutStartPointSide = true;

                                // 固定する側の点座標とパラメータ（初期値、始点側を固定）
                                Point2d fixPoint = line.StartPoint;
                                double fixParam = line.StartParameter;
                                // 終点側を固定（始点側を短く）する場合
                                if(cutStartPointSide)
                                {
                                    fixPoint = line.EndPoint;
                                    fixParam = line.EndParameter;
                                    newLineLength *= -1.0;
                                }

                                PointOnShape pointOnShape = line.GetPointAtLength(fixParam, newLineLength);
                                Point2d newPoint = pointOnShape.Point;
                                if(!IsPointEquals(fixPoint, newPoint))
                                {
                                    LineShape trimLine = (LineShape)line.Clone();

                                    if(cutStartPointSide)
                                        trimLine.SetPoint(newPoint, fixPoint);
                                    else
                                        trimLine.SetPoint(fixPoint, newPoint);

                                    // リストに追加
                                    trimFrameLines.Add(trimLine);
                                    trimed = true;
                                }
                                // 点が同じになる場合は消す
                                else
                                {
                                    erase = true;
                                }
                            }
                            // 長さがマイナスになる場合も消す
                            else
                            {
                                erase = true;
                            }

                            break;
                        }
                    }

                    if(trimed || erase)
                        break;
                }

                // トリムや消す対象外だったものはそのままリストに追加
                if(!trimed && !erase)
                    trimFrameLines.Add(line);
            }

            return trimFrameLines;
        }

        /// <summary>
        /// 時間の値から文字作成
        /// </summary>
        /// <param name="timeValue"></param>
        /// <returns></returns>
        private TextShape CreateTimeText(double timeValue)
        {
            Document doc = app.ActiveDocument;		//ドキュメントの取得
            Debug.Assert(doc != null);

            string solarTimeText = timeValue.ToString("F");
            //if(solarTimeText.LastIndexOf(".00") == solarTimeText.Length - 3)
            // solarTimeText = solarTimeText.Replace(".00", ".0");
            // 小数点以下第2位が0の場合は消す
            if(solarTimeText.LastIndexOf(".") == solarTimeText.Length - 3 && solarTimeText.Substring(solarTimeText.Length - 1) == "0")
                solarTimeText = solarTimeText.Substring(0, solarTimeText.Length - 1);

            Point2d textLocatePoint = new Point2d(0.0, 0.0);
            double textAngle = 0.0;
            TextShape textTime = app.ShapeFactory.CreateText(solarTimeText, textLocatePoint, textAngle);
            doc.TextSettings.CopyToShape(textTime);
            textTime.DirectionVertical = false;// 横書き

            textTime.FontHeight = GetTextFontHeight() * customSettings.shadowTimeTextHightRatio;// 文字サイズを調整 

            return textTime;
        }

        /// <summary>
        /// 時間表記文字の配置点と角度取得
        /// </summary>
        /// <param name="lines"></param>
        /// <param name="basePoint"></param>
        /// <param name="locatePoint"></param>
        /// <param name="locateAngle">Degrees</param>
        /// <returns></returns>
        private Boolean GetTimeTextPointAndAngle(List<LineShape> lines, Point2d basePoint, out Point2d locatePoint, out double locateAngle)
        {
            locatePoint = new Point2d();
            locateAngle = 0.0;
            double maxDistance = 0.0;

            if(lines.Count == 0)
                return false;

            for(int i = 0; i < lines.Count; i++)
            {
                LineShape line = lines[i];
                Point2d centerPoint = CenterPoint(line.StartPoint, line.EndPoint);
                double distance = app.Geometry.GetDistance(basePoint, centerPoint);
                if(distance > maxDistance)
                {
                    locatePoint = centerPoint;
                    locateAngle = Get2PointsAngle(line.StartPoint, line.EndPoint); // Degrees

                    maxDistance = distance;
                }
            }

            // 文字が外側になるように角度を調整
            double tempTextHight = maxDistance * 0.05;// 文字高さを基準点間距離の5%で計算
            Point2d textTopPoint1 = new Point2d(locatePoint.X, locatePoint.Y + tempTextHight);
            textTopPoint1 = app.Geometry.RotatePoint(textTopPoint1, locatePoint, locateAngle);

            Point2d textTopPoint2 = new Point2d(locatePoint.X, locatePoint.Y + tempTextHight);
            textTopPoint2 = app.Geometry.RotatePoint(textTopPoint2, locatePoint, locateAngle + 180.0);

            double distance1 = app.Geometry.GetDistance(basePoint, textTopPoint1);
            double distance2 = app.Geometry.GetDistance(basePoint, textTopPoint2);

            // 180度回転させた方が基準点からみて外側になる場合は角度を180度させる
            if(distance2 > distance1)
                locateAngle += 180.0;

            // 0度以上360度未満に補正
            locateAngle = app.Geometry.WrapAngleTo360(locateAngle);

            return true;
        }
        
        /// <summary>
        /// 単独の線（他と繋がらない）のリストとして最後の線リストか
        /// (指定したインデックスの線リストは、以降の線リストのリスト内の他の線リストと接続することはないか)
        /// </summary>
        /// <param name="targetIndex"></param>
        /// <param name="shadowOutlinesList"></param>
        /// <returns></returns>
        private bool IsLastSingleFrameLines(int targetIndex, List<List<LineShape>> shadowOutlinesList)
        {
            int shadowOutlinesListCount = shadowOutlinesList.Count;
            if(targetIndex >= shadowOutlinesListCount)
                return false;

            List<LineShape> targetLineShapes = shadowOutlinesList[targetIndex];

            for(int i = targetIndex + 1; i < shadowOutlinesListCount; i++)
            {                
                List<LineShape> otherLineShapes = shadowOutlinesList[i];
                for(int j = 0; j < otherLineShapes.Count; j++)
                {
                    LineShape otherLine = otherLineShapes[j];
                    Point2d otherLineStartPoint = otherLine.StartPoint;
                    Point2d otherLineEndPoint = otherLine.EndPoint;

                    for(int k = 0; k < targetLineShapes.Count; k++)
                    {
                        LineShape targetLine = targetLineShapes[k];
                        Point2d targetLineStartPoint = targetLine.StartPoint;
                        Point2d targetLineEndPoint = targetLine.EndPoint;

                         // 繋がって(端点が一致して)いる場合
                         if(IsPointEquals(otherLineStartPoint, targetLineStartPoint) || IsPointEquals(otherLineStartPoint, targetLineEndPoint)
                                || IsPointEquals(otherLineEndPoint, targetLineStartPoint) || IsPointEquals(otherLineEndPoint, targetLineEndPoint))
                         {
                             return false;
                         }
                    }
                }
            }

            return true;
        }


        /// <summary>
        /// 影倍率表、日影長さ表、方位角倍率図、設定内容コマンド実行中にマウスが移動したときに呼ばれる
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void CommandB_MouseMove(Object sender, CommandMouseMoveEventArgs e)
        {
            Command command = (Command)sender;
            // Command.GetTemporaryParameterItemByID を使用してコマンドの一時的なパラメータアイテムを取得します。
            // 一時的なパラメータアイテムからは、すでにパラメータの入力が確定している場合は確定している入力値が、
            // 入力が確定していない場合はマウスの現在位置などの一時的な値が取得できます。

            // 一時パラメータを取得する
            PointParameterItem pointParameterItem = (PointParameterItem)command.GetTemporaryParameterItemByID(locatePointParameterId);
            if(pointParameterItem == null)
                return;

            if(pointParameterItem.IsEmpty)
                return;

            // 配置点
            Point2d locatePoint = pointParameterItem.Point;

            //ドキュメントの取得
            Document doc = app.ActiveDocument;
            Drawing drawing = doc.CurrentDrawing;
            Point2d orgPoint = new Point2d(0.0, 0.0);

            // 図形の作成
            if(rubberBandShapes == null)
            {
                // 配置点はとりあえず原点で作成する
                Shape[] shapes = CreateShapesForCommandB(command, drawing, orgPoint);
                if(shapes == null)
                    return;

                rubberBandShapes = new List<Shape>();
                rubberBandShapes.AddRange(shapes);
            }


            List<Shape> rubberBandShapes2 = new List<Shape>();
            foreach(Shape shape in rubberBandShapes)
            {
                // マウスの位置に移動
                Shape shape2 = shape.Clone();
                shape2.Transform(orgPoint, 1.0, 1.0, 0.0, locatePoint);
                rubberBandShapes2.Add(shape2);
            }

            try
            {
                // ラバーバンド描画
                command.DrawShapeRubberBand(rubberBandShapes2.ToArray());
            }
            catch
            {
                throw;
            }
        }


        /// <summary>
        /// 影倍率表、日影長さ表、方位角倍率図コマンド実行時のパラメータの変化のイベント
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void CommandB_ParameterChanged(Object sender, CommandParameterChangedEventArgs e)
        {
            //ラバーバンド表示を削除する
            ClearRubberBand();

            Command command = (Command)sender;

            //ドキュメントの取得
            Document doc = app.ActiveDocument;
            Drawing drawing = doc.CurrentDrawing;

            // 方位角倍率図コマンドの場合
            if(command.UniqueName == ShadowDrawingAzimuthMagnificationDrawingCommandName)
            {
                int result = DueNorthSearch(doc, out double dueNorthDeg);
                if(result != 1)
                {
                    if(result == 0)
                        MessageBox.Show("真北が設定されていません。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    else if(result == -1)
                        MessageBox.Show("真北が図面中に複数あります。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);

                    return;
                }
            }


            Point2d locatePoint = new Point2d(0.0, 0.0);
            PointParameterItem pointParameterItem = (PointParameterItem)command.GetParameterItemByID(locatePointParameterId);
            if(pointParameterItem.IsEmpty)
                return;

            locatePoint = pointParameterItem.Point;

            // 図形の作成
            Shape[] shapes = CreateShapesForCommandB(command, drawing, locatePoint);
            if(shapes == null)
            {
                e.CommandState = CommandState.Continue;
                return;
            }

            // Undoの開始
            doc.UndoManager.BeginUndoUnit();

            // 図形追加
            ArrayList addArray = new ArrayList();
            addArray.AddRange(shapes);

            int shapeCount = addArray.Count;
            ArrayList addGroupShapeArray = new ArrayList(); // グループ化するための図形リスト

            for(int i = 0; i < shapeCount; i++)
            {
                Shape shape = (Shape)addArray[i];

                // レイヤのセット
                Layer currentLayer = doc.LayerTable.CurrentLayer;
                shape.Layer = currentLayer;

                // 追加
                Shape addShape = drawing.Shapes.Add(shape);

                // グループ
                addGroupShapeArray.Add(addShape);
            }

            // グループ化
            Shape shapeGroup = drawing.Shapes.AddDrawGroup((Shape[])addGroupShapeArray.ToArray(typeof(Shape)), false);

            // Undo の終了
            doc.UndoManager.EndUndoUnit();

            e.CommandState = CommandState.End;
        }

        /// <summary>
        /// 影倍率表、日影長さ表、方位角倍率図、設定内容コマンド実行時の図形リスト作成
        /// </summary>
        /// <param name="command"></param>
        /// <param name="drawing"></param>
        /// <param name="locatePoint">配置点</param>
        /// <returns>図形リスト（色や線種レイヤなどの基本設定はセットしていない）</returns>
        private Shape[] CreateShapesForCommandB(Command command, Drawing drawing, Point2d locatePoint)
        {
            Document doc = drawing.Document;

            double azimuthLength = 100.0;   // 方位角線の長さ初期値
            double dueNorthDeg = 0.0;       // 真北の角度(deg)

            // 観測地点の緯度
            double latitudeDeg = 0;             // (deg) 
            double solarDeclinationDeg = 0.0;   // 初期値:春秋分(deg)日赤緯(冬至:-23.45°、春秋分:0°、夏至:23.45°、任意時期)
            double thisHeight = 0.0;            // 高さ
            double locationHeight = 0.0;        // 測定面の高さ
            double calcHeight = 5.0;            // 計算高さ

            // 観測地点の緯度、日赤緯、測定面の高さ
            if(!GetShadowBaseData(out latitudeDeg, out solarDeclinationDeg, out locationHeight))
                return null;

            double latitudeRad = latitudeDeg * (Math.PI / 180d);
            double solarDeclinationRad = solarDeclinationDeg * (Math.PI / 180d);


            // 日影長さ表コマンドの場合
            if(command.UniqueName == ShadowDrawingShadowLengthTableCommandName)
            {
                // 日影長さ計算用の高さパラメータ取得
                // 未入力ならここまで
                LengthParameterItem lengthParameterItem = (LengthParameterItem)command.GetParameterItemByID(thisHeightParameterId);
                if(lengthParameterItem == null)
                    return null;

                if(lengthParameterItem.IsEmpty)
                    return null;

                thisHeight = lengthParameterItem.Length;
                calcHeight = thisHeight - locationHeight;// 計算高さ
                if(calcHeight <= 0.0) // 0以下ならエラー
                    return null;
            }
            // 方位角倍率図コマンドの場合
            else if(command.UniqueName == ShadowDrawingAzimuthMagnificationDrawingCommandName)
            {
                // 真北の角度を探す
                int result = DueNorthSearch(doc, out dueNorthDeg);
                if(result != 1)
                    return null;
 
                // 方位角線の長さパラメータ取得
                // 未入力ならここまで
                LengthParameterItem lengthParameterItem = (LengthParameterItem)command.GetParameterItemByID(azimuthLengthParameterId);
                if(lengthParameterItem == null)
                    return null;

                if(lengthParameterItem.IsEmpty)
                    return null;

                azimuthLength = lengthParameterItem.Length;

                if(azimuthLength <= 10) // 10より大きい値に
                    return null;
            }

            // 日影データ作成 //
            bool h9Toh15 = timeRangeComboBox.SelectedIndex == 1 ? true : false;// true:測定時間が9:00～15:00の場合
            double startSolarTime = 8;
            double endSolarTime = 16;
            // 測定範囲が9:00～15:00の場合
            if(h9Toh15)
            {
                startSolarTime = 9;
                endSolarTime = 15;
            }

            List<ShadowData> shadowDataList = GetShadowDataList(latitudeRad, solarDeclinationRad, 1800, startSolarTime, endSolarTime);

            ArrayList shapes = new ArrayList();// 戻り値の図形リスト

            // 影倍率表コマンドか日影長さ表コマンドの場合
            if(command.UniqueName == ShadowDrawingShadowMagnificationTableCommandName || command.UniqueName == ShadowDrawingShadowLengthTableCommandName)
            {
                int listSize = shadowDataList.Count;

                // 表の列数
                int columnsCount = 6;
                if(command.UniqueName == ShadowDrawingShadowLengthTableCommandName)
                    columnsCount = 3;

                double textHeight = GetTextFontHeight();
                double cellHeight = textHeight * 1.8;
                double solarTimeCellWidth = textHeight * 4.0;              // 時刻のセル幅
                double solarHeightCellWidth = textHeight * 6.0;            // 太陽高度のセル幅
                double solarDirectionCellWidth = textHeight * 6.0;         // 太陽方位角のセル幅
                double shadowMagnificationCellWidth = textHeight * 5.0;    // 影の倍率のセル幅
                double xMagnificationCellWidth = textHeight * 5.0;         // X 倍率のセル幅
                double yMagnificationCellWidth = textHeight * 5.0;         // Y 倍率のセル幅
                double shadowLengthCellWidth = textHeight * 8.5;           // 日影長さ(m)のセルの幅
                double[] cellWidth = null;
                string[] name = null;
                // 影倍率表コマンドの場合
                if(command.UniqueName == ShadowDrawingShadowMagnificationTableCommandName)
                {
                    cellWidth = new double[] { solarTimeCellWidth, solarHeightCellWidth, solarDirectionCellWidth, shadowMagnificationCellWidth, xMagnificationCellWidth, yMagnificationCellWidth };
                    name = new string[] { "時刻", "太陽高度", "太陽方位角", "影長倍率", "X 倍率", "Y 倍率" };
                }
                // 日影長さ表コマンドの場合
                else if(command.UniqueName == ShadowDrawingShadowLengthTableCommandName)
                {
                    cellWidth = new double[] { solarTimeCellWidth, shadowMagnificationCellWidth, shadowLengthCellWidth };
                    name = new string[] { "時刻", "影長倍率", "日影長さ(m)" };
                }

                double listWidth = 0.0;
                double listHeight = (double)(listSize + 1) * cellHeight;// 項目名分+1
                LineShape lineShape;
                Point2d startPoint;
                Point2d endPoint;
                string str = "";
                Point2d cellBottomRightPoint;

                // 表題
                string titleInfoText = GetTitleInfoText(command, latitudeDeg, solarDeclinationDeg, thisHeight, locationHeight, calcHeight);// 表題
                Point2d textLocatePoint = locatePoint;
                textLocatePoint.Y += textHeight * 0.25;
                TextShape text0 = app.ShapeFactory.CreateText(titleInfoText, textLocatePoint, 0.0);
                doc.TextSettings.CopyToShape(text0);
                text0.DirectionVertical = false;// 横書き
                text0.FontHeight = textHeight;// 文字高さセット
                text0.Alignment = Alignment.BottomLeft;// 左下
                shapes.Add(text0);


                // 表の縦線
                for(int i = 0; i < columnsCount; i++)
                {
                    listWidth += cellWidth[i];

                    // 最初だけは左側の線を追加
                    if(i == 0)
                    {
                        startPoint = locatePoint;
                        endPoint = startPoint;
                        endPoint.Y -= listHeight;
                        lineShape = app.ShapeFactory.CreateLine(startPoint, endPoint);
                        doc.BasicShapeSettings.CopyToShape(lineShape);
                        shapes.Add(lineShape);
                    }

                    startPoint = locatePoint;
                    endPoint = startPoint;
                    startPoint.X += listWidth;
                    endPoint.X = startPoint.X;
                    endPoint.Y -= listHeight;
                    // 線追加
                    lineShape = app.ShapeFactory.CreateLine(startPoint, endPoint);
                    doc.BasicShapeSettings.CopyToShape(lineShape);
                    shapes.Add(lineShape);
                }

                // 横の線と数値
                for(int i = 0; i < listSize; i++)
                {
                    ShadowData shadowData = shadowDataList[i];

                    // 最初に項目名の線追加
                    if(i == 0)
                    {
                        startPoint = locatePoint;
                        endPoint = startPoint;
                        endPoint.X += listWidth;
                        lineShape = app.ShapeFactory.CreateLine(startPoint, endPoint);
                        doc.BasicShapeSettings.CopyToShape(lineShape);
                        shapes.Add(lineShape);

                        startPoint.Y -= cellHeight;
                        endPoint.Y = startPoint.Y;
                        lineShape = app.ShapeFactory.CreateLine(startPoint, endPoint);
                        doc.BasicShapeSettings.CopyToShape(lineShape);
                        shapes.Add(lineShape);

                        cellBottomRightPoint = startPoint;

                        for(int j = 0; j < columnsCount; j++)
                        {
                            // 項目名
                            str = name[j];

                            cellBottomRightPoint.X += cellWidth[j];

                            // 中中基準で配置
                            Point2d middleMiddlePoint = cellBottomRightPoint;
                            middleMiddlePoint.X -= cellWidth[j] * 0.5;
                            middleMiddlePoint.Y += cellHeight * 0.5;
                            TextShape text = app.ShapeFactory.CreateText(str, middleMiddlePoint, 0.0); // 回転
                            doc.TextSettings.CopyToShape(text);
                            text.DirectionVertical = false;// 横書き
                            text.FontHeight = textHeight;// 文字高さセット
                            text.Alignment = Alignment.MiddleMiddle;// 配置点：中中

                            shapes.Add(text);
                        }

                    }

                    startPoint = locatePoint;
                    endPoint = startPoint;
                    startPoint.Y -= cellHeight * (double)(i + 2);
                    endPoint = startPoint;
                    endPoint.X += listWidth;
                    // 線追加
                    lineShape = app.ShapeFactory.CreateLine(startPoint, endPoint);
                    doc.BasicShapeSettings.CopyToShape(lineShape);
                    shapes.Add(lineShape);

                    cellBottomRightPoint = startPoint;

                    for(int j = 0; j < columnsCount; j++)
                    {
                        // 時刻
                        if(j == 0)
                        {
                            str = TimeToString(shadowData.solarTime);
                        }
                        else if(j == 1)
                        {
                            // 影倍率表コマンドの場合
                            if(command.UniqueName == ShadowDrawingShadowMagnificationTableCommandName)
                            {
                                // 太陽高度
                                str = DegreeToDmsString(shadowData.solarHeight * (180d / Math.PI), 1);
                            }
                            // 日影長さ表コマンドの場合
                            else if(command.UniqueName == ShadowDrawingShadowLengthTableCommandName)
                            {
                                // 影長倍率
                                double magnification = Math.Round(shadowData.shadowMagnification, 3, MidpointRounding.AwayFromZero);
                                str = magnification.ToString("F3");
                            }
                        }
                        else if(j == 2)
                        {
                            // 影倍率表コマンドの場合
                            if(command.UniqueName == ShadowDrawingShadowMagnificationTableCommandName)
                            {
                                // 太陽方位角
                                str = DegreeToDmsString(shadowData.solarDirection * (180d / Math.PI), 1);
                            }
                            // 日影長さ表コマンドの場合
                            else if(command.UniqueName == ShadowDrawingShadowLengthTableCommandName)
                            {
                                // 日影長さ(m)
                                double magnification = Math.Round(shadowData.shadowMagnification, 3, MidpointRounding.AwayFromZero);
                                magnification *= calcHeight;
                                str = magnification.ToString("F3");
                            }
                        }
                        else if(j == 3)
                        {
                            // 影長倍率
                            double magnification = Math.Round(shadowData.shadowMagnification, 3, MidpointRounding.AwayFromZero);
                            str = magnification.ToString("F3");
                        }
                        else if(j == 4)
                        {
                            // X 倍率
                            double x = Math.Round(shadowData.xMagnification, 3, MidpointRounding.AwayFromZero);
                            str = x.ToString("F3");
                        }
                        else if(j == 5)
                        {
                            // Y倍率
                            double y = Math.Round(shadowData.yMagnification, 3, MidpointRounding.AwayFromZero);
                            str = y.ToString("F3");
                        }

                        cellBottomRightPoint.X += cellWidth[j];

                        // 配置
                        if(str != "")
                        {
                            Point2d middleRightPoint = cellBottomRightPoint;
                            middleRightPoint.X -= textHeight * 0.5;
                            middleRightPoint.Y += cellHeight * 0.5;
                            TextShape text = app.ShapeFactory.CreateText(str, middleRightPoint, 0.0);
                            doc.TextSettings.CopyToShape(text);
                            text.DirectionVertical = false;// 横書き
                            text.FontHeight = textHeight;// 文字高さセット
                            text.Alignment = Alignment.MiddleRight;// 配置点：中右
                            shapes.Add(text);
                        }
                    }

                }

            }
            // 方位角倍率図コマンドコマンドの場合
            else if(command.UniqueName == ShadowDrawingAzimuthMagnificationDrawingCommandName)
            {
                // 表題"方位角倍率図"文字
                double textHeight = GetTextFontHeight();// 文字高さ
                Point2d locateTextPoint = locatePoint;
                string titleInfoText = GetTitleInfoText(command, latitudeDeg, solarDeclinationDeg);// 表題
                TextShape text2 = app.ShapeFactory.CreateText(titleInfoText, locateTextPoint, dueNorthDeg); // 回転
                doc.TextSettings.CopyToShape(text2);
                text2.DirectionVertical = false;// 横書き
                text2.FontHeight = textHeight;// 文字高さセット
                text2.Alignment = Alignment.TopMiddle;// 配置点：中上
                shapes.Add(text2);

                int listSize = shadowDataList.Count;
                for(int i = 0; i < listSize; i++)
                {
                    ShadowData shadowData = shadowDataList[i];

                    Point2d startPoint = locatePoint;
                    Point2d endPoint = locatePoint;
                    endPoint.Y += azimuthLength;

                    // 太陽方位角に合わせる
                    double rotateAngle = -(shadowData.solarDirection * (180d / Math.PI)) + dueNorthDeg; // Radians to Degrees
                    // 回転
                    endPoint = app.Geometry.RotatePoint(endPoint, startPoint, rotateAngle);

                    // 方位角の線追加
                    LineShape lineShape = app.ShapeFactory.CreateLine(startPoint, endPoint);
                    doc.BasicShapeSettings.CopyToShape(lineShape);
                    shapes.Add(lineShape);

                    // 太陽方位角
                    string strSolarDirection = DegreeToDmsString(shadowData.solarDirection * (180d / Math.PI), 1); // Radians to Degrees
                    Point2d textLocatePoint = locatePoint;
                    textLocatePoint.Y += azimuthLength * 0.6;
                    textLocatePoint = app.Geometry.RotatePoint(textLocatePoint, locatePoint, rotateAngle);
                    TextShape text3 = app.ShapeFactory.CreateText(strSolarDirection, textLocatePoint, rotateAngle + 90.0); // 回転
                    doc.TextSettings.CopyToShape(text3);
                    text3.DirectionVertical = false;// 横書き
                    text3.FontHeight = textHeight;// 文字高さセット
                    text3.Alignment = Alignment.BottomMiddle;// 中下
                    shapes.Add(text3);
                    // 「太陽方位角」文字
                    if(i == 0)
                    {
                        Point2d textLocatePoint31 = locatePoint;
                        textLocatePoint31.X -= text3.FontHeight * 1.1;
                        textLocatePoint31.Y += azimuthLength * 0.6;
                        textLocatePoint31 = app.Geometry.RotatePoint(textLocatePoint31, locatePoint, rotateAngle);
                        TextShape text31 = app.ShapeFactory.CreateText("太陽方位角", textLocatePoint31, rotateAngle + 90.0); // 回転
                        doc.TextSettings.CopyToShape(text31);
                        text31.FontHeight = textHeight;// 文字高さセット
                        text31.Alignment = Alignment.BottomMiddle;// 中下
                        shapes.Add(text31);
                    }

                    // 影長倍率
                    double magnification = Math.Round(shadowData.shadowMagnification, 3, MidpointRounding.AwayFromZero);
                    string strShadowMagnification = magnification.ToString("F3");
                    Point2d textLocatePoint2 = locatePoint;
                    textLocatePoint2.Y += azimuthLength * 0.85;
                    textLocatePoint2 = app.Geometry.RotatePoint(textLocatePoint2, locatePoint, rotateAngle);
                    TextShape text4 = app.ShapeFactory.CreateText(strShadowMagnification, textLocatePoint2, rotateAngle + 90.0); // 回転
                    doc.TextSettings.CopyToShape(text4);
                    text4.DirectionVertical = false;// 横書き
                    text4.FontHeight = textHeight;// 文字高さセット
                    text4.Alignment = Alignment.BottomMiddle;// 中下
                    shapes.Add(text4);
                    // 「影長倍率」文字
                    if(i == 0)
                    {
                        Point2d textLocatePoint41 = locatePoint;
                        textLocatePoint41.X -= text4.FontHeight * 1.1;
                        textLocatePoint41.Y += azimuthLength * 0.85;
                        textLocatePoint41 = app.Geometry.RotatePoint(textLocatePoint41, locatePoint, rotateAngle);
                        TextShape text41 = app.ShapeFactory.CreateText("影長倍率", textLocatePoint41, rotateAngle + 90.0); // 回転
                        doc.TextSettings.CopyToShape(text41);
                        text41.DirectionVertical = false;// 横書き
                        text41.FontHeight = textHeight;// 文字高さセット
                        text41.Alignment = Alignment.BottomMiddle;// 中下
                        shapes.Add(text41);
                    }

                    // 時間
                    string solarTime = TimeToString(shadowData.solarTime);
                    textLocatePoint = locatePoint;
                    textLocatePoint.Y += azimuthLength + doc.TextSettings.FontHeight * 0.5; // 文字高さの半分分離す
                    textLocatePoint = app.Geometry.RotatePoint(textLocatePoint, locatePoint, rotateAngle);
                    TextShape text5 = app.ShapeFactory.CreateText(solarTime, textLocatePoint, rotateAngle + 90.0); // 回転
                    doc.TextSettings.CopyToShape(text5);
                    text5.DirectionVertical = false;// 横書き
                    text5.FontHeight = textHeight;// 文字高さセット
                    text5.Alignment = Alignment.MiddleLeft;// 左中
                    shapes.Add(text5);
                }

            }
            // 設定内容コマンドの場合
            else if(command.UniqueName == ShadowDrawingSettingsOutptCommandName)
            {
                // 文字
                double textHeight = GetTextFontHeight(); // 文字高さ
                Point2d locateTextPoint = locatePoint;
                string titleInfoText = GetTitleInfoText(command, latitudeDeg, solarDeclinationDeg, 0.0, locationHeight);// 表題
                TextShape text2 = app.ShapeFactory.CreateText(titleInfoText, locateTextPoint, 0.0);
                doc.TextSettings.CopyToShape(text2);
                text2.DirectionVertical = false;// 横書き
                text2.FontHeight = textHeight;// 文字高さセット
                text2.Alignment = Alignment.TopLeft;// 配置点：左上
                shapes.Add(text2);
            }

            return (Shape[])shapes.ToArray(typeof(Shape));
        }

        /// <summary>
        /// 現在のドキュメントの文字高さを取得する
        /// </summary>
        /// <returns></returns>
        double GetTextFontHeight()
        {
            Document doc = app.ActiveDocument;
            double textFontHeight = doc.TextSettings.FontHeight;

            // 用紙を表示している場合は部分図レイアウトの逆スケール倍した値
            if(GetSizeScale(out double scaleX, out double scaleY))
                textFontHeight *= scaleX;

            return textFontHeight;
        }

        /// <summary>
        /// 大きさに適用するスケールを取得
        /// </summary>
        /// <param name="scaleX"></param>
        /// <param name="scaleY"></param>
        /// <returns></returns>
        Boolean GetSizeScale(out double scaleX, out double scaleY)
        {
            // 結果の初期化
            scaleX = 1.0;
            scaleY = 1.0;

            //ドキュメントの取得
            Document doc = app.ActiveDocument;
            // 現在のレイアウト
            Layout layout = doc.CurrentLayout;
            if(layout == null)
                return false;

            if(layout.IsCurrent)
            {
                double layoutScaleX = layout.ScaleX;
                double layoutScaleY = layout.ScaleY;

                if(layoutScaleX <= RcPrecisionScale || layoutScaleY <= RcPrecisionScale)
                    return false;

                scaleX = 1.0 / layoutScaleX;
                scaleY = 1.0 / layoutScaleY;

                return true;
            }

            return true;
        }


        /// <summary>
        /// 日影図作成が可能な図面かチェック
        /// </summary>
        /// <param name="doc"></param>
        /// <param name="showErrMessage">エラーの場合エラーメッセージを表示するか</param>
        /// <returns></returns>
        Boolean CheckDocument(Document doc, bool showErrMessage)
        {
            // 同じ部分図が複数の用紙に部分図レイアウトとして配置されているような図面は実行させない。
            // 意図せず他の用紙の見た目に変更が加わることがあるため。
            
            List<List<int>> partsDrawingIDListList = new List<List<int>>();
            List<int> partsDrawingIDList;

            // 用紙にあるレイアウトを取得
            DrawingCollection drawings = doc.Drawings;
            for(int i = 0; i < drawings.Count; i++)
            {
                Drawing drawing = drawings[i];

                // 用紙
                if(drawing is Paper)
                {
                    //部分図レイアウト
                    LayoutCollection layouts = drawing.Layouts;
                    partsDrawingIDList = new List<int>();
                    for(int j = 0; j < layouts.Count; j++)
                    {
                        Layout layout = layouts[j];
                        int drawingID = layout.Drawing.ID;
                        if(!partsDrawingIDList.Contains(drawingID))
                            partsDrawingIDList.Add(drawingID);
                    }

                    if(partsDrawingIDList.Count > 0)
                        partsDrawingIDListList.Add(partsDrawingIDList);

                }
            }

            // 用紙が一つだけならチェックなし
            int paperDrawingCount = partsDrawingIDListList.Count;
            if(paperDrawingCount <= 1)
                return true;

            for(int i = 0; i < paperDrawingCount; i++)
            {
                partsDrawingIDList = partsDrawingIDListList[i];

                foreach(int drawingID in partsDrawingIDList)
                {
                    for(int j = i + 1; j < paperDrawingCount; j++)
                    {
                        List<int> partsDrawingIDList2 = partsDrawingIDListList[j];

                        foreach(int drawingID2 in partsDrawingIDList2)
                        {
                            // 他の用紙に同じ部分図の部分図レイアウトがあった場合
                            if(drawingID == drawingID2)
                            {
                                Drawing drawing = drawings.GetDrawingByID(drawingID);
                                // カレント以外で表示されていなければ対象外
                                if(!drawing.IsCurrent && !drawing.Visible)
                                    continue;

                                if(showErrMessage)
                                {
                                    string drawingName = drawing.Name;
                                    string strErr = "表示している部分図("+ drawingName + ")のレイアウトが複数の用紙に配置されているため実行できません。";
                                    MessageBox.Show(strErr, AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                                }

                                return false;
                            }
                        }
                    }
                }
            }

            return true;
        }

        /// <summary>
        /// 日影図作成が可能な状態かチェック
        /// </summary>
        /// <param name="doc"></param>
        /// <param name="textShapeList"></param>
        /// <returns></returns>
        Boolean CheckLayout(Document doc, List<TextShape> textShapeList)
        {
            // チェックするのは高さ文字のある部分図だけ
            // 同じ部分図の部分図レイアウトは一つだけ
            // 部分図レイアウトの配置角度はすべて0度限定
            // 高さの文字のある部分図で、スケールはXY同一スケールで、全部分図レイアウトで同じ、配置点も同じ。

            int err = 0;
            Drawing paperDrawing = null;
            Layout currentlayout = null;

            // ビューが部分図の場合は配置角度、スケールは関係ないのでチェックしない
            TabView tabView = app.ActiveTabView;
            if(tabView is DrawingView drawingView)
            {
                Drawing drawing = drawingView.Drawing;
                // 部分図
                if(drawing is Part)
                {
                    return true;
                }
                // 用紙
                else if(drawing is Paper)
                {
                    paperDrawing = drawing;
                }
                // その他
                else
                {
                    err = 99;
                    goto ERRLABEL;
                }
            }

            // 用紙自身がカレントのレイアウトの場合は実行させない（影の線をm->mm換算して追加するためサイズが適当ではない）
            currentlayout = doc.CurrentLayout;
            if(currentlayout.Drawing is Paper)
            {
                err = 100;
                goto ERRLABEL;
            }

            // 念のためカレント部分図レイアウトが表示している用紙上のものかチェック
            Debug.Assert(paperDrawing != null);
            LayoutCollection papersLayouts = paperDrawing.Layouts;
            bool looked = false;
            for(int i = 0; i < papersLayouts.Count; i++)
            {
                Layout layout = papersLayouts[i];
                if(currentlayout.ID == layout.ID)
                {
                    looked = true;
                    break;
                }
            }
            if(!looked)
            {
                err = 101;
                goto ERRLABEL;
            }


            List<int> chekDrawingIDList = new List<int>();
            for(int i = 0; i < textShapeList.Count; i++)
            {
                TextShape textShape = textShapeList[i];
                int drawingID = textShape.Drawing.ID;
                if(!chekDrawingIDList.Contains(drawingID))
                    chekDrawingIDList.Add(drawingID);
            }

            Point2d layoutLocatePoint = new Point2d();
            double layoutScaleX = -1.0;
            double layoutScaleY = -1.0;
            double layoutAngle = 0.0;
            int roopCounter = 0;

            for(int i = 0; i < papersLayouts.Count; i++)
            {
                Layout layout = papersLayouts[i];

                // 用紙自身は関係なし
                if(layout.Drawing is Paper)
                {
                    // 用紙上に高さ文字がある場合はエラーにする
                    if(chekDrawingIDList.Contains(layout.Drawing.ID))
                    {
                        err = 10;
                        goto ERRLABEL;
                    }

                    continue;
                }

                // 高さ文字のない部分図も関係なし
                if(!chekDrawingIDList.Contains(layout.Drawing.ID))
                    continue;
                
                roopCounter++;
                if(roopCounter == 1)
                {
                    layoutLocatePoint = layout.Origin;
                    layoutScaleX = layout.ScaleX;
                    layoutScaleY = layout.ScaleY;
                    layoutAngle = layout.Angle;
                }
                else
                {
                    if(!IsPointEquals(layoutLocatePoint, layout.Origin))
                        err = 1;
                    else if(Math.Abs(layoutScaleX - layout.ScaleX) > RcPrecisionScale)
                        err = 2;
                    else if(Math.Abs(layoutScaleY - layout.ScaleY) > RcPrecisionScale)
                        err = 3;
                    else if(Math.Abs(layoutAngle - layout.Angle) > RcPrecisionAngular * (180d / Math.PI))
                        err = 4;
                }
            }

            ERRLABEL:
            if(err != 0)
            {
                string strErr;
                 if(err == 99)
                    strErr = "このビューでは実行できません。";
                else if(err == 100)
                    strErr = "カレントのレイアウトが用紙になっています。";
                else if(err == 101)
                    strErr = "実行しようとしてる用紙上以外の部分図レイアウトがカレントになっています・";
                else if(err == 10)
                    strErr = "用紙に高さ設定の文字があります。";
                else
                    strErr = "部分図レイアウトの配置点やスケール、配置角度が異なるものがあります。";

                MessageBox.Show(strErr, AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                return false;
            }

            return true;
        }


        /// <summary>
        /// 影倍率表、日影長さ表、方位角倍率図の表題、または設定内容の文字列取得
        /// </summary>
        /// <param name="command"></param>
        /// <param name="latitudeDeg">(deg) 緯度</param>
        /// <param name="solarDeclinationDeg">(deg) 日赤緯</param>
        /// <param name="thisHeight">高さ(日影長さ表でのみ使用)</param>
        /// <param name="locationHeight">測定面の高さ(日影長さ表でのみ使用)</param>
        /// <param name="calcHeight">計算高さ(日影長さ表でのみ使用)</param>
        /// <returns>表題の文字列</returns>
        private string GetTitleInfoText(Command command, double latitudeDeg, double solarDeclinationDeg, double thisHeight = 0.0, double locationHeight = 0.0, double calcHeight = 0.0)
        {
            string titleInfoText = "";

            string latitudeStr = DegreeToString(latitudeDeg, 2);
            string seasonStr = seasonComboBox.Text;
            if(seasonStr == "任意時期")
            {
                seasonStr = "日赤緯";
                seasonStr += " = " + DegreeToString(solarDeclinationDeg, 2);
            }

            string infoStr = "[緯度 = " + latitudeStr + "][ " + seasonStr + " ]";

            // 影倍率表コマンドの場合
            if(command.UniqueName == ShadowDrawingShadowMagnificationTableCommandName)
            {
                titleInfoText = "影倍率表" + "  " + infoStr;
            }
            // 日影長さ表コマンドの場合
            else if(command.UniqueName == ShadowDrawingShadowLengthTableCommandName)
            {
                titleInfoText = "日影長さ表" + "  " + infoStr;
                // 高さ
                thisHeight = Math.Round(thisHeight, 3, MidpointRounding.AwayFromZero);
                string str = thisHeight.ToString("F3");
                titleInfoText += "\r\n" + "高さ = " + str + "(m)";
                // 測定面の高さ
                locationHeight = Math.Round(locationHeight, 3, MidpointRounding.AwayFromZero);
                str = locationHeight.ToString("F3");
                titleInfoText += "測定面の高さ = " + str + "(m)";
                // 計算高さ
                calcHeight = Math.Round(calcHeight, 3, MidpointRounding.AwayFromZero);
                str = calcHeight.ToString("F3");
                titleInfoText += "\r\n" + "計算高さ = " + str + "(m)";

            }
            // 方位角倍率図コマンドの場合
            else if(command.UniqueName == ShadowDrawingAzimuthMagnificationDrawingCommandName)
            {
                titleInfoText = "方位角倍率図" + "\r\n" + infoStr;
            }
            // 設定内容コマンドの場合
            else if(command.UniqueName == ShadowDrawingSettingsOutptCommandName)
            {
                // 測定面の高さ
                locationHeight = Math.Round(locationHeight, 3, MidpointRounding.AwayFromZero);
                string str = locationHeight.ToString();
                string str2 = timeRangeComboBox.Text;
                titleInfoText = "測定面の高さ = " + str + "(m)" + " 緯度 = " + latitudeStr + "[ " + seasonStr + " ] " + "測定時間:" + str2;

            }

            return titleInfoText;
        }


        /// <summary>
        /// 時間文字作成取得
        /// </summary>
        /// <param name="solarTime"></param>
        /// <returns></returns>
        string TimeToString(double solarTime)
        {
            string strSolarTime;

            int h = (int)Math.Floor(solarTime);
            int m = (int)((solarTime - (double)h) * 60.0);

            strSolarTime = h.ToString() + ":" + string.Format("{0:00}", m);

            return strSolarTime;
        }

        /// <summary>
        /// 角度値から°表記の文字列を取得
        /// </summary>
        /// <param name="degree">角度値(deg)</param>
        /// <param name="precision">精度</param>
        /// <param name="withDo">°(度)記号付加フラグ</param>
        /// <returns></returns>
        private string DegreeToString(double degree, int precision, bool withDo = true)
        {
            string doChar = "";
            if(withDo)
                doChar = "°";

            string dmsStr = app.NumericTextConverter.DegreeToDmsText(degree, AngleUnitFormatType.Degrees, precision, doChar, false, true, true, true, true);
            return dmsStr;
        }

        /// <summary>
        /// 角度値から度分文字列を取得
        /// </summary>
        /// <param name="degree">角度値(deg)</param>
        private string DegreeToDmsString(double degree, int precision)
        {
            string dmsStr = app.NumericTextConverter.DegreeToDmsText(degree, AngleUnitFormatType.DegMinSec, precision, "°", false, true, true, true, true);
            return dmsStr;
        }

        /// <summary>
        /// 日影計算の基本データ取得
        /// </summary>
        /// <param name="latitude">観測地点の緯度(deg)</param>
        /// <param name="solarDeclination">日赤緯(deg)</param>
        /// <param name="locationHeight">測定面の高さ</param>
        /// <returns>true 成功</returns>
        private Boolean GetShadowBaseData(out double latitude, out double solarDeclination, out double locationHeight)
        {
            // 観測地点の緯度
            latitude = 35.7; // (deg) 初期値

            // 日赤緯(冬至:-23.45°、春秋分:0°、夏至:23.45°、任意時期)
            solarDeclination = 0.0;// 初期値:春秋分 (deg)

            // 測定面の高さ
            locationHeight = 0.0; // 初期値

            // 観測地点の緯度と測定面の高さ
            if(!GetShadowBaseDataSub(out latitude, out solarDeclination))
                return false;

            string locationHeightTextBoxText = locationHeightTextBox.Text;
            //文字列をdouble型に変換
            if(!double.TryParse(locationHeightTextBoxText, out locationHeight))
            {
                MessageBox.Show("測定面の高さの数値が正しくありません。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                return false;
            }

            return true;
        }
    
        /// <summary>
        /// 日影計算の基本データ取得
        /// </summary>
        /// <param name="latitude">観測地点の緯度(deg)</param>
        /// <param name="solarDeclination">日赤緯(deg)</param>
        /// <returns>true 成功</returns>
        private Boolean GetShadowBaseDataSub(out double latitude, out double solarDeclination)
        {
            // 観測地点の緯度
            latitude = 35.7; // (deg) 初期値

            // 日赤緯(冬至:-23.45°、春秋分:0°、夏至:23.45°、任意時期)
            solarDeclination = 0.0;// 初期値:春秋分 (deg)

            string latitudeTextBoxText = latitudeTextBox.Text;
            //文字列をdouble型に変換
            if(!double.TryParse(latitudeTextBoxText, out latitude) || latitude < 0.0 || latitude > 60.0)
            {
                MessageBox.Show("緯度の数値が正しくありません。0～60°で指定してください。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                return false;
            }

            // 日赤緯(冬至:-23.45°、春秋分:0°、夏至:23.45°、任意時期)
            // 冬至
            if(seasonComboBox.SelectedIndex == 0)
                solarDeclination = -23.45;
            // 夏至
            else if(seasonComboBox.SelectedIndex == 2)
                solarDeclination = 23.45;
            // 任意時期
            else if(seasonComboBox.SelectedIndex == 3)
            {
                string anySeasonTextBoxText = anySeasonTextBox.Text;
                //文字列をdouble型に変換
                if(!double.TryParse(anySeasonTextBoxText, out solarDeclination))
                {
                    MessageBox.Show("任意時期の数値が正しくありません。", AddInProgramName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return false;
                }
            }

            return true;
        }

         /// <summary>
        /// 日影データリスト取得
        /// </summary>
        /// <param name="latitudeRad"></param>
        /// <param name="solarDeclinationRad"></param>
        /// <param name="intervalType">データ取得時間間隔タイプ(4:4秒、10:10秒、60:1分、1800:30分、3600:1時間)</param>
        /// <param name="startSolarTime">測定開始時間が8:00なら8.0
        /// <param name="endSolarTime">測定終了時間が16:00なら16.0
        /// <returns></returns>
        private List<ShadowData> GetShadowDataList(double latitudeRad, double solarDeclinationRad, int intervalType = 3600, double startSolarTime = 8, double endSolarTime = 16)
        {
            // ここの計算では間隔は秒単位で計算できるようにしてあるが、とりあえず以下のタイプだけということにしておく
            Debug.Assert(intervalType == 4 || intervalType == 10 || intervalType == 60 || intervalType == 1800 || intervalType == 3600);
            int intervalTime = intervalType;

            Debug.Assert(startSolarTime < endSolarTime);

            var shadowDataList = new List<ShadowData>();

            // 日影データ作成
            double solarTime = startSolarTime;
            int counter = 0;
            do
            {
                solarTime = startSolarTime + ((double)(intervalTime * counter) / 3600.0);
              
                ShadowData shadowData = new ShadowData();
                GetShadowData(latitudeRad, solarDeclinationRad, solarTime, shadowData);

                shadowDataList.Add(shadowData);

                counter++;

            } while(solarTime + (double)intervalTime / 3600.0  * 0.5 < (double)endSolarTime);// 0.5計算の誤差考慮

            return shadowDataList;
        }

        /// <summary>
        /// 指定した時間の日影データを取得する
        /// </summary>
        /// <param name="latitudeRad">観測地点の緯度(rad)</param>
        /// <param name="solarDeclinationRad">日赤緯(rad)</param>
        /// <param name="solarTime">真太陽時(ex. 12:30 ---> 12.5)</param>
        /// <param name="shadowData">out 日影基本データ</param>
        protected void GetShadowData(double latitudeRad, double solarDeclinationRad, double solarTime, ShadowData shadowData)
        {
            shadowData.solarTime = solarTime;
            RCShadowDataCalculator.GetShadowData(latitudeRad, solarDeclinationRad, solarTime, out shadowData.solarHeight, out shadowData.solarDirection, out shadowData.shadowMagnification, out shadowData.xMagnification, out shadowData.yMagnification);
        }

        /// <summary>
        /// タブコントロール（タブ部分）のオーナー描画
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void tabControl1_DrawItem(object sender, DrawItemEventArgs e)
        {
            // TabControlを取得
            TabControl tab = (TabControl)sender;

            // タブページのテキストを取得
            string txt = tab.TabPages[e.Index].Text;

            // タブのテキストと背景を描画するためのブラシとペン
            SolidBrush selectTabBackBrush = null;
            System.Drawing.Brush textBrush = null;// 文字のブラシ

            // 文字の色(選択されているタブ)
            if((e.State & DrawItemState.Selected) == DrawItemState.Selected)
               textBrush = new SolidBrush(tab.TabPages[e.Index].ForeColor);// タブページの文字と同じ色にする
            else
               textBrush = System.Drawing.Brushes.Black;

            // 選択されているタブ
            if((e.State & DrawItemState.Selected) == DrawItemState.Selected)
            {
                // 背景はタブページと同じ色にする
                System.Drawing.Color selectTabBackColor = tab.TabPages[e.Index].BackColor;
                selectTabBackBrush = new SolidBrush(selectTabBackColor);

                // 背景の描画
                Rectangle tabRect = e.Bounds;
                Rectangle tabRect2 = tab.GetTabRect(e.Index);

                int offsetX = 2;
                tabRect.X = tabRect2.X + offsetX;
                tabRect.Width = tabRect2.Width - (int)(offsetX * 1.5);
                tabRect.Y = tabRect2.Y + 1;

                e.Graphics.FillRectangle(selectTabBackBrush, tabRect);
            }

            // 文字の描画
            // StringFormatを作成
            StringFormat sf = new StringFormat();
            // 中中に表示する
            sf.Alignment = StringAlignment.Center;
            sf.LineAlignment = StringAlignment.Center;
 
            // 文字描画
            Rectangle textDrawRect = e.Bounds;// 範囲
            e.Graphics.DrawString(txt, e.Font, textBrush, textDrawRect, sf);
            
            // 最後のタブの場合、タブ領域の隣からタブコントロール全体の右端まで色を塗る
            if(e.Index == tab.TabCount - 1)
            {
                // 最後のタブの場合、タブ領域の隣からタブコントロール全体の右端まで色を塗る
                System.Drawing.Color notTabAreaBackColor = this.BackColor;// フォームの背景と同じ色にする
				System.Drawing.Brush notTabAreaBrush = new SolidBrush(notTabAreaBackColor);
				Rectangle lastTabRect = tab.GetTabRect(e.Index);
                e.Graphics.FillRectangle(notTabAreaBrush, lastTabRect.Right, 0, tab.Width - lastTabRect.Right, lastTabRect.Height + lastTabRect.Top);
			}

            e.Graphics.Dispose();
        }

        /// <summary>
        /// オプションフォームの表示
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OptionButton_Click(object sender, EventArgs e)
        {
            using(var optionForm = new OptionForm())
            {
                //optionForm.ApplyStyleFrom(this);
                optionForm.SetOption(addInSettings);
                if(optionForm.ShowDialog(this) == DialogResult.OK)
                {
                    optionForm.GetOption(addInSettings);
                }
            }
        }
       
    }
}
